<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace IRI\Bundle\WikiTagBundle\Command;

use Symfony\Component\DependencyInjection\ContainerAwareInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Output\Output;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\UpdateCommand;
use Doctrine\ORM\Tools\SchemaTool;
use Symfony\Bundle\DoctrineBundle\Command\Proxy\DoctrineCommandHelper;


/**
 * Command to generate the SQL needed to update the database schema to match
 * the current mapping information.
 *
 * This file is a direct adaptation of the Symfony\Bundle\DoctrineBundle\Command\Proxy\UpdateSchemaDoctrineCommand
 *
 */
class UpdateSchemaDoctrineCommand extends UpdateCommand implements ContainerAwareInterface
{
    protected function configure()
    {
        parent::configure();

        $this
            ->setName('wikitag:schema:update')
            ->setDescription('Executes (or dumps) the SQL needed to update the database schema to match the current mapping metadata')
            ->addOption('em', null, InputOption::VALUE_OPTIONAL, 'The entity manager to use for this command')
            ->setHelp(<<<EOT
The <info>doctrine:schema:update</info> command generates the SQL needed to
synchronize the database schema with the current mapping metadata of the
default entity manager.

For example, if you add metadata for a new column to an entity, this command
would generate and output the SQL needed to add the new column to the database:

<info>php app/console doctrine:schema:update --dump-sql</info>

Alternatively, you can execute the generated queries:

<info>php app/console doctrine:schema:update --force</info>

You can also update the database schema for a specific entity manager:

<info>php app/console doctrine:schema:update --em=default</info>
EOT
        );
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        DoctrineCommandHelper::setApplicationEntityManager($this->getApplication(), $input->getOption('em'));

        parent::execute($input, $output);
    }
    
   /**
    * @var ContainerInterface
    */
    private $container;
    
    protected function getContainer()
    {
        if (null === $this->container) {
            $this->container = $this->getApplication()->getKernel()->getContainer();
        }
    
        return $this->container;
    }
    
    /**
     * @see ContainerAwareInterface::setContainer()
     */
    public function setContainer(ContainerInterface $container = null)
    {
        $this->container = $container;
    }
    
    
    protected function filterUpdateSchema($sqls)
    {
    
        // get service
        $schema_utils = $this->getContainer()->get("wikitag.shema_utils");
    
        $res_sqls = $schema_utils->filter_foreign_key($sqls);
        $res_sqls = $schema_utils->filter_index_creation($res_sqls);
    
    
        return $res_sqls;
    }
    
    protected function executeSchemaCommand(InputInterface $input, OutputInterface $output, SchemaTool $schemaTool, array $metadatas)
    {
        // Defining if update is complete or not (--complete not defined means $saveMode = true)
        $saveMode = ($input->getOption('complete') !== true);
    
        $sqls = $schemaTool->getUpdateSchemaSql($metadatas, $saveMode);
        $sqls = $this->filterUpdateSchema($sqls);
        if (0 == count($sqls)) {
            $output->writeln('Nothing to update - your database is already in sync with the current entity metadata.');
    
            return;
        }
    
        $dumpSql = (true === $input->getOption('dump-sql'));
        $force = (true === $input->getOption('force'));
        if ($dumpSql && $force) {
            throw new \InvalidArgumentException('You can pass either the --dump-sql or the --force option (but not both simultaneously).');
        }
    
        if ($dumpSql) {
            $output->writeln(implode(';' . PHP_EOL, $sqls));
        } else if ($force) {
            $output->writeln('Updating database schema...');
            
            $emHelper = $this->getHelper('em');
            
            $conn = $emHelper->getEntityManager()->getConnection();
            
            foreach ($sqls as $sql) {
                $conn->executeQuery($sql);
            }
            $output->writeln(sprintf('Database schema updated successfully! "<info>%s</info>" queries were executed', count($sqls)));
        } else {
            $output->writeln('<comment>ATTENTION</comment>: This operation should not be executed in a production environment.');
            $output->writeln('           Use the incremental update to detect changes during development and use');
            $output->writeln('           the SQL DDL provided to manually update your database in production.');
            $output->writeln('');
    
            $output->writeln(sprintf('The Schema-Tool would execute <info>"%s"</info> queries to update the database.', count($sqls)));
            $output->writeln('Please run the operation by passing one of the following options:');
    
            $output->writeln(sprintf('    <info>%s --force</info> to execute the command', $this->getName()));
            $output->writeln(sprintf('    <info>%s --dump-sql</info> to dump the SQL statements to the screen', $this->getName()));
        }
    }
    
    
}
