<?php
/*
 * This file is part of the WikiTagBundle package.
 *
 * (c) IRI <http://www.iri.centrepompidou.fr/>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace IRI\Bundle\WikiTagBundle\Services;
 
use IRI\Bundle\WikiTagBundle\Entity\DocumentTag;
use Symfony\Component\DependencyInjection\ContainerAware;
use Symfony\Component\DependencyInjection\ContainerInterface;
use IRI\Bundle\WikiTagBundle\Utils\WikiTagUtils;

class DocumentService extends ContainerAware
{
    /**
     * Get the container associated with this service.
     * @return ContainerInterface
     */
    public function getContainer()
    {
        return $this->container;
    }
    
    /**
     * Public constructor with container as parameter for contruct injection.
     * @param ContainerInterface $container
     */
    public function __construct(ContainerInterface $container)
    {
        $this->setContainer($container);
    }
    
    private $doctrine;
    
    public function getDoctrine()
    {
        if(is_null($this->doctrine))
        {
            $this->doctrine = $this->getContainer()->get('doctrine');
        }
        return $this->doctrine;
    }
    

    /**
     * Copy the list of tags of one document to another.
     * The ids are the ids of the "host" document.
	 *
     * @param mixed $id_doc_src the source document id
     * @param mixed $id_doc_tgt the target document id
     */
    public function copyTags($id_doc_src, $id_doc_tgt)
    {
        
        $doctrine = $this->getDoctrine();
        $em = $doctrine->getEntityManager();
        
        $doc_rep = $em->getRepository("WikiTagBundle:Document");
        
        $src_doc = $doc_rep->findOneByExternalId($id_doc_src);
        if(is_null($src_doc))
        {
            throw new Exception("cloneTags: no source doc");
        }
        
        $tgt_doc = $doc_rep->findOneByExternalId($id_doc_tgt);
        if(is_null($tgt_doc))
        {
            throw new Exception("cloneTags: no target doc");
        }
        
        
        $doc_rep->copyTags($src_doc, $tgt_doc);
        
        $em->flush();
        
    }
    
    /**
     *
     * Add a new tag to a "host" document.
     * If the label already exists, an exception is raised.
     *
     * @param mixed $id_doc
     * @param string|array $tag_label : the label of the new tag
     */
    public function addTags($id_doc, $tag_labels)
    {
        // We get the DocumentTags
        $em = $this->getDoctrine()->getEntityManager();
        
        $need_flush = false;
        
        
        if(!is_array($tag_labels)) {
            $tag_labels = array($tag_labels);
        }
        
        foreach ($tag_labels as $tag_label) {
        
            $normalized_tag_label = WikiTagUtils::normalizeTag($tag_label);
            
            $query = $em->createQuery("SELECT COUNT(dt.id) FROM WikiTagBundle:DocumentTag dt JOIN dt.tag t WHERE dt.document = :id_doc AND t.normalizedLabel = :label");
            $query->setParameters(array("id_doc"=>$id_doc, "label"=>$normalized_tag_label));
            
            $nb_tags = $query->getSingleScalarResult();
            
            // If the label was found, we sent a bad request
            if($nb_tags > 0){
                throw new WikiTagServiceException(sprintf("Le tag %s existe déjà pour cette fiche.", $tag_label), 400, null, "duplicate_tag");
            }
            // returns array($tag, $revision_id, $created)
            try {
                $ar = $this->getDoctrine()->getRepository('WikiTagBundle:Tag')->getOrCreateTag($tag_label);
            }
            catch (\Exception $e){
                throw new WikiTagServiceException($e->getMessage(), 500 , $e, "wikipedia_request_failed");
            }
            
            $tag = $ar[0];
            $revision_id = $ar[1];
            $created = $ar[2];
            
            $tags = $this->getDoctrine()->getRepository('WikiTagBundle:DocumentTag')->findByDocumentExternalId($id_doc, array('tag'=>$tag->getId()));
            $nb_tags = count($tags);
            
            if($created==true || $nb_tags==0){
                $new_order_ar = $this->getDoctrine()->getRepository('WikiTagBundle:DocumentTag')->getMaxOrder($id_doc);
                // The result is a double array. And reset(reset($newOrderAr)) is not allowed. And a string is returned.
                $a1 = reset($new_order_ar);
                $new_order = intval(reset($a1)) + 1;
                $new_DT = new DocumentTag();
                $new_DT->setDocument($this->getDoctrine()->getRepository('WikiTagBundle:Document')->findOneByExternalId($id_doc));
                $new_DT->setTag($tag);
                $new_DT->setOriginalOrder($new_order);
                $new_DT->setTagOrder($new_order);
                $new_DT->setWikipediaRevisionId($revision_id);
                $em->persist($new_DT);
                $need_flush = true;
                
            }
        }
        
        if($need_flush) {
            $em->flush();
        }
        
        
    }
    
    public function getTagLabels($id_doc)
    {
        $rep = $this->getDoctrine()->getRepository('WikiTagBundle:Document');
        $doc = $rep->findOneByExternalId($id_doc);
        
        if(is_null($doc)) {
            throw new WikiTagServiceException("Unknown document id");
        }
        
        return $rep->getTagsStr($doc);

    }
    
        
}