from sqlalchemy import Column, Integer, String, ForeignKey, create_engine
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.orm import relationship, backref
import unittest #@UnresolvedImport
from sqlalchemy.orm import sessionmaker
from iri_tweet.utils import ObjectsBuffer

Base = declarative_base()

class User(Base):
    __tablename__ = 'users'
    
    id = Column(Integer, primary_key=True)
    name = Column(String)
    fullname = Column(String)
    password = Column(String)
    
    def __init__(self, name, fullname, password):
        self.name = name
        self.fullname = fullname
        self.password = password
    
    def __repr__(self):
        return "<User('%s','%s', '%s')>" % (self.name, self.fullname, self.password)


class Address(Base):
    __tablename__ = 'addresses'
    id = Column(Integer, primary_key=True)
    email_address = Column(String, nullable=False)
    user_id = Column(Integer, ForeignKey('users.id'))
    
    user = relationship("User", backref=backref('addresses', order_by=id))
    
    def __init__(self, user_id, email_address):
        self.email_address = email_address
        self.user_id = user_id
    
    def __repr__(self):
        return "<Address('%s')>" % self.email_address



class TestObjectBuffer(unittest.TestCase):
    
    def setUp(self):
        self.engine = create_engine('sqlite:///:memory:', echo=False)
        Base.metadata.create_all(self.engine)
        sessionMaker = sessionmaker(bind=self.engine)
        self.session = sessionMaker()
        
    def testCreateUser(self):
        ed_user = User('ed', 'Ed Jones', 'edspassword')
        self.session.add(ed_user)
        self.assertTrue(ed_user.id is None)
        self.session.commit()
        self.assertTrue(ed_user.id is not None)
        
    def testSimpleBuffer(self):
        obj_buffer = ObjectsBuffer()
        obj_proxy = obj_buffer.add_object(User, ['ed1', 'Ed1 Jones', 'edspassword'], None, False)
        self.assertTrue(obj_proxy.id() is None)
        obj_buffer.persists(self.session)
        self.assertTrue(obj_proxy.id() is None)
        self.session.commit()
        self.assertTrue(obj_proxy.id() is not None)
        
    def testSimpleBufferFlush(self):
        obj_buffer = ObjectsBuffer()
        obj_proxy = obj_buffer.add_object(User, ['ed2', 'Ed2 Jones', 'edspassword'], None, True)
        self.assertTrue(obj_proxy.id() is None)
        obj_buffer.persists(self.session)
        self.assertTrue(obj_proxy.id() is not None)
        self.session.commit()
        self.assertTrue(obj_proxy.id() is not None)
        
    def testRelationBuffer(self):
        obj_buffer = ObjectsBuffer()
        user1_proxy = obj_buffer.add_object(User, ['ed3', 'Ed3 Jones', 'edspassword'], None, True)
        obj_buffer.add_object(Address, [user1_proxy.id,'ed3@mail.com'], None, False)
        obj_buffer.add_object(Address, [user1_proxy.id,'ed3@other.com'], None, False)
        user2_proxy = obj_buffer.add_object(User, ['ed4', 'Ed3 Jones', 'edspassword'], None, True)
        obj_buffer.add_object(Address, [user2_proxy.id,'ed4@mail.com'], None, False)
        obj_buffer.persists(self.session)
        self.session.commit()
        ed_user = self.session.query(User).filter_by(name='ed3').first()
        self.assertEquals(2, len(ed_user.addresses))
        ed_user = self.session.query(User).filter_by(name='ed4').first()
        self.assertEquals(1, len(ed_user.addresses))

        
    def tearDown(self):
        self.session.close()


if __name__ == '__main__':
    unittest.main()