/*
* This file is part of the TraKERS\Middleware package.
*
* (c) IRI <http://www.iri.centrepompidou.fr/>
*
* For the full copyright and license information, please view the LICENSE
* file that was distributed with this source code.
*/

/*
 * Projet : TraKERS
 * Module : MIDDLEWARE
 * Sous-Module : Tracking
 * Classe : ModeManagement
 * 
 * Auteur : alexandre.bastien@iri.centrepompidou.fr
 * 
 * Fonctionnalités : Permet de gérer les différents modes qui découpent l'interaction avec le middleware.
 * Découpe l'interaction avec le middleware en différents modes.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Trakers.Communication;
using Trakers.MainModule.Events;
using Trakers.Debug;

namespace Trakers.MainModule
{
    public class ModeManagement
    {
        //Il s'agit de la fonctoin permettant de notifier le serveur en cas de changement de mode.
        public delegate void ModChangeHandler(object o, ModChangeEventArgs e);
        //L'événement on change de mode.
        public static event ModChangeHandler ModChangeEvent;
        
        //Accès au module d'interaction principal.
        private KinectMain kinectMain;
        //Accès au debug.
        private DebugWindow debug;
        //Accès au serveur.
        private Server server;
        //La liste des modes.
        public enum Mode {NO_USER, USER_INCOMING, MOSAIC, FILTER, VIDEO, TIMELINE, SEARCH};
        //Mode actuel.
        private Mode currentMode;

        /*
        *  Constructeur qui prend en paramètres le module principal et le serveur.
        */
        public ModeManagement(Server _server, DebugWindow _debug, KinectMain _kinectMain)
        {
            server = _server;
            debug = _debug;
            kinectMain = _kinectMain;
            server = _server;
            currentMode = Mode.NO_USER;

            //Fonction appelée lorsqu'on change de mode.
            ModChangeListener modChangeListener = new ModChangeListener();
            ModChangeEvent += new ModChangeHandler(modChangeListener.showAndSend);
        }

        /*
        *  Détecte tout changement de mode en se basant sur la proximité de l'utilisateur.
        */
        public void DetectProximityBasedModes(float proximity)
        {
            //Si l'utilisateur se déplace dans la zone de détection.
            //On traite le problème en plusieurs limites, on discrétise la zone.
            //On considère qu'il n'y a pas d'utilisateur si sa proximité est trop faible.
            if (proximity < 10f && currentMode != Mode.NO_USER)
            {
                currentMode = Mode.NO_USER;
                ModChangeEventArgs modChangeEvent = new ModChangeEventArgs(server, debug, "NO-USER");
                OnModChangeEvent(modChangeEvent);
            }
            //S'il n'est pas encore au point d'interaction maximale, on considère qu'il est en chemin.
            else if (proximity >= 10f && proximity < 100f && currentMode != Mode.USER_INCOMING)
            {
                currentMode = Mode.USER_INCOMING;
                ModChangeEventArgs modChangeEvent = new ModChangeEventArgs(server, debug, "USER-INCOMING");
                OnModChangeEvent(modChangeEvent);
            }
            //S'il est arrivé au point, il a la mosaïque complète.
            else if(proximity == 100f && currentMode != Mode.MOSAIC)
            {
                currentMode = Mode.MOSAIC;
                ModChangeEventArgs modChangeEvent = new ModChangeEventArgs(server, debug, "MOSAIC");
                OnModChangeEvent(modChangeEvent);
            }
        }

        /*
        *  Détecte si l'utilisateur veut filtrer la mosaïque.
        */
        public void DetectFilterMode()
        {
            //Ne fonctionne que si on se trouve dans la mosaïque.
            if (currentMode == Mode.MOSAIC)
            {
                currentMode = Mode.FILTER;
                ModChangeEventArgs modChangeEvent = new ModChangeEventArgs(server, debug, "FILTER");
                OnModChangeEvent(modChangeEvent);
            }
        }

        /*
        *  Détecte si l'utilisateur veut entrer dans une vidéo.
        */
        public void DetectVideoMode()
        {
            //Si on vient de la mosaïque, d'un filtrage, de la timeline ou d'une recherche, on peut entrer.
            if (currentMode == Mode.MOSAIC || currentMode == Mode.FILTER ||
            currentMode == Mode.TIMELINE || currentMode == Mode.SEARCH)
            {
                currentMode = Mode.VIDEO;
                ModChangeEventArgs modChangeEvent = new ModChangeEventArgs(server, debug, "VIDEO");
                OnModChangeEvent(modChangeEvent);
            }
        }

        /*
        *  Détecte si l'utilisateur veut entrer dans la timeline.
        */
        public void DetectTimeLineMode()
        {
            //Si on est dans une vidéo, on peut entrer.
            if (currentMode == Mode.VIDEO)
            {
                currentMode = Mode.TIMELINE;
                ModChangeEventArgs modChangeEvent = new ModChangeEventArgs(server, debug, "TIMELINE");
                OnModChangeEvent(modChangeEvent);
            }
        }

        /*
        *  Détecte si l'utilisateur veut effectuer une recherche dans cette vidéo.
        */
        public void DetectSearchMode()
        {
            //Si on est dans une vidéo, on peut entrer.
            if (currentMode == Mode.VIDEO)
            {
                currentMode = Mode.SEARCH;
                ModChangeEventArgs modChangeEvent = new ModChangeEventArgs(server, debug, "SEARCH");
                OnModChangeEvent(modChangeEvent);
            }
        }

        /*
        *  Initialise l'événement et fait appel aux fonctions du listener quand on change de mode.
        */
        public static void OnModChangeEvent(ModChangeEventArgs e)
        {
            if (ModChangeEvent != null)
                ModChangeEvent(new object(), e);
        }
    }
}
