﻿/*
* This file is part of the TraKERS\Middleware package.
*
* (c) IRI <http://www.iri.centrepompidou.fr/>
*
* For the full copyright and license information, please view the LICENSE_MIDDLEWARE
* file that was distributed with this source code.
*/

/*
 * Projet : TraKERS
 * Module : MIDDLEWARE
 * Sous-Module : Debug
 * Classe : DebugWindow
 * 
 * Auteur : alexandre.bastien@iri.centrepompidou.fr
 * 
 * Fonctionnalités : Reçoit des notifications des sous-modules Tracking, Communication et Exception.
 * Intéragit avec la fenêtre XAML de debug de façon à afficher un rendu visuel.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using System.Drawing;
using System.Windows.Media.Media3D;
using Microsoft.Kinect;

using Coding4Fun.Kinect.Wpf;

using Trakers.Tracking;
using System.Threading;
using Trakers.Tracking.Events;
using Trakers.Tracking.Gestures;
using System.Resources;
using System.Reflection;
using System.Timers;
using Trakers.Communication;

namespace Trakers.Debug
{
    public partial class DebugWindow : Window
    {
        //Gestionnaire de ressources.
        private ResourceManager rm;
        //Timer.
        private System.Timers.Timer _timer;
        //Membre permettant d'atteindre la classe KinectMain du sous-module Tracking.
        private KinectMain kinectMain;
        //Tableau contenant une image en couleurs.
        private byte[] colorPixelData;
        //Indique si la kinect est allumée/éteinte.
        private bool on;
        //Indique si la fenêtre de debug est actuellement en cours de fermeture.
        private bool closing;
        //Indique si l'image doit être raffraichie.
        private Boolean refreshImage;

        /*
        * Constructeur : Affiche la fenêtre de debug en lui passant en paramètre une instanciation de la
        * classe KinectMain.
        * Au départ, la kinect est éteinte.
        */
        public DebugWindow(KinectMain main)
        {
            rm = new ResourceManager("Trakers.Properties.resources", Assembly.GetExecutingAssembly());
            InitializeComponent();
            kinectMain = main;
            on = true;
            closing = false;
            refreshImage = true;

            try
            {
                //On instancie le timer à N ms.
                _timer = new System.Timers.Timer(kinectMain.getTimerElapsing());
                //Dès que le timer est expiré, on appelle _timer_Elapsed.
                _timer.Elapsed += new ElapsedEventHandler(_timer_Elapsed);
            }
            catch (Exception){}
        }

        /*
         * Méthode appelée à l'expiration du timer pour les gestures et modes.
         */
        public void _timer_Elapsed(object sender, ElapsedEventArgs e)
        {
            //On débloque le raffraichissement de l'image.
            refreshImage = true;
            //On arrête le timer.
            _timer.Stop();
        }

        /*
        * Getter pour le membre indiquant la fermeture de la fenêtre de debug.
        */
        public bool isClosing()
        {
            return closing;
        }

        /*
        * Est appelée à la fermeture de la fenêtre.
        */
        private void Window_Closed(object sender, EventArgs e)
        {
            closing = true;
            //On éteint la Kinect (pour éviter qu'elle reste allumée même lorsque le programme est éteint).
            kinectMain.KinectClose();
        }

        /*
        * Bouton ON/OFF.
        */
        private void Switch_Click(object sender, RoutedEventArgs e)
        {
            //S'il valait faux, il vaut vrai maintenant et inversement.
            on = !on;
            //Si la kinect est allumée.
            if (on)
            {
                //Il affiche OFF (pour éteindre la kinect).
                Switch.Content = "OFF";
                //On vide le label des exceptions.
                ExceptionLbl.Content = "";
                //On initialise la Kinect.
                kinectMain.KinectInitialization();
            }
            else
            {
                //Il affiche ON (pour allumer la kinect).
                Switch.Content = "ON";
                //On remet à zéro tous les éléments du retour visuel.
                R1.Fill = System.Windows.Media.Brushes.DarkGray;
                R2.Fill = System.Windows.Media.Brushes.DarkGray;
                R3.Fill = System.Windows.Media.Brushes.DarkGray;
                R4.Fill = System.Windows.Media.Brushes.DarkGray;
                DistanceLbl.Content = "Distance :";
                LeftHand.Background = System.Windows.Media.Brushes.DarkGray;
                LeftHand.Content = "";
                RightHand.Background = System.Windows.Media.Brushes.DarkGray;
                RightHand.Content = "";
                DebugCanvas.Children.RemoveRange(1, DebugCanvas.Children.Count - 1);
                //On éteint la Kinect.
                kinectMain.KinectClose();
            }
        }

        /*
        *  Récupère le flux video et met à jour le rendu visuel de debug.
        */
        public void RefreshVideo(AllFramesReadyEventArgs e)
        {
            if (refreshImage)
            {
                bool receivedData = false;
                ColorImageFrame colorImageFrameData;
                using (colorImageFrameData = e.OpenColorImageFrame())
                {
                    //Si on ne reçoit pas de trames de la kinect.
                    if (colorImageFrameData == null)
                    {
                        //L'image est supprimée.
                        DebugImage.Source = null;
                    }
                    //Si le tableau stockant l'image en cours est nul.
                    if (colorPixelData == null)
                        //On alloue un nouveau tableau.
                        colorPixelData = new byte[colorImageFrameData.PixelDataLength];
                    else
                    {
                        try
                        {
                            //Sinon on met à jour le tableau en copiant le contenu de la trame dans le tableau.
                            colorImageFrameData.CopyPixelDataTo(colorPixelData);
                            receivedData = true;
                        }
                        catch (Exception) { }
                    }
                }
                //Si on a des données dans le tableau et que la kinect est allumée.
                if (receivedData && on)
                {
                    //On met à jour l'image de la caméra.
                    DebugImage.Source = BitmapSource.Create(colorImageFrameData.Width, colorImageFrameData.Height, 96, 96, PixelFormats.Bgr32, null, colorPixelData, colorImageFrameData.Width * colorImageFrameData.BytesPerPixel);
                    DebugImage.HorizontalAlignment = System.Windows.HorizontalAlignment.Center;
                    DebugImage.VerticalAlignment = System.Windows.VerticalAlignment.Center;
                    DebugImage.Stretch = Stretch.Fill;
                    //On annule l'image associée aux gestures.
                    Gestures.Source = null;
                }
            }
        }

        /*
        * Affiche la distance de l'utilisateur dans le rendu visuel.
        * Sous forme de nombre en m et de rectangles changeant de couleur en fonction de la distance.
        */
        public void showDistance(float proximity)
        {
            DistanceLbl.Content = "Proximity : " + proximity + "%";

            if (proximity == 0)
            {
                R0.Fill = System.Windows.Media.Brushes.DarkGray;
                R1.Fill = System.Windows.Media.Brushes.DarkGray;
                R2.Fill = System.Windows.Media.Brushes.DarkGray;
                R3.Fill = System.Windows.Media.Brushes.DarkGray;
                R4.Fill = System.Windows.Media.Brushes.DarkGray;
                R5.Fill = System.Windows.Media.Brushes.DarkGray;
                R6.Fill = System.Windows.Media.Brushes.DarkGray;
                R7.Fill = System.Windows.Media.Brushes.DarkGray;
                R8.Fill = System.Windows.Media.Brushes.DarkGray;
                R9.Fill = System.Windows.Media.Brushes.DarkGray;
                R10.Fill = System.Windows.Media.Brushes.DarkGray;
            }
            else if (proximity > 0 && proximity < 10)
            {
                R0.Fill = System.Windows.Media.Brushes.DarkGray;
                R1.Fill = System.Windows.Media.Brushes.DarkGray;
                R2.Fill = System.Windows.Media.Brushes.DarkGray;
                R3.Fill = System.Windows.Media.Brushes.DarkGray;
                R4.Fill = System.Windows.Media.Brushes.DarkGray;
                R5.Fill = System.Windows.Media.Brushes.DarkGray;
                R6.Fill = System.Windows.Media.Brushes.DarkGray;
                R7.Fill = System.Windows.Media.Brushes.DarkGray;
                R8.Fill = System.Windows.Media.Brushes.DarkGray;
                R9.Fill = System.Windows.Media.Brushes.DarkGray;
                R10.Fill = System.Windows.Media.Brushes.Red;
            }
            else if (proximity > 10 && proximity < 20)
            {
                R0.Fill = System.Windows.Media.Brushes.DarkGray;
                R1.Fill = System.Windows.Media.Brushes.DarkGray;
                R2.Fill = System.Windows.Media.Brushes.DarkGray;
                R3.Fill = System.Windows.Media.Brushes.DarkGray;
                R4.Fill = System.Windows.Media.Brushes.DarkGray;
                R5.Fill = System.Windows.Media.Brushes.DarkGray;
                R6.Fill = System.Windows.Media.Brushes.DarkGray;
                R7.Fill = System.Windows.Media.Brushes.DarkGray;
                R8.Fill = System.Windows.Media.Brushes.DarkGray;
                R9.Fill = System.Windows.Media.Brushes.Red;
                R10.Fill = System.Windows.Media.Brushes.Red;
            }
            else if (proximity > 20 && proximity < 30)
            {
                R0.Fill = System.Windows.Media.Brushes.DarkGray;
                R1.Fill = System.Windows.Media.Brushes.DarkGray;
                R2.Fill = System.Windows.Media.Brushes.DarkGray;
                R3.Fill = System.Windows.Media.Brushes.DarkGray;
                R4.Fill = System.Windows.Media.Brushes.DarkGray;
                R5.Fill = System.Windows.Media.Brushes.DarkGray;
                R6.Fill = System.Windows.Media.Brushes.DarkGray;
                R7.Fill = System.Windows.Media.Brushes.DarkGray;
                R8.Fill = System.Windows.Media.Brushes.Red;
                R9.Fill = System.Windows.Media.Brushes.Red;
                R10.Fill = System.Windows.Media.Brushes.Red;
            }
            else if (proximity > 30 && proximity < 40)
            {
                R0.Fill = System.Windows.Media.Brushes.DarkGray;
                R1.Fill = System.Windows.Media.Brushes.DarkGray;
                R2.Fill = System.Windows.Media.Brushes.DarkGray;
                R3.Fill = System.Windows.Media.Brushes.DarkGray;
                R4.Fill = System.Windows.Media.Brushes.DarkGray;
                R5.Fill = System.Windows.Media.Brushes.DarkGray;
                R6.Fill = System.Windows.Media.Brushes.DarkGray;
                R7.Fill = System.Windows.Media.Brushes.Orange;
                R8.Fill = System.Windows.Media.Brushes.Red;
                R9.Fill = System.Windows.Media.Brushes.Red;
                R10.Fill = System.Windows.Media.Brushes.Red;
            }
            else if (proximity > 40 && proximity < 50)
            {
                R0.Fill = System.Windows.Media.Brushes.DarkGray;
                R1.Fill = System.Windows.Media.Brushes.DarkGray;
                R2.Fill = System.Windows.Media.Brushes.DarkGray;
                R3.Fill = System.Windows.Media.Brushes.DarkGray;
                R4.Fill = System.Windows.Media.Brushes.DarkGray;
                R5.Fill = System.Windows.Media.Brushes.DarkGray;
                R6.Fill = System.Windows.Media.Brushes.Orange;
                R7.Fill = System.Windows.Media.Brushes.Orange;
                R8.Fill = System.Windows.Media.Brushes.Red;
                R9.Fill = System.Windows.Media.Brushes.Red;
                R10.Fill = System.Windows.Media.Brushes.Red;
            }
            else if (proximity > 50 && proximity < 60)
            {
                R0.Fill = System.Windows.Media.Brushes.DarkGray;
                R1.Fill = System.Windows.Media.Brushes.DarkGray;
                R2.Fill = System.Windows.Media.Brushes.DarkGray;
                R3.Fill = System.Windows.Media.Brushes.DarkGray;
                R4.Fill = System.Windows.Media.Brushes.DarkGray;
                R5.Fill = System.Windows.Media.Brushes.Orange;
                R6.Fill = System.Windows.Media.Brushes.Orange;
                R7.Fill = System.Windows.Media.Brushes.Orange;
                R8.Fill = System.Windows.Media.Brushes.Red;
                R9.Fill = System.Windows.Media.Brushes.Red;
                R10.Fill = System.Windows.Media.Brushes.Red;
            }
            else if (proximity > 60 && proximity < 70)
            {
                R0.Fill = System.Windows.Media.Brushes.DarkGray;
                R1.Fill = System.Windows.Media.Brushes.DarkGray;
                R2.Fill = System.Windows.Media.Brushes.DarkGray;
                R3.Fill = System.Windows.Media.Brushes.DarkGray;
                R4.Fill = System.Windows.Media.Brushes.Yellow;
                R5.Fill = System.Windows.Media.Brushes.Orange;
                R6.Fill = System.Windows.Media.Brushes.Orange;
                R7.Fill = System.Windows.Media.Brushes.Orange;
                R8.Fill = System.Windows.Media.Brushes.Red;
                R9.Fill = System.Windows.Media.Brushes.Red;
                R10.Fill = System.Windows.Media.Brushes.Red;
            }
            else if (proximity > 70 && proximity < 80)
            {
                R0.Fill = System.Windows.Media.Brushes.DarkGray;
                R1.Fill = System.Windows.Media.Brushes.DarkGray;
                R2.Fill = System.Windows.Media.Brushes.DarkGray;
                R3.Fill = System.Windows.Media.Brushes.Yellow;
                R4.Fill = System.Windows.Media.Brushes.Yellow;
                R5.Fill = System.Windows.Media.Brushes.Orange;
                R6.Fill = System.Windows.Media.Brushes.Orange;
                R7.Fill = System.Windows.Media.Brushes.Orange;
                R8.Fill = System.Windows.Media.Brushes.Red;
                R9.Fill = System.Windows.Media.Brushes.Red;
                R10.Fill = System.Windows.Media.Brushes.Red;
            }
            else if (proximity > 80 && proximity < 90)
            {
                R0.Fill = System.Windows.Media.Brushes.DarkGray;
                R1.Fill = System.Windows.Media.Brushes.DarkGray;
                R2.Fill = System.Windows.Media.Brushes.Yellow;
                R3.Fill = System.Windows.Media.Brushes.Yellow;
                R4.Fill = System.Windows.Media.Brushes.Yellow;
                R5.Fill = System.Windows.Media.Brushes.Orange;
                R6.Fill = System.Windows.Media.Brushes.Orange;
                R7.Fill = System.Windows.Media.Brushes.Orange;
                R8.Fill = System.Windows.Media.Brushes.Red;
                R9.Fill = System.Windows.Media.Brushes.Red;
                R10.Fill = System.Windows.Media.Brushes.Red;
            }
            else if (proximity > 90 && proximity < 100)
            {
                R0.Fill = System.Windows.Media.Brushes.DarkGray;
                R1.Fill = System.Windows.Media.Brushes.Yellow;
                R2.Fill = System.Windows.Media.Brushes.Yellow;
                R3.Fill = System.Windows.Media.Brushes.Yellow;
                R4.Fill = System.Windows.Media.Brushes.Yellow;
                R5.Fill = System.Windows.Media.Brushes.Orange;
                R6.Fill = System.Windows.Media.Brushes.Orange;
                R7.Fill = System.Windows.Media.Brushes.Orange;
                R8.Fill = System.Windows.Media.Brushes.Red;
                R9.Fill = System.Windows.Media.Brushes.Red;
                R10.Fill = System.Windows.Media.Brushes.Red;
            }
            else if (proximity == 100)
            {
                R0.Fill = System.Windows.Media.Brushes.Green;
                R1.Fill = System.Windows.Media.Brushes.Yellow;
                R2.Fill = System.Windows.Media.Brushes.Yellow;
                R3.Fill = System.Windows.Media.Brushes.Yellow;
                R4.Fill = System.Windows.Media.Brushes.Yellow;
                R5.Fill = System.Windows.Media.Brushes.Orange;
                R6.Fill = System.Windows.Media.Brushes.Orange;
                R7.Fill = System.Windows.Media.Brushes.Orange;
                R8.Fill = System.Windows.Media.Brushes.Red;
                R9.Fill = System.Windows.Media.Brushes.Red;
                R10.Fill = System.Windows.Media.Brushes.Red;
            }
        }

        /*
        * Affiche la détection de la main droite via un label.
        */
        public void showRightHandRect(bool show)
        {
            if (show)
                RightHand.Background = System.Windows.Media.Brushes.Blue;
            else
                RightHand.Background = System.Windows.Media.Brushes.DarkGray;
        }

        /*
        * Affiche la détection de la main gauche via un label.
        */
        public void showLeftHandRect(bool show)
        {
            if (show)
                LeftHand.Background = System.Windows.Media.Brushes.Blue;
            else
                LeftHand.Background = System.Windows.Media.Brushes.DarkGray;
        }

        /*
        * Dessine les noeuds du squelette dans le rendu visuel.
        */
        public void drawJoints(JointCollection joints, Skeleton first)
        {
            if (refreshImage)
            {
                //On enlève tout élément du Canvas à part l'image, de manière à mettre à jour la position du squelette. 
                DebugCanvas.Children.RemoveRange(1, DebugCanvas.Children.Count - 1);

                //Pour chaque noeud.
                foreach (Joint joint in first.Joints)
                {
                    //On crée une ellipse de taille 20 et de largeur 20.
                    Ellipse node = new Ellipse();
                    node.Height = 20;
                    node.Width = 20;

                    //S'il s'agit d'un noeud de tête, on le colorie en rouge, sinon en bleu.
                    if (joint.JointType == JointType.Head)
                        node.Fill = System.Windows.Media.Brushes.Red;
                    else if (joint.JointType == JointType.ShoulderCenter)
                        node.Fill = System.Windows.Media.Brushes.Green;
                    else if (joint.JointType == JointType.HipCenter)
                        node.Fill = System.Windows.Media.Brushes.Green;
                    else if (joint.JointType == JointType.HandRight)
                        node.Fill = System.Windows.Media.Brushes.Red;
                    else
                        node.Fill = System.Windows.Media.Brushes.Blue;

                    //On met à la bonne échelle les coordonnées des positions des noeuds.
                    Joint scaledJoint = Coding4Fun.Kinect.Wpf.SkeletalExtensions.ScaleTo(joint, 600, 400, 0.75f, 0.75f);

                    //On positionne le noeud dans le Canvas, et on l'ajoute.
                    Canvas.SetLeft(node, scaledJoint.Position.X);
                    Canvas.SetTop(node, scaledJoint.Position.Y);
                    DebugCanvas.Children.Add(node);
                }
            }
        }

        /*
        * Dessine un os, en ayant en paramètres deux noeuds.
        */
        public void drawBone(Joint j1, Joint j2)
        {
            //On crée une nouvelle ligne verte d'épaisseur 8 entre les deux noeuds et on l'ajoute au Canvas.
            Line line = new Line();
            line.Stroke = System.Windows.Media.Brushes.Green;
            line.X1 = j1.Position.X;
            line.X2 = j2.Position.X;
            line.Y1 = j1.Position.Y;
            line.Y2 = j2.Position.Y;
            line.StrokeThickness = 8;
            DebugCanvas.Children.Add(line);
        }

        /*
        * Dessine le squelette (ensemble des os), en ayant en paramètres tous les noeuds.
        */
        public void showSkeleton(Joint hipCenter, Joint spine, Joint shoulderCenter, Joint head, Joint shoulderLeft, Joint elbowLeft, Joint wristLeft, Joint handLeft, Joint shoulderRight, Joint elbowRight, Joint wristRight, Joint handRight, Joint hipLeft, Joint kneeLeft, Joint ankleLeft, Joint footLeft, Joint hipRight, Joint kneeRight, Joint ankleRight, Joint footRight)
        {
            if (refreshImage)
            {
                //On met les noeuds deux par deux en fonction de leur position dans le squelette.
                drawBone(head, shoulderCenter);
                drawBone(shoulderCenter, shoulderLeft);
                drawBone(shoulderLeft, elbowLeft);
                drawBone(elbowLeft, wristLeft);
                drawBone(wristLeft, handLeft);
                drawBone(shoulderCenter, shoulderRight);
                drawBone(shoulderRight, elbowRight);
                drawBone(elbowRight, wristRight);
                drawBone(wristRight, handRight);
                drawBone(shoulderCenter, spine);
                drawBone(spine, hipCenter);
                drawBone(hipCenter, hipLeft);
                drawBone(hipLeft, kneeLeft);
                drawBone(kneeLeft, ankleLeft);
                drawBone(ankleLeft, footLeft);
                drawBone(hipCenter, hipRight);
                drawBone(hipRight, kneeRight);
                drawBone(kneeRight, ankleRight);
                drawBone(ankleRight, footRight);
            }
        }

        /*
        * Cache le squelette et le reste de l'interface à part l'image.
        */
        public void hideSkeleton()
        {
            //On vide le canvas mais en gardant l'image.
            if(DebugCanvas.Children.Count > 1)
                DebugCanvas.Children.RemoveRange(1, DebugCanvas.Children.Count - 1);
            //On colore en gris tous les indicateurs.
            R0.Fill = System.Windows.Media.Brushes.DarkGray;
            R1.Fill = System.Windows.Media.Brushes.DarkGray;
            R2.Fill = System.Windows.Media.Brushes.DarkGray;
            R3.Fill = System.Windows.Media.Brushes.DarkGray;
            R4.Fill = System.Windows.Media.Brushes.DarkGray;
            R5.Fill = System.Windows.Media.Brushes.DarkGray;
            R6.Fill = System.Windows.Media.Brushes.DarkGray;
            R7.Fill = System.Windows.Media.Brushes.DarkGray;
            R8.Fill = System.Windows.Media.Brushes.DarkGray;
            R9.Fill = System.Windows.Media.Brushes.DarkGray;
            R10.Fill = System.Windows.Media.Brushes.DarkGray;
            LeftHand.Background = System.Windows.Media.Brushes.DarkGray;
            RightHand.Background = System.Windows.Media.Brushes.DarkGray;
        }

        /*
        * Affiche la position de la main gauche dans le rendu visuel.
        */
        public void showLeftHandCoord(String coord)
        {
            LeftHand.Content = coord;
        }

        /*
        * Affiche la position de la main gauche dans le rendu visuel.
        */
        public void showRightHandCoord(String coord)
        {
            RightHand.Content = coord;
        }

        /*
         * Méthode associée à l'événement : Ouvrir la fenêtre de paramétrage via le menu.
         */
        private void Parameters_Click(object sender, RoutedEventArgs e)
        {
            DebugParameters param = new DebugParameters(kinectMain);

            try
            {
                param.ShowDialog();
            }
            catch (Exception)
            {
                ExceptionLbl.Content = rm.GetString("loadParamFail");
            }
        }

        /*
         * Méthode associée à l'événement : Quitter via le menu.
         */
        private void Quit_Click(object sender, RoutedEventArgs e)
        {
            closing = true;
            //On éteint la Kinect (pour éviter qu'elle reste allumée même lorsque le programme est éteint).
            kinectMain.KinectClose();
            Application.Current.Shutdown();
        }

        /*
         * Permet d'initialiser la Kinect dès que la fenêtre est lancée.
         */
        private void Window_Loaded(object sender, RoutedEventArgs e)
        {
            kinectMain.KinectInitialization();
        }

        /*
         * Méthode d'affichage des gestures.
         */
        public void showGesture(String gesture)
        {
            if (refreshImage)
            {
                refreshImage = false;
                _timer.Start();
                Bitmap bitmap = null;
                //S'il s'agit de telle ou telle gesture, on prend l'image correspondante dans les ressources,
                //on la convertit et on l'affiche.
                switch (gesture)
                {
                    case "SWIPE-LEFT": bitmap = (Bitmap)rm.GetObject("swipe_left");
                        break;
                    case "SWIPE-RIGHT": bitmap = (Bitmap)rm.GetObject("swipe_right");
                        break;
                    case "PUSH-RIGHT": bitmap = (Bitmap)rm.GetObject("push_right");
                        break;
                    case "PUSH-LEFT": bitmap = (Bitmap)rm.GetObject("push_left");
                        break;
                    case "PUSH-BOTH": bitmap = (Bitmap)rm.GetObject("push_both");
                        break;
                    case "PULL-RIGHT": bitmap = (Bitmap)rm.GetObject("pull_right");
                        break;
                    case "PULL-LEFT": bitmap = (Bitmap)rm.GetObject("pull_left");
                        break;
                    case "PULL-BOTH": bitmap = (Bitmap)rm.GetObject("pull_both");
                        break;
                }
                Gestures.Source = CreateBitmapSourceFromBitmap(bitmap);
            }

            DebugImage.Source = null;
            hideSkeleton();
        }

        /*
         * Méthode d'indication de raffraichissement de l'image ("on la raffraichit ou pas ?").
         */
        public void setRefreshImage(bool refresh)
        {
            refreshImage = refresh;
        }

        /*
         * Méthode de conversion de Bitmap (des ressources) en BitmapSource (du debug).
         */
        public static BitmapSource CreateBitmapSourceFromBitmap(Bitmap bitmap)
        {
            if (bitmap == null)
                return null;

            return System.Windows.Interop.Imaging.CreateBitmapSourceFromHBitmap(
                bitmap.GetHbitmap(),
                IntPtr.Zero,
                Int32Rect.Empty,
                BitmapSizeOptions.FromEmptyOptions());
        }
    }
}
