﻿/*
 * Projet : TraKERS
 * Module : MIDDLEWARE
 * Sous-Module : Tracking/Gestures
 * Classe : SwipeDetector
 * 
 * Auteur : alexandre.bastien@iri.centrepompidou.fr
 * 
 * Fonctionnalités : Permet de détecter si l'utilisateur a effectué un Swipe, en se basant sur
 * des règles appliquées à la positions des noeuds dans le temps.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Microsoft.Kinect;

namespace Trakers.Tracking.Gestures
{
    public class SwipeDetector : GestureDetector
    {
        public enum Direction {LEFT, RIGHT, TOP, DOWN};

        Debug.DebugWindow debug;

        public SwipeDetector(Debug.DebugWindow _d) : base()
        {
            debug = _d;
            gesturePeriod = (float)0.5;
            indexesPerSecond = 30;
            indexesToCheck = (int)(gesturePeriod * indexesPerSecond);
        }

        /*
         * Lit les noeuds de l'historique du squelette afin de détecter un Swipe left.
         * Règles :
         * Se fait avec la main droite.
         * Chaque nouvelle position de la main doit être à la gauche de la précédente.
         * Chaque nouvelle position de la main ne doit pas s'éloigner verticalement de plus d'une certaine distance.
         * Le geste doit mesurer horizontalement une certaine distance.
         */
        public bool CheckForSwipeLeft()
        {
            //Crée un historique de squelette local, puisque l'historique est mis à jour toutes les ~1/30 s.
            List<List<Joint>> localHistory = new List<List<Joint>>(history);

            //Si il n'y a pas assez de positions dans l'historique local pour vérifier le geste.
            if (localHistory.Count < indexesToCheck+1)
                return false;
            
            //La distance de référence est ici la distance entre le milieu du dos et le milieu des épaules.
            refDistance = Math.Abs(localHistory[0][spineID].Position.Y - localHistory[0][shoulderCenterID].Position.Y);
            //On commence la position pour les indexesToCheck dernières postures (celle à l'index 0 étant la dernière).
            startPoint = localHistory[localHistory.Count - indexesToCheck][handRightID].Position;

            //De la position p1 à pn, on suit l'algorithme.
            for (int i = localHistory.Count - indexesToCheck + 1; i < localHistory.Count; i++)
            {
                //Debug temporaire pour vérifier la validité de certaines contraintes durant la gesture.
               /*if (localHistory[i][handRightID].Position.Y > localHistory[i][headID].Position.Y)
                    debug.R1.Fill = System.Windows.Media.Brushes.Blue;
                else
                    debug.R1.Fill = System.Windows.Media.Brushes.DarkGray;

                if(localHistory[i][handRightID].Position.Y < localHistory[i][hipCenterID].Position.Y)
                    debug.R2.Fill = System.Windows.Media.Brushes.Blue;
                else
                    debug.R2.Fill = System.Windows.Media.Brushes.DarkGray;

                if(localHistory[i][handRightID].Position.X < localHistory[i - 1][handRightID].Position.X)
                    debug.R3.Fill = System.Windows.Media.Brushes.Blue;
                else
                    debug.R3.Fill = System.Windows.Media.Brushes.DarkGray;

                if(Math.Abs(localHistory[i][handRightID].Position.Y - startPoint.Y) < refDistance/2)
                    debug.R4.Fill = System.Windows.Media.Brushes.Blue;
                else
                    debug.R4.Fill = System.Windows.Media.Brushes.DarkGray;*/

                //Si la position Y de la main est plus haute que la tête
                //OU si la position Y de la main est plus basse que la hanche
                //OU si la nouvelle position X de la main est à droite de la précédente
                //OU si la nouvelle position Y de la main est plus éloignée de la distance N par rapport à la première position Y
                //Alors on retourne faux.
                if (localHistory[i][handRightID].Position.Y < localHistory[i][headID].Position.Y ||
                localHistory[i][handRightID].Position.Y > localHistory[i][hipCenterID].Position.Y ||
                localHistory[i][handRightID].Position.X > localHistory[i - 1][handRightID].Position.X ||
                Math.Abs(localHistory[i][handRightID].Position.Y - startPoint.Y) > refDistance/2)
                    return false;
            }

            //Si la distance horizontale du geste a été plus courte que la distance N
            //Alors on retourne faux.
            //float dist = (localHistory[localHistory.Count - 1][handRightID].Position.X - localHistory[localHistory.Count - indexesToCheck][handRightID].Position.X);
            
            if (Math.Abs(localHistory[0][handRightID].Position.X - localHistory[localHistory.Count - indexesToCheck][handRightID].Position.X) < refDistance/2)
                return false;

            //Si la dernière position de la main droite est sur le côté droit du corps
            //OU si la première position calculée de la main droite est sur le côté gauche du corps
            //Alors on retourne faux.
            if(localHistory[localHistory.Count - 1][handRightID].Position.X > localHistory[localHistory.Count - 1][hipCenterID].Position.X ||
               localHistory[localHistory.Count - indexesToCheck][handRightID].Position.X < localHistory[localHistory.Count - 1][hipCenterID].Position.X)
                return false;

            //System.Console.Out.WriteLine(Math.Abs(localHistory[0][handRightID].Position.X - localHistory[localHistory.Count - indexesToCheck][handRightID].Position.X) + " < " + refDistance/2);
            //On supprime l'historique local.
            localHistory.Clear();
            //Si on est arrivé jusqu'ici, toutes les conditions pour un swipe left ont été remplies.
            return true;
        }

        /*
         * Lit les noeuds de l'historique du squelette afin de détecter un Swipe right.
         * Règles :
         * Se fait avec la main gauche.
         * Chaque nouvelle position de la main doit être à la droite de la précédente.
         * Chaque nouvelle position de la main ne doit pas s'éloigner verticalement de plus d'une certaine distance.
         * Le geste doit mesurer horizontalement une certaine distance.
         */
        public bool CheckForSwipeRight()
        {
            //Crée un historique de squelette local, puisque l'historique est mis à jour toutes les ~1/30 s.
            List<List<Joint>> localHistory = new List<List<Joint>>(history);

            //Si il n'y a pas assez de positions dans l'historique local pour vérifier le geste.
            if (localHistory.Count < indexesToCheck + 1)
                return false;

            //La distance de référence est ici la distance entre le milieu du dos et le milieu des épaules.
            refDistance = Math.Abs(localHistory[0][spineID].Position.Y - localHistory[0][shoulderCenterID].Position.Y);
            //On commence la position pour les indexesToCheck dernières postures (celle à l'index 0 étant la dernière).
            startPoint = localHistory[localHistory.Count - indexesToCheck][handLeftID].Position;

            //De la position p1 à pn, on suit l'algorithme.
            for (int i = localHistory.Count - indexesToCheck + 1; i < localHistory.Count; i++)
            {
                //Debug temporaire pour vérifier la validité de certaines contraintes durant la gesture.
                /*if (localHistory[i][handLeftID].Position.Y > localHistory[i][headID].Position.Y)
                    debug.R1.Fill = System.Windows.Media.Brushes.Cyan;
                else
                    debug.R1.Fill = System.Windows.Media.Brushes.DarkGray;

                if (localHistory[i][handLeftID].Position.Y < localHistory[i][hipCenterID].Position.Y)
                    debug.R2.Fill = System.Windows.Media.Brushes.Cyan;
                else
                    debug.R2.Fill = System.Windows.Media.Brushes.DarkGray;

                if (localHistory[i][handLeftID].Position.X > localHistory[i - 1][handLeftID].Position.X)
                    debug.R3.Fill = System.Windows.Media.Brushes.Cyan;
                else
                    debug.R3.Fill = System.Windows.Media.Brushes.DarkGray;

                if (Math.Abs(localHistory[i][handLeftID].Position.Y - startPoint.Y) < refDistance / 2)
                    debug.R4.Fill = System.Windows.Media.Brushes.Cyan;
                else
                    debug.R4.Fill = System.Windows.Media.Brushes.DarkGray;*/

                //Si la position Y de la main est plus haute que la tête
                //OU si la position Y de la main est plus basse que la hanche
                //OU si la nouvelle position X de la main est à gauche de la précédente
                //OU si la nouvelle position Y de la main est plus éloignée de la distance N par rapport à la première position Y
                //Alors on retourne faux.
                if (localHistory[i][handLeftID].Position.Y < localHistory[i][headID].Position.Y ||
                localHistory[i][handLeftID].Position.Y > localHistory[i][hipCenterID].Position.Y ||
                localHistory[i][handLeftID].Position.X < localHistory[i - 1][handLeftID].Position.X ||
                Math.Abs(localHistory[i][handLeftID].Position.Y - startPoint.Y) > refDistance / 2)
                    return false;
            }

            //Si la distance horizontale du geste a été plus courte que la distance N
            //Alors on retourne faux.
            //float dist = (localHistory[localHistory.Count - 1][handLeftID].Position.X - localHistory[localHistory.Count - indexesToCheck][handLeftID].Position.X);

            if (Math.Abs(localHistory[0][handLeftID].Position.X - localHistory[localHistory.Count - indexesToCheck][handLeftID].Position.X) < refDistance / 2)
                return false;

            //Si la dernière position de la main droite est sur le côté gauche du corps
            //OU si la première position calculée de la main droite est sur le côté droit du corps
            //Alors on retourne faux.
            if (localHistory[localHistory.Count - 1][handLeftID].Position.X < localHistory[localHistory.Count - 1][hipCenterID].Position.X ||
               localHistory[localHistory.Count - indexesToCheck][handLeftID].Position.X > localHistory[localHistory.Count - 1][hipCenterID].Position.X)
                return false;

            //System.Console.Out.WriteLine(Math.Abs(localHistory[0][handLeftID].Position.X - localHistory[localHistory.Count - indexesToCheck][handLeftID].Position.X) + " < " + refDistance / 2);
            //On supprime l'historique local.
            localHistory.Clear();
            //Si on est arrivé jusqu'ici, toutes les conditions pour un swipe right ont été remplies.
            return true;
        }
    }
}
