﻿/*
 * Projet : KINECT PROJECTS
 * Module : MIDDLEWARE
 * Sous-Module : Communication
 * Classe : Server
 * 
 * Auteur : alexandre.bastien@iri.centrepompidou.fr
 * 
 * Fonctionnalités : Reçoit des notifications du module sous-module Tracking.
 * Traduit les notifications sous forme de messages OSC et les envoie au Front Atelier.
 * Forme des messages :
 * - Notification de main dans le champ de recherche : Point3D indiquant la position de la main dans l'espace.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Threading;
using Tuio;
using System.Windows;
using Microsoft.Kinect;
using Trakers.Tracking;
using System.Windows.Media.Media3D;
using Trakers.Tracking.Events;

namespace Trakers.Communication
{
    public class Server
    {
        //Serveur TUIO, provenant de la DLL TuioServer créé par Bespoke.
        private TuioServer server;

        //Permet de savoir si un curseur pour la main gauche/droite a été créé.
        private bool leftHandCursorCreated;
        private bool rightHandCursorCreated;

        /*
        * Constructeur : On initialise le serveur avec une adresse et un port, au début les curseurs
        * ne sont pas créés et on indique au ThreadPool une fonction de callback de manière à vérifier
        * s'il reçoit des notifications.
        */
        public Server(String host, int port)
        {
            server = new TuioServer(host, port);
            leftHandCursorCreated = false;
            rightHandCursorCreated = false;
            ThreadPool.QueueUserWorkItem(ThreadPoolCallback);
        }

        /*
        * Getter du serveur.
        */
        public TuioServer getServer()
        {
            return server;
        }

        /*
        * Méthode appelée lors d'une notification de type : main gauche entrée dans le champ.
        */
        public void LeftHandTracked(object sender, LeftHandTrackedEventArgs e)
        {
            //Si le curseur de la main gauche n'est pas créé, alors on le crée.
            if (!leftHandCursorCreated)
            {
                server.AddTuioCursor(0, SkeletonPointToPoint3D(e.handJoint.Position));
                leftHandCursorCreated = true;
            }
            //S'il existe, on le met simplement à jour.
            else
            {
                server.UpdateTuioCursor(0, SkeletonPointToPoint3D(e.handJoint.Position));
            }
        }

        /*
        * Méthode appelée lors d'une notification de type : main droite entrée dans le champ.
        */
        public void RightHandTracked(object sender, RightHandTrackedEventArgs e)
        {
            //Si le curseur de la main droite n'est pas créé, alors on le crée.
            if (!rightHandCursorCreated)
            {
                server.AddTuioCursor(1, SkeletonPointToPoint3D(e.handJoint.Position));
                rightHandCursorCreated = true;
            }
            //S'il existe, on le met simplement à jour.
            else
            {
                server.UpdateTuioCursor(1, SkeletonPointToPoint3D(e.handJoint.Position));
            }
        }

        /*
        * Méthode appelée lors d'une notification de type : main gauche sortie du champ.
        */
        public void LeftHandQuit(object sender, LeftHandQuitEventArgs e)
        {
            //Si le curseur de la main gauche existe, alors on le supprime.
            if (leftHandCursorCreated)
            {
                server.DeleteTuioCursor(0);
                leftHandCursorCreated = false;
            }
        }

        /*
        * Méthode appelée lors d'une notification de type : main droite sortie du champ.
        */
        public void RightHandQuit(object sender, RightHandQuitEventArgs e)
        {
            //Si le curseur de la main droite existe, alors on le supprime.
            if (rightHandCursorCreated)
            {
                server.DeleteTuioCursor(1);
                rightHandCursorCreated = false;
            }
        }

        /*
        * Permet de convertir un point de position de noeud en Point3D.
        */
        private Point3D SkeletonPointToPoint3D(SkeletonPoint p)
        {
            return new Point3D((double)p.X, (double)p.Y, (double)p.Z);
        }

        /*
        * Méthode de callback vérifiant toutes les 25 ms les nouvelles notifications.
        * Il est à noter que si le temps de rafraîchissement des trop rapide, les messages n'ont pas
        * le temps d'être envoyés.
        */
        private void ThreadPoolCallback(Object threadContext)
        {
            while (true)
            {
                //On initialise le message OSC.
                server.InitFrame();
                //On l'envoie au client (au host et au port spécifiés dans le constructeur).
                server.CommitFrame();
                //On attend 25 ms.
                Thread.Sleep(25);
            }
        }
    }
}
