/*
* This file is part of the TraKERS\Front IDILL package.
*
* (c) IRI <http://www.iri.centrepompidou.fr/>
*
* For the full copyright and license information, please view the LICENSE
* file that was distributed with this source code.
*/

/*
 * Projet : TraKERS
 * Module : Front IDILL
 * Fichier : mosaic.js
 * 
 * Auteur : alexandre.bastien@iri.centrepompidou.fr
 * 
 * Fonctionnalités : Définit la "classe" mosaïque et définit des fonctions d'intéractions.
 */

/*
 * Classe définissant la mosaïque.
 * Elle contient sa longueur, le nombre d'images total, une liste d'urls pour les vidéos, leurs snapshots principaux et leur position.
 * Contient également les dimensions en px de la mosaïque.
 * Est appelé dans :
 * la page d'index, afin de créer la mosaique.
 */
function Mosaic(config, default_conf)
{
    this.gestures = ["fall", "jump", "circle", "screw", "bend", "arc", "knee-up", "right-angle", "wave", "no-motion", "contact", "up-down", "grand-jete"];

    //Chemin du fichier de configuration.
    this.config_path = config;
    //Configuration par défaut en cas de valeur erronnée.
    this.default_config = default_conf;
    this.config = new Object();
    //Tableaux des urls des vidéos, des snapshots et de leur position dans la mosaïque.
    this.videos = [];
    this.urls = [];
    this.sources = [];
    this.imgs = [];
    this.opacities = [];
    this.timeToGoAt = [];
    this.ids = [];
    this.fillingIds = [];
    this.currentRandomVideoIdx = 0;
    
    //Dictionnaire pour les courbes de recherche.
    this.dictionary = [];
	
	//Strings des notifications.
	this.notificationStrings = {
	"select":"Select", 
	"confirm":"Confirm", 
	"timeline":"Timeline move", 
	"move":"Move", 
	"search":"Search", 
	"next":"Next", 
	"previous":"Previous", 
	"unzoom":"Unzoom", 
	"unknown":"Unknown gesture", 
	"fall":"Fall", 
	"jump":"Jump", 
	"spin":"Spin", 
	"screw":"Screw", 
	"bend":"Bend", 
	"arc":"Arc", 
	"knee_up":"Knee up", 
	"breakdance":"Breakdance", 
	"wave":"Wave", 
	"no_motion":"No motion", 
	"contact":"Contact"};
	
	this.helpText = {
	"search_title":"Search", 
	"search_2hands_text":"Curves to be drawn with both hands",  
	"search_mouse_text":"Curves to be mouse-drawn",  
	"search_body_text":"Gestures to be performed with your body", 
	"controls_title":"Controls", 
	"controls_1hand_text":"Gestures to be performed with your hand", 
	"controls_timeline":"Move in timeline", 
	"controls_mos_horizontal":"Unzoom", 
	"controls_mos_vertical":"Unzoom", 
	"controls_move_down":"Move", 
	"controls_move_left":"Move", 
	"controls_move_right":"Move", 
	"controls_move_up":"Move", 
	"controls_next":"Next", 
	"controls_previous":"Previous", 
	"controls_selection":"Selection"
	}
    
    //Dernières positions des pointeurs.
    this.mainPointerLastX = null;
    this.mainPointerLastY = null;
    this.secondPointerLastX = null;
    this.secondPointerLastY = null;
    //Dernières positions avant le lancement de la fonction d'idle.
    this.mainPointerIdleStartX = null;
    this.mainPointerIdleStartY = null;
    this.secondPointerIdleStartX = null;
    this.secondPointerIdleStartY = null;
    
    //Coordonnées de la souris dans le mode d'interaction souris.
    this.mousePosX = null;
    this.mousePosY = null;
    //Coordonnées précédentes de la souris dans le mode d'interaction souris.
    this.mousePosLastX = null;
    this.mousePosLastY = null;
    //Valeur du déplacement entre un mouse up et un mouse down.
    this.mouseUpDownDelta = 0;
    //Valeur du déplacement entre un touch start et un touch end.
    this.touchUpDownDelta = 0;
	//Coordonnées de la souris au dernier mouse down.
	this.mouseDownPosX = null;
	this.mouseDownPosY = null;
    
    //Dimensions de la mosaïque en pixels.
    this.width = null;
    this.height = null;
    //Dimensions d'un snapshot en pixels.
    this.snapshotWidth = null;
    this.snapshotHeight = null;
    //Espacement entre les snapshots en pixels.
    this.marginWidth = null;
    
    //Booléens permettant ou non certaines intéractions selon le contexte.
    this.zoomed = false;
    this.fullscreen = false;
    this.canMoveToNeighbour = false;
    this.mainPointerExitBorder = false;
    this.secondPointerExitBorder = false;
    this.isMainPointerDisplayed = false;
    this.isSecondPointerDisplayed = false;
    this.helpDisplayed = false;
    //Indique si l'utilisateur a manuellement pausé la vidéo.
    this.userPaused = false;
    //Indique si on est en train de se déplacer vers un voisin.
    this.currentlyMoving = false;
    //Indique si on est en train de dézoomer vers la mosaïque.
    this.currentlyUnzooming = false;
    //Indique si on peut s'approcher de kinect.
    this.canStart = false;
    //Indique si on est actuellement sur un snapshot.
    this.isOnASnapshot = false;
    //Indique si l'idle des pointeurs est disponible (deux mains détectées).
    this.pointersIdleAvailable = false;
    //Indique si le timeout pour l'idle des pointeurs à besoin d'être lancé.
    this.pointersIdleNeedLaunch = false;
    //Indique si les deux mains sont là.
    this.areBothPointersHere = false;
    //Indique si le timeout pour la détection de deux pointeurs a été lancé.
    this.areBothPointersTimeoutLaunched = false;
    //Indique si la mosaïque a été filtrée.
    this.isMosaicFiltered = false;
    //Indique si on est actuellement dans une recherche par gesture.
    this.isCurrentlyInASearchByGesture = false;
    //Indique si un pointeur est déjà sur une notification de recherche par gesture.
    this.alreadyOnNotification = false;
    //Si on a fait un swipe.
    this.isSwipe = false;
    //On peut swiper.
    this.canSwipe = false;
    //On passe vers une autre video automatiquement à la fin d'une lecture.
    this.autoMove = false;
    //Si l'utilisateur a demandé à sélectionner la TL.
    this.isTLRequested = false;
    //Le pointeur gauche a sélectionné la TL.
    this.isTLSelectedBySecondPointer = false;
    //Le pointeur droit a sélectionné la TL.
    this.isTLSelectedByMainPointer = false;
    //On peut afficher l'aide.
    this.canNotifyHelp = false;
    //Indique si la mosaique est en train d'être filtrée.
    this.isMosaicFiltering = false;
    this.arrowLeftLoading = false;
    this.arrowRightLoading = false;
    this.arrowUpLoading = false;
    this.arrowDownLoading = false;
    //On est dans une recherche par courbes.
    this.isSearchByCurvesOn = false;
    this.canDrawNextCurve = false;
    //Dans le mode d'interaction souris, indique si on se situe actuellement sur un snapshot entièrement prézoomé.
    this.isOnAPrezoomSN = false;
    //Indique si une courbe de recherche donne au moins un résultat.
    this.curvesGesturesFound = false;
    //Indique si on souhaite supprimer la recherche en cours.
    this.gestureDelRequested = false;
    //Code de gesture actuellement calculé par les détecteurs de courbes.
    this.actualCode = '';
    //Indique si l'utilisateur est entré dans la zone de recherche.
    this.isUserInSearchZone = false;
	//Indique si on a fait un mouse down.
	this.isMouseDown = false;
	//Indique si on a fait un touch start.
	this.isTouchStart = false;
	//Indique si on a fait un touch move.
	this.isTouchMove = false;
	//Indique si on est en train de prézoomer.
	this.isPrezooming = false;
	//Indique si l'icone d'aide a été agrandie.
	this.isHelpIconZoomed = false;
	//Indique si l'icone d'aide est dans une interaction de zoom/dezoom en cours.
	this.isHelpIconZooming = false;
	//Indique à l'utilisateur s'il doit retirer ses mains pour refaire une recherche par courbes.
	this.mustTakeOutHands = false;
	
	//Indique si on est sur une tablette.
	this.isTablet = ('ontouchstart' in document.documentElement);
    
    //Timeout (attente) pour le zoom après un préZoom.
    this.zoomTimeout = null;
    //Timeout (attente) pour le passage vers un voisin.
    this.moveToNeighbourTimeout = null;
    this.mainPointerExitBorderTimeout = null;
    this.secondPointerExitBorderTimeout = null;
    //Idle time pour les pointeurs afin d'informer le front qu'on souhaite faire une recherche.
    this.pointersSearchIdleTimeout = null;
    //Vérifie toutes les N ms que les deux pointeurs sont détectés.
    this.areBothPointersHereTimeout = null;
    //Délai de suppression d'une notification de recherche par gesture.
    this.removeNotificationByGestureTimeout = null;
    //Délai de suppression d'une notification de recherche par gesture infructueuse.
    this.removeFailedNotificationByGestureTimeout = null;
    //Délai avant la suppression de notification swipe.
    this.notifySwipeTimeout = null;
    //Délai pour la sélection de la TL.
    this.selectTLTimeout = null;
    //Délai pour slider sur la TL.
    this.canSlideInTLTimeout = null;
    //Délai pour afficher l'aide.
    this.canNotifyHelpTimeout = null;
    this.arrowLeftTimeout = null;
    this.arrowRightTimeout = null;
    this.arrowUpTimeout = null;
    this.arrowDownTimeout = null;
    
    this.arrowSpinnerTimeout = null;
    this.nouserTimeout = null;
    this.nextDrawCurveTimeout = null;
    
    //Dernier message INCOMING (pour éviter d'effectuer n fois la même action.
    this.lastIncomingMessage = '';
    
    //Type de marqueur recherché dans la mosaïque (en mode filter).
    this.filterSearchedType = "";
    
    //Mode actuel.
    this.currentMode = "NO-USER";
    //Snapshot sur lequel on a zoomé.
    this.previousZoomedSN = null;
    //Snapshot sur lequel on a prezoomé.
    this.previousPrezoomDiv = null;
    //Son ID.
    this.previousId = null;
    //Dernier snapshot prézoomé non null.
    this.lastNonNullSN = null;
    //Largeur de la marge pour le centrage vertical de la mosaïque.
    this.MPTop_margin = null;
    this.top_margin = null;
    
    //Gestures actuellement cherchées dans les vidéos.
    this.currentSearchGesture = [];
    
    //Position des voisins lors d'un zoom.
    this.neighboursIds = [];
    //ID du snapshot du milieu lors d'un zoom.
    this.centerId = null;
    
    //Voisins sélectionnés par les pointeurs.
    this.mainPointerNeighbourSelectedId = null;
    this.secondPointerNeighbourSelectedId = null;
    
    //Snapshots a afficher.
    this.snapshotsToShow = 1;
    
    //Lecteur.
    this.player = null;
    //Si le lecteur est prêt.
    this.playerIsReady = false;
    
    //Annotations (pour les marqueurs los d'un filtrage).
    this.annotations = [];
    
    //Client websocket pour recevoir les notifications du Middleware.
    this.client = null;
    
    //Coordonnées et dimensions d'un snapshot zoomé.
    this.snTop = 0;
    this.snLeft = 0;
    this.snWidth = 0;
    this.snHeight = 0;
    
    this.searchCanvas = null;
    //Position actuelle de la vidéo zoomée.
    this.notifyTopVideo = null;
    this.notifyLeftVideo = null;
    this.loadParameters(this.config_path);
}

/*
 * Méthode d'affichage de la mosaïque.
 * Génère une matrice de imgs.
 * Est appelé dans le fichier :
 * mosaic > fonction loadMosaic, afin de créer les éléments visuels de la mosaique.
 */
Mosaic.prototype.createMosaic = function()
{
	var _this = this;
	
    //On ajoute le player au body pour le lancer la première fois qu'on zoom.
	$('body').append('<div class="player" id="video"></div><div class="LdtPlayer" id="LdtPlayer"></div>');
	
	//On recharge la fenêtre si sa taille a changé.
	$(window).resize(function()
	{
		// _.debounce(window.location.reload(), _this.config.timeReloadAfterResize);
	});
	
    var initPanel = '<div id="initPanel"></div>';
    var mp = $('#mainPanel');
    mp.append(initPanel);
    $('#initPanel').css(
    {
        background: 'transparent',
        width: mp.width(),
        height: mp.height(),
        top: mp.position().top,
        left: mp.position().left,
        'margin-top': this.MPTop_margin
    });
    
    var len = this.config.imagesByLine, imgs = this.config.imagesToShow, imgsTotal = this.config.imagesTotal;
    
    //S'il s'agit d'un rectangle.
    if(imgs % len == 0)
    {
        this.lastIncomingMessage = 'INCOMING-0';
        
        var str = '';
        
        if(this.imgs.length >= imgs)
        {
            for(var i = 0 ; i < imgs ; i++)
            {
                //On charge les images de petite taille pour ne pas surcharger la mosaïque lors de l'affichage global.
                str += '<div id="snapshotDiv-' + i + '" class="snapshotDivs" style="opacity: 0;"><img id="snapshot-' + i + '" class="snapshots" src="snapshots-little/' + this.imgs[i] + '" /></div>';
                //Au départ aucune vidéo n'a de gesture de recherche.
                this.currentSearchGesture[i] = '';
            }
        }
        
        return str;
    }
    else
    {
        alert("Le nombre d'images a afficher doit être divisible par la longueur de la mosaïque !");
    }
}

/*
 * Permet de raffraichir la mosaïque.
 * Est appelé dans le fichier :
 * mosaic > fonction loadFromJson, afin de charger la mosaique une fois que les fichiers de métadonnées ont été lus.
 */
Mosaic.prototype.loadMosaic = function()
{
    // console.log('LOAD');
    var createMosaic = this.createMosaic();
    
    if(createMosaic == '')
    {
        return;
    }
    
    var _this = this;
    
    //On affecte les chemins vers les images à la mosaïque.
    this.previousZoomedSN;
    //On met à jour la mosaïque.
    $('#mainPanel').html(createMosaic);
	//On ajoute le panneau de protection (principalement pour le mode tablettes).
	$('body').append('<div id="ghostPanel"></div>');
    //On récupère la taille des bordures.
    this.marginWidth = $('.snapshotDivs').css('margin-bottom');
    this.marginWidth = parseFloat(this.marginWidth)*2;
    //On calcule la taille des divs contenant les snapshots.
    this.width = $('#mainPanel').innerWidth();
    //On ne calculera pas tout de suite la hauteur de la mosaique étant donnée qu'elle est calculée par la suite dynamiquement.
    this.snapshotWidth = this.width / this.config.imagesByLine - this.marginWidth;
    this.snapshotHeight = this.snapshotWidth*9/16;
    $('.snapshotDivs').css('width', this.snapshotWidth).css('height', this.snapshotHeight).css('margin-left', this.marginWidth/2);
    
    this.height = $('#mainPanel').innerHeight();
    //On centre verticalement la mosaïque.
    this.MPTop_margin = ($(document).height() - $('#mainPanel').height())/2;
    $('#mainPanel').css('margin-top', this.MPTop_margin).css('margin-bottom', this.MPTop_margin);
    
    //On fait coincider le background du body avec celui de la mosaïque.
    $('body').css('background-position', '0px ' + this.MPTop_margin + 'px');
    
    this.addPointers();
    
    //Si dans le metadata player _ n'est pas déclaré, on le déclare.
    if(typeof _ !== "undefined" && typeof IriSP._ === "undefined")
    {
        IriSP._ = _;
    }
    
    if(typeof $ !== "undefined" && typeof IriSP.jQuery === "undefined")
    {
        IriSP.jQuery = $;
    }
    
    //On charge les marqueurs.
    var sourceManager = new IriSP.Model.Directory(),
    globalAnnotations = new IriSP.Model.List(sourceManager),
    nbFichiers = _this.urls.length,
    fichiersCharges = 0;
    
    //Récupère les annotations.
    for (var i = 0; i < nbFichiers; i++)
    {
        _this.sources[i] = sourceManager.remoteSource({url: _this.urls[i], serializer: IriSP.serializers.ldt});
        _this.sources[i].onLoad(function()
        {
            var source = this;
            globalAnnotations.addElements(source.getAnnotations());
            fichiersCharges++;
            if (fichiersCharges == nbFichiers)
            {
                //Instructions à exécuter quand tout est chargé.
                _this.annotations = globalAnnotations;
                // console.log(_this.annotations.length + ' annotations loaded from ' + nbFichiers + ' files.');
                
                //Si on gère les interactions à la souris.
                if(_this.config.mouseInteractions)
                {
					//On bind les événements utiles.
					_this.eventBinder();
                }
                
                //Si on a activé la préphase mais qu'on est en mode Kinect.
                if(_this.config.noUserModeEnabled && !_this.config.mouseInteractions)
                {
                    //On active la préphase.
                    _this.init();
                    _this.showNImages(0);
                    _this.currentMode = "NO-USER";
                }
                else
                {
                    //Sinon on affiche directement la mosaique.
                    _this.showNImages(20);
                    _this.currentMode = "MOSAIC";
                }
                
                //Le premier snapshot sur lequel on zoom en préphase.
                _this.previousZoomedSN = $('#snapshotDiv-' + _this.fillingIds[0]);
            }
        });
    }
}

/*
 * Bind les événements souris ou touch dans le mode d'interaction souris.
*/
Mosaic.prototype.eventBinder = function()
{
	var _this = this;
	
	//On affiche l'icone d'aide.
	this.helpIcon();
	
	//Si on est sur tablette, on utilise l'événement touch start.
	if(this.isTablet)
	{
		document.addEventListener('touchstart', function(e)
		{
			if(e.touches.length == 1)
			{
				var touch = e.touches[0];
				_this.onTouchStart(touch);
			}
			e.preventDefault();
		}, false);
	}
	//Sinon souris.
	else
	{
		//Si on fait un mouse down sur le body, on vérifie enregistre le déplacement de la souris jusqu'au prochain mouse up.
		$(window).mousedown(function (e)
		{
			_this.onMouseDown(e);
			return false;
		});
	}
	
	//Si on est sur tablette, on utilise l'événement touch start.
	if(this.isTablet)
	{
		document.addEventListener('touchmove', function(e)
		{
			if(e.touches.length == 1)
			{
				var touch = e.touches[0];
				_this.refreshPointers(touch.pageX, touch.pageY, true);
				_this.mousePosX = touch.pageX;
				_this.mousePosY = touch.pageY;
				_this.onTouchMove(touch);
			}
			e.preventDefault();
		}, false);
	}
	//Sinon souris.
	else
	{
		//On écoute le déplacement de la souris.
		$(window).mousemove(function(e)
		{
			_this.refreshPointers(e.pageX, e.pageY, true);
			_this.mousePosX = e.pageX;
			_this.mousePosY = e.pageY;
			_this.onMouseMove(e);
			return false;
		});
	}
	
	//Si on est sur tablette, on utilise l'événement touch start.
	if(this.isTablet)
	{
		document.addEventListener('touchend', function(e)
		{
			_this.onTouchEnd(e);
			e.preventDefault();
		}, false);
	}
	//Sinon souris.
	else
	{
		//Si on fait un mouse up après ce mouse down.
		$(window).mouseup(function()
		{
			_this.onMouseUp();
			return false;
		});
	}
	
	if(!this.isTablet)
	{
		//Si on fait un clic.
		$(window).click(function(e)
		{
			_this.onClick(e.pageX, e.pageY);
			_this.removeSearchNotificationIfOnIt(e.pageX, e.pageY);
			return false;
		});
		
		$(window).on('mousewheel', function(event, delta, deltaX, deltaY)
		{
			//Si on est dans un mode autre qu'on mode zoomé ou que l'aide est affichée, on part.
			if(_this.currentMode != "VIDEO" && _this.currentMode != "SEARCH" && _this.currentMode != "TIMELINE" || _this.helpDisplayed)
			{
				return;
			}
			
			//Quand on "tire" la molette vers soi, on dezoom.
			if (delta < 0)
			{
				_this.unzoom();
			}
		});
	}
}

/*
 * Charge les paramètres du Front. Local (true/false) est le mode de chargement des données.
 * Est appelé dans le fichier :
 * mosaic > fonction Mosaic, est appelée dans le constructeur de manière à charger les paramètres de configuration avant la création de la mosaique.
*/
Mosaic.prototype.loadParameters = function(file_path)
{
    var _this = this;
    
    //Variables censées être des ints.
    var supposedToBeInt = ['imagesByLine', 'imagesToShow', 'totalImages', 'timeReloadAfterResize', 'timePrezoom', 'timePreUnzoom', 'zoomTime', 'timeUnzoom', 'timeNeighbourGlowing', 'timeNeighbourUnglowing', 'timeMovingToNeighbour', 'timeSearchFade', 'timeNotifyFade', 'timeFilterFade', 'timeANFade', 'timeFilling', 'zoomedMargin', 'timeoutZoom', 'timeoutUnzoom', 'timeoutMoveToNeighbour', 'timeoutPointersIdle', 'timeoutAreBothPointersHere', 'timeoutRemoveNotificationByGesture', 'timeoutNotifySwipe', 'timeoutSelectTL', 'timeoutSlideTL', 'timeoutCanNotifyHelp', 'timeoutRemoveSpinner', 'timeoutNouser', 'timeoutNextDrawCurve', 'touchUpDownDeltaTreshold', 'mouseUpDownDeltaTreshold', 'kinectUpDownDeltaTreshold', 'helpBorderSize'];
    //Variables censées êtres des floats.
    var supposedToBeFloat = ['zoomPercentage', 'prezoomPercentage'];
    
    //On lit le fichier de configuration.
    $.getJSON(file_path, function(data)
    {
        //Pour chaque ligne du fichier.
        for(key in data)
        {
            var val = data[key];
            
            //Si la valeur est prise pour un int.
            if(_.include(supposedToBeInt, key))
            {
                //On la parse et s'il y a une erreur, on l'indique dans la console avant d'affecter la valeur par défaut.
                //Sinon on lui affecte la valeur lue.
                var intVal = parseInt(val);
                if(isNaN(intVal))
                {
                    _this.config[key] = _this.default_config[key];
                    console.log("param[" + key + "] : Valeur " + val + " incorrecte (non Int). Valeur par défaut " + _this.default_config[key] + " chargée à la place.");
                }
                else
                {
                    _this.config[key] = intVal;
                }
            }
            //Si la valeur est prise pour un float.
            else if(_.include(supposedToBeFloat, key))
            {
                //On la parse et s'il y a une erreur, on l'indique dans la console avant d'affecter la valeur par défaut.
                //Sinon on lui affecte la valeur lue.
                var floatVal = parseFloat(val);
                if(isNaN(floatVal))
                {
                    _this.config[key] = _this.default_config[key];
                    console.log("param[" + key + "] : Valeur " + val + " incorrecte (non Float). Valeur par défaut " + _this.default_config[key] + " chargée à la place.");
                }
                else
                {
                    _this.config[key] = floatVal;
                }
            }
            else
            {
				//S'il s'agit de la langue, on met par défaut si la valeur est nulle.
				if(key == 'lang' && val == '')
				{
					_this.config[key] = _this.default_config[key];
				}
                //Sinon si c'est une string, on l'affecte.
				else
				{
					_this.config[key] = val;
				}
            }
        }
        
        //On remplit le tableau d'ids.
        for(var i = 0 ; i < _this.config.totalImages ; i++)
            _this.ids.push(i);
        //On les mélange.
        _this.ids = _.shuffle(_this.ids);
        
        //On remplit le tableau d'ids destiné à afficher les snapshots au fur et à mesure.
        for(var i = 0 ; i < _this.config.imagesToShow ; i++)
            _this.fillingIds.push(i);
        //On les mélange.
        _this.fillingIds = _.shuffle(_this.fillingIds);
        
        //Si la config spécifie le chargement comme étant local/en ligne, on charge le fichier des vidéos approprié.
        if(_this.config.local)
        {
            _this.loadFromJson(_this.config.videoConfigFileLocal);
        }
        else
        {
            _this.loadFromJson(_this.config.videoConfigFileOnline);
        }
        
        //On initialise le client dans le cas d'intéractions Kinect.
        if(!_this.config.mouseInteractions)
        {
            _this.client = new Client(_this.config.host, _this.config.port, _this);
        }
        
        //On prend le dictionnaire stockant les codes des courbes de recherche.
        _this.getDictionary();
        //On charge le fichier des langues.
        _this.getLang();
    });
}

/*
 * Phase principale (utilisateur non détecté). Une vidéo se lance aléatoirement.
 * L'interface est identique à celle du zoom, mais sans interaction possible avec les voisins, ni les controles timeline.
 * Lors de la fin d'une lecture, on dézoome vers la mosaïque, puis on rezoome vers un autre snapshot (aléatoire).
 * Est appelé dans les fichiers :
 * mosaic > fonction loadMosaic.
 * zoomInteractions > fonction unzoom.
*/
Mosaic.prototype.init = function()
{
    var _this = this;
    
    //Si l'index de lecture des vidéos en mode sans utilisateur excède le nombre maximal de vidéos, il est remis à 0 pour tourner en boucle.
    if(this.currentRandomVideoIdx > this.config.imagesToShow)
    {
        this.currentRandomVideoIdx = 0;
    }
    
    //Le snapshot sur lequel on doit zoomé a pour id le tableau de lecture aléatoire à l'index de lecture.
    this.previousZoomedSN = $('#snapshotDiv-' + this.fillingIds[this.currentRandomVideoIdx]);
    this.previousId = $('img', this.previousZoomedSN).attr('id');
    
    //On fait apparaître le snapshot avant de zoomer dessus.
    this.previousZoomedSN.fadeTo(this.config.timePrezoom, 1, function()
    {
        _this.zoom();
        _this.currentRandomVideoIdx++;
    });
}

/*
 * Remplissage de la mosaïque en fonction du nombre d'images à afficher.
 * Est appelé dans le fichier :
 * mosaic > fonctions loadMosaic et manageControlEvents.
*/
Mosaic.prototype.showNImages = function(n)
{
    //Si on bouge vers un voisin, on n'utilise pas cette fonction.
    if(this.currentlyMoving)
    {
        return;
    }
    
    //Si il y a plus d'un snapshot à afficher, on entre dans le mode INCOMING avec en paramètre le nombre à afficher.
    if(n > 1 && n < this.config.imagesToShow)
    {
        //On met en mode incoming avec le nombre de snapshots à afficher.
        this.currentMode = "INCOMING-" + n;
        //On dézoom.
        this.unzoom();
        
        //On initialise les gestures de recherche.
        for(var i = 0 ; i < this.config.imagesToShow ; i++)
        {
            this.currentSearchGesture[i] = '';
        }
        
        //On enlève les notifications.
        this.removeNotifications();
        //On annule les recherches.
        this.isMosaicFiltered = false;
        this.isCurrentlyInASearchByGesture = false;
        //On masque les pointeurs.
        $('#mainPointer').fadeTo(this.config.timePrezoom, 0);
        $('#secondPointer').fadeTo(this.config.timePrezoom, 0);
        //On enlève le spinner.
        $('#spinner').remove();
        //On déselectionne tous les voisins.
        this.deselectAllNeighbours();
        //On enlève tous les prézooms en cours.
        $('.prezoomContainers').remove();
    }
    
    //Si on doit afficher la mosaique complète.
    if(n >= this.config.imagesToShow)
    {
        //Si on est en mode sans utilisateur ou bien en mode d'utilisateur approchant.
        if(this.currentMode == "NO-USER" || this.currentMode.indexOf("INCOMING-") > -1)
        {
            //Si on est en intéractions Kinect.
            if(!this.config.mouseInteractions)
            {
                //On met le mode à jour.
                this.currentMode = "INCOMING-20";
                //On dézoom.
                this.unzoom();
            }
            
            //On passe en mode mosaique.
            this.currentMode = "MOSAIC";
            //On enlève les notifications, et on met les notifications de sélection et de recherche.
            this.removeNotifications();
            this.mosaicSelectionAndSearch();
            //On retarde le mécanisme de redémarrage.
            clearTimeout(this.nouserTimeout);
        }
    }
    
    //Pour les snapshots à afficher.
    for(var i = 0 ; i < n ; i++)
    {
        //Si les snapshots ne sont pas affichés.
        if($('#snapshotDiv-' + this.fillingIds[i]).css('opacity') < 1)
        {
            //On les fait apparaître.
            $('#snapshotDiv-' + this.fillingIds[i]).fadeTo(this.config.timeFilling, '1');
        }
    }
    //Pour ceux à masquer.
    for(var i = n ; i < this.config.imagesToShow ; i++)
    {
        //Si les snapshots ne sont pas masqués et qu'il ne s'agit pas du dernier snapshot en lecture aléatoire (mode NO-USER).
        if($('#snapshotDiv-' + this.fillingIds[i]).css('opacity') > 0 && this.fillingIds[i] != this.currentRandomVideoIdx)
        {
            //On les masque.
            $('#snapshotDiv-' + this.fillingIds[i]).fadeTo(this.config.timeFilling, '0');
        }
    }
}

/*
 * Gère les événements de contrôle dans la mosaïque.
 * Est appelé dans le fichier :
 * client > fonction processMsg.
*/
Mosaic.prototype.manageControlEvents = function(event)
{
    var _this = this;
    
    if(typeof event === 'undefined')
    {
        return;
    }
    
    var gestureReceived = '';
    
    //Si l'utilisateur arrive et qu'on a la préphase activée.
    if(event.indexOf("INCOMING-") != -1 && this.config.noUserModeEnabled)
    {
        //Si on peut démarrer.
        if(this.canStart)
        {
            //Si on veut afficher plus de snapshots que disponibles, on affiche le maximum.
            if(this.snapshotsToShow > this.config.imagesToShow)
            {
                this.snapshotsToShow = this.config.imagesToShow;
            }
            else
            {
                //On récupère le nombre de snapshots à afficher.
                var params = event.split('-');
                this.snapshotsToShow = params[1];
            }
            
            //Si la position de l'utilisateur a changé.
            if(event != this.lastIncomingMessage)
            {
                //On la met à jour et on affiche la mosaique en conséquence.
                this.lastIncomingMessage = event;
                this.showNImages(this.snapshotsToShow);
            }
        }
        
        //On retarde le mécanisme de redémarrage.
        clearTimeout(this.nouserTimeout);
        
        //Après un délai d'absence de l'utilisateur on redémarre.
        this.nouserTimeout = setTimeout(function()
        {
            // window.location.reload();
        }, this.config.timeoutNouser);
    }
    //S'il n'y a qu'un snapshot à afficher et qu'on est en préphase, on redémarre.
    else if((event == "NO-USER" || event == "INCOMING-0" || event == "INCOMING-1") && this.config.noUserModeEnabled)
    {
        // window.location.reload();
    }
    //Si on a fait un swipe.
    else if(event.indexOf("SWIPE") != -1)
    {
        //Si le player est près et qu'on n'est pas en train de faire un swipe.
        if(this.player && this.player.widgets && this.playerIsReady && !this.isSwipe)
        {
            //On est en train de faire un swipe.
            this.isSwipe = true;
            
            //Si on est en mode de recherche, que la mosaique est filtrée et qu'on est arrivé au dernier marqueur.
            if(this.currentMode == 'SEARCH' && this.isMosaicFiltered && !this.player.widgets[0].isAMarkerAhead(this.currentSearchGesture[this.centerId]))
            {
                //On passe à la prochaine vidéo qui contient des résultats.
                this.playNextVideo();
            }
            
			//On spécifie s'il s'agit d'un swipe left ou right.
			var swipeType = ((event.indexOf("LEFT") != -1) ? 'left' : 'right');
			var isSwipeLeft = ((event.indexOf("LEFT") != -1) ? true : false);
			
			//On passe au marqueur suivant/précédent en fonction du type de swipe.
			this.player.widgets[0].switchToMarker(isSwipeLeft, this.currentSearchGesture[this.centerId]);
			//Si on est en mode vidéo.
			if(this.currentMode == 'VIDEO')
			{
				//On affiche la notification.
				this.removeNotifications();
				this.videoSwipe(swipeType);
			}
			//Si on est en mode recherche dans une vidéo et qu'on n'a pas de gesture de recherche.
			else if(this.currentMode == 'SEARCH' && this.currentSearchGesture[this.centerId] == '')
			{
				//On affiche la notification.
				this.removeNotifications();
				this.searchSearchAndSwipe(swipeType);
			}
			//Si on est en mode recherche dans une vidéo et qu'on a une gesture de recherche.
			else if(this.currentMode == 'SEARCH' && this.currentSearchGesture[this.centerId] != '')
			{
				//On affiche la notification.
				this.removeNotifications();
				this.searchGestureAndSwipe(this.currentSearchGesture[this.centerId], 'valid', swipeType);
			}
            
            //On le fait disparaitre au bout d'un certain temps.
            this.notifySwipeTimeout = setTimeout(function()
            {
                _this.isSwipe = false;
                _this.removeNotifications();
                
                //Si on est en mode de recherche dans une vidéo et qu'aucune gesture n'est recherchée.
                if(_this.currentMode == 'SEARCH' && _this.currentSearchGesture[_this.centerId] == '')
                {
                    _this.searchSearch();
                }
                //Si on est en mode de recherche dans une vidéo et qu'on a une recherche par gesture.
                else if(_this.currentMode == 'SEARCH' && _this.currentSearchGesture[_this.centerId] != '')
                {
                    _this.searchGesture(_this.currentSearchGesture[_this.centerId], 'valid');
                }
                
            }, this.config.timeoutNotifySwipe);
        }
    }
    //Si on reçoit l'instruction correspondant à une des quatre gestures de recherche corporelle.
    else if(event.indexOf("BEND") != -1 || event.indexOf('KNEE-UP') != -1 || event.indexOf('FALL') != -1 || event.indexOf('JUMP') != -1)
    {
        //On la met en minuscule.
        gestureReceived = event.toLowerCase();
        //On met à jour la gesture recherchée.
        this.currentSearchGesture[this.centerId] = gestureReceived;
    }
    //Si on a effectué un hello, qu'on peut notifier l'aide et que les deux mains ne sont pas dans la zone de recherche.
    else if(event.indexOf("HELLO") != -1 && this.canNotifyHelp && !this.areBothPointersHere)
    {
        //On affiche différentes aides en fonction de si on se trouve dans une vidéo ou non.
        if(this.currentMode == 'SEARCH')
        {
            this.notifyHelp(false);
        }
        else if(this.currentMode == 'FILTER')
        {
            this.notifyHelp(true);
        }
    }
    
    //Si on a reçu une gesture.
    if(gestureReceived != '')
    {
        //Si on est en mode recherche et que le player est prêt.
        if(this.currentMode == "SEARCH" && this.playerIsReady)
        {
            //On recherche dans la vidéo.
            this.player.widgets[0].searchByGesture(gestureReceived);
            this.isCurrentlyInASearchByGesture = this.player.widgets[0].isCurrentlyInASearchByGesture;
            //On notifie.
            this.removeNotifications();
            this.searchGesture(gestureReceived, 'valid');
        }
        //Si on est dans un filtrage.
        else if(this.currentMode == "FILTER")
        {
            if(this.isMosaicFiltered)
            {
                //On notifie et on filtre.
                this.removeNotifications();
                this.filterGesture(gestureReceived, 'valid');
                this.searchFilter(gestureReceived);
            }
        }
        
        //Si l'aide est affichée, on l'enlève.
        if(this.helpDisplayed)
        {
            this.removeHelp();
        }
    }
}

/*
 * Fonction qui s'exécute lorsque le player a chargé.
 * Est appelé dans le fichier :
 * mosaic > fonction loadPlayer.
*/
Mosaic.prototype.onPlayerLoad = function()
{
	var _this = this;
	
	//Si on est sur une tablette, on affiche les controls.
	if(this.isTablet)
	{
		$('video').attr('controls', 'controls');
	}
	
	if(this.currentMode == 'NO-USER')
	{
		//On peut s'approcher de la kinect.
		this.canStart = true;
	}
	
	//Lorsque le player est en pause (par exemple lorsque le curseur arrive à la fin de la timeline).
	if(this.player.popcorn)
	{
		this.player.popcorn.listen('pause', function()
		{
			//Si la pause est naturelle (fin de la timeline, dézoom, déplacement vers un voisin).
			if(!_this.userPaused)
			{
				//Si c'est en mode sans utilisateur.
				if(_this.currentMode == 'NO-USER')
				{
					//On dézoome.
					_this.unzoom();
				}
				//Sinon, si ce n'est pas causé par un déplacement ou un dézoom.
				else if(!_this.currentlyMoving && !_this.currentlyUnzooming)
				{
					//Si on est en mode timeline et qu'on est en pause, c'est probablement que l'user a placé le curseur à la fin.
					if(_this.currentMode != 'TIMELINE')
					{
						_this.playNextVideo();
					}
				}
			}
		});
		
		//Si les marqueurs ont été chargés.
		this.player.popcorn.on("markersready", function()
		{
			_this.onMarkersReady();
		});
	}
}

/*
 * Fonction qui s'exécute lorsque les marqueurs d'une video on été placés.
 * Est appelé dans le fichier :
 * mosaic > fonction onPlayerLoad.
*/
Mosaic.prototype.onMarkersReady = function()
{
	var _this = this;
	
	//Le player est prêt.
	this.playerIsReady = true;
	
	if(this.player.widgets[0])
	{
		//On spécifie à la timeline dans quel mode d'intéraction on est.
		this.player.widgets[0].setMouseInteractions(this.config.mouseInteractions);
		//Idem pour la langue.
		if(this.gesturesText.length > 0)
		{
			this.player.widgets[0].setLang(this.gesturesText);
		}
	}
	
	//Si on est en mode video ou recherche ou timeline.
	if(this.currentMode == 'VIDEO' || this.currentMode == 'SEARCH' || this.currentMode == 'TIMELINE')
	{
		//On peut faire des swipes.
		this.canSwipe = true;
	}
	
	//Si aucune recherche par gesture n'est effectuée.
	if(this.currentSearchGesture[this.centerId] == '')
	{
		//On enlève le filtre.
		this.removeFilter();
	}
	//Sinon.
	else
	{
		//On entre en mode recherche dans une vidéo.
		this.currentMode = 'SEARCH';
		//On recherche la gesture.
		this.player.widgets[0].searchByGesture(this.currentSearchGesture[this.centerId]);
		this.isCurrentlyInASearchByGesture = this.player.widgets[0].isCurrentlyInASearchByGesture;
		
		//On va au premier marqueur trouvé.
		if(this.timeToGoAt[this.centerId] === 0 && this.player.widgets[0].atLeastOneSearchMarker(this.currentSearchGesture[this.centerId]))
		{
			this.player.widgets[0].goToFirstSearchedMarker(this.currentSearchGesture[this.centerId]);
		}
		//Si aucun marqueur n'est trouvé.
		else
		{
			//On va juste là où on était la dernière fois qu'on a joué la vidéo.
			this.player.popcorn.currentTime(this.timeToGoAt[this.centerId]);
		}
	}
}

/*
 * Chargement du player basé sur le metadataplayer.
 * Est appelé dans les fichiers :
 * neighbours > fonction moveToNeighbour.
 * zoomInteractions > zoom.
*/
Mosaic.prototype.loadPlayer = function(newZoomTop, newZoomLeft, newSnWidth, newSnHeight, zoomTop, zoomLeft, timeToGo)
{
    var _this = this;
    
    //On configure les options de lancement.
    IriSP.libFiles.defaultDir = "../lib/";
    IriSP.widgetsDir = "./player/metadataplayer/"
    
    var videoToPlay = this.videos[this.centerId];
    var currentMetadata = this.urls[this.centerId];
    
    var _metadata = {
        url: currentMetadata,
        format: 'ldt'
    };
    
    var _config = {
        gui: {
            zoomTop: zoomTop - this.marginWidth*2,
            zoomLeft: zoomLeft,
            width: newSnWidth,
            height: newSnHeight,
            container: 'LdtPlayer',
            default_options: {
                metadata: _metadata
            },
            css:'./player/metadataplayer/LdtPlayer-core.css',
            widgets: [
                {
                    type: "Timeline"
                }
            ]
        },
        player:{
            type: 'html5', // player type
            // type: 'jwplayer', // player type
            video: videoToPlay,
            live: true,
            height: newSnHeight,
            width: newSnWidth,
            autostart: true
        }
    };
    
    //On positionne le player.
    $('.LdtPlayer').css(
    {
        position: 'absolute',
        'background-color': '#000000',
        top: newZoomTop,
        left: newZoomLeft,
		'z-index': 500
    });
    
    //On démarre le player.
    this.player = null;
    
    this.player = new IriSP.Metadataplayer(_config, _metadata);
    
    this.player.onLoad(function()
    {
        _this.onPlayerLoad();
    });
}

/*
 * Charge les vidéos, les snapshots et les annotations depuis un fichier json.
 * Est appelé dans le fichier :
 * mosaic > fonction loadParameters.
*/
Mosaic.prototype.loadFromJson = function(path)
{
    var _this = this;
    var i = 0;
    
    //On ouvre le fichier contenant les vidéos et les adresses des métadonnées.
    $.getJSON(path, function(data)
    {
        $.each(data, function(key, val)
        {
            $.each(val, function(key_video, val_video)
            {
                //On extrait les métadonnées.
                $.getJSON(val_video.metadata, function(meta)
                {
                    //Si on est en chargement local.
                    if(_this.config.local)
                    {
                        //On cherche les vidéos dans les métadonnées.
                        _this.affectVideoById(val_video.metadata, meta.medias[0].url.replace('rtmp://media.iri.centrepompidou.fr/ddc_player/', './player/videos/').replace('mp4:', '').replace('video/', '').replace('ldtplatform/', '').replace('.m4v', '.mp4'));
                    }
                    //Si on est en chargement en ligne.
                    else
                    {
                        //On met l'adresse des videos en ligne.
                        _this.affectVideoById(val_video.metadata, meta.medias[0].url.replace('rtmp://', 'http://').replace('/ddc_player/', '/').replace('mp4:', '').replace('.m4v', '.mp4'));
                    }
                });
                
                //On affecte les images et métadonnées.
                _this.imgs[_this.ids[i]] = val_video.snapshot;
                _this.urls[_this.ids[i]] = val_video.metadata;
                //Au départ, on commence à 0 ms dans les vidéos.
                _this.timeToGoAt[_this.ids[i]] = 0;
                i++;
            });
        });
        
        //On charge la mosaique.
        _this.loadMosaic();
    });
}

/*
 * Affecte une vidéo au tableau des vidéos selon son id.
 * Est appelé dans le fichier :
 * mosaic > fonction loadFromJson.
*/
Mosaic.prototype.affectVideoById = function(metadata_id, video)
{
    for (i = 0 ; i < this.urls.length ; i++)
    {
        if(this.urls[i] == metadata_id)
        {
            this.videos[i] = video;
            break;
        }
    }
}

/*
 * Affiche la date actuelle pour l'affichage de messages dans la console.
 * Est appelé dans chaque console.log() où on veut afficher la date à laquelle l'instruction est exécutée.
*/
Mosaic.prototype.date = function()
{
   var date, h, min, s;
   date = new Date();
   h = date.getHours();
   min = date.getMinutes();
   s = date.getSeconds();
   if (h < 10)
      h = "0" + h;
   if (min < 10)
      min = "0" + min;
   if (s < 10)
      s = "0" + s;
   return (h + ":" + min + ":" + s);
};