﻿/*
* This file is part of the TraKERS\Middleware package.
*
* (c) IRI <http://www.iri.centrepompidou.fr/>
*
* For the full copyright and license information, please view the LICENSE_MIDDLEWARE
* file that was distributed with this source code.
*/

/*
 * Projet : TraKERS
 * Module : MIDDLEWARE
 * Sous-Module : Tracking/Gestures
 * Classe : CircleDetector
 * 
 * Auteur : alexandre.bastien@iri.centrepompidou.fr
 * 
 * Fonctionnalités : Permet de détecter si l'utilisateur a effectué un cercle, en se basant sur
 * des règles appliquées à la positions des noeuds dans le temps.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Microsoft.Kinect;
using System.Windows.Media.Media3D;
using System.Drawing;

namespace Trakers.MainModule.Gestures
{
    public class CircleDetector : GestureDetector
    {
        public CircleDetector() : base()
        {
            gesturePeriod = (float)1;
            indexesPerSecond = 30;
            indexesToCheck = (int)(gesturePeriod * indexesPerSecond);
        }

        /*
         * Lit les noeuds de l'historique du squelette afin de détecter un cercle.
         * Règles :
         * Se fait avec une main.
         * Chaque point est à la même distance du barycentre.
         * Traitement :
         * On .
         */
        public bool CheckForCircle()
        {
            //Crée un historique de squelette local, puisque l'historique est mis à jour toutes les ~1/30 s.
            List<List<Joint>> localHistory = new List<List<Joint>>(history);

            //Si il n'y a pas assez de positions dans l'historique local pour vérifier le geste.
            if (localHistory.Count < indexesToCheck + 1)
                return false;

            //La distance de référence est ici la distance entre le milieu du dos et le milieu des épaules.
            refDistance = Math.Abs(localHistory[0][(int)JointType.Spine].Position.Y - localHistory[0][(int)JointType.ShoulderCenter].Position.Y);
            //On commence la position pour les indexesToCheck dernières postures (celle à l'index 0 étant la dernière).
            startPoint = localHistory[localHistory.Count - indexesToCheck][(int)JointType.HandRight].Position;

            //Barycentres pour les mains.
            PointF leftBarycenter = new PointF(0, 0);
            PointF rightBarycenter = new PointF(0, 0);
            //Distances moyennes des points aux barycentres.
            float averageDistToLeftBarycenter = 0;
            float averageDistToRightBarycenter = 0;

            //Index du point de départ dans la détection.
            int beginIndex = localHistory.Count - indexesToCheck;

            //Calcul du barycentre de la main gauche.
            for (int i = beginIndex; i > 0; i--)
            {
                leftBarycenter.X += localHistory[i][(int)JointType.HandLeft].Position.X;
                leftBarycenter.Y += localHistory[i][(int)JointType.HandLeft].Position.Y;
            }
            leftBarycenter.X /= indexesToCheck;
            leftBarycenter.Y /= indexesToCheck;

            //Calcul du barycentre de la main droite.
            for (int i = beginIndex; i > 0; i--)
            {
                rightBarycenter.X += localHistory[i][(int)JointType.HandRight].Position.X;
                rightBarycenter.Y += localHistory[i][(int)JointType.HandRight].Position.Y;
            }
            rightBarycenter.X /= indexesToCheck;
            rightBarycenter.Y /= indexesToCheck;

            //Estimation de la distance moyenne d'un point au barycentre gauche.
            for (int i = beginIndex; i > 0; i--)
            {
                float ptX = localHistory[i][(int)JointType.HandLeft].Position.X;
                float ptY = localHistory[i][(int)JointType.HandLeft].Position.Y;
                averageDistToLeftBarycenter += (float)Distance2D(ptX, leftBarycenter.X, ptY, leftBarycenter.Y);
            }
            averageDistToLeftBarycenter /= indexesToCheck;

            //Estimation de la distance moyenne d'un point au barycentre droit.
            for (int i = beginIndex; i > 0; i--)
            {
                float ptX = localHistory[i][(int)JointType.HandRight].Position.X;
                float ptY = localHistory[i][(int)JointType.HandRight].Position.Y;
                averageDistToRightBarycenter += (float)Distance2D(ptX, rightBarycenter.X, ptY, rightBarycenter.Y);
            }
            averageDistToRightBarycenter /= indexesToCheck;

            //De la position p1 à pn, on suit l'algorithme.
            for (int i = localHistory.Count - indexesToCheck + 1; i < localHistory.Count; i++)
            {
                //Si la position Y de la main est plus haute que la tête
                //OU si la position Y de la main est plus basse que la hanche
                //OU si la nouvelle position X de la main est à droite de la précédente
                //OU si la nouvelle position Y de la main est plus éloignée de la distance N par rapport à la première position Y
                //Alors on retourne faux.
                if (localHistory[i][(int)JointType.HandRight].Position.Y < localHistory[i][(int)JointType.Head].Position.Y ||
                localHistory[i][(int)JointType.HandRight].Position.Y > localHistory[i][(int)JointType.HipCenter].Position.Y ||
                localHistory[i][(int)JointType.HandRight].Position.X > localHistory[i - 1][(int)JointType.HandRight].Position.X ||
                Math.Abs(localHistory[i][(int)JointType.HandRight].Position.Y - startPoint.Y) > refDistance / 2)
                    return false;
            }

            //Si la distance horizontale du geste a été plus courte que la distance N
            //Alors on retourne faux.
            if (Math.Abs(localHistory[0][(int)JointType.HandRight].Position.X - localHistory[localHistory.Count - indexesToCheck][(int)JointType.HandRight].Position.X) < refDistance / 2)
                return false;

            //Si la dernière position de la main droite est sur le côté droit du corps
            //OU si la première position calculée de la main droite est sur le côté gauche du corps
            //Alors on retourne faux.
            if (localHistory[localHistory.Count - 1][(int)JointType.HandRight].Position.X > localHistory[localHistory.Count - 1][(int)JointType.HipCenter].Position.X ||
               localHistory[localHistory.Count - indexesToCheck][(int)JointType.HandRight].Position.X < localHistory[localHistory.Count - 1][(int)JointType.HipCenter].Position.X)
                return false;

            //On supprime l'historique local.
            localHistory.Clear();
            //Si on est arrivé jusqu'ici, toutes les conditions pour un swipe left ont été remplies.
            return true;
        }

        public double Distance2D(float x1, float x2, float y1, float y2)
        {
            return Math.Sqrt((x1 - x2)*(x1 - x2) + (y1 - y2)*(y1 - y2));
        }
    }
}
