/*
* This file is part of the TraKERS\Middleware package.
*
* (c) IRI <http://www.iri.centrepompidou.fr/>
*
* For the full copyright and license information, please view the LICENSE
* file that was distributed with this source code.
*/

/*
 * Projet : TraKERS
 * Module : MIDDLEWARE
 * Sous-Module : Tracking/Gestures
 * Classe : JumpDetector
 * 
 * Auteur : alexandre.bastien@iri.centrepompidou.fr
 * 
 * Fonctionnalités : Permet de détecter si l'utilisateur a sauté, en se basant sur
 * des règles appliquées à la positions des noeuds dans le temps.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Microsoft.Kinect;
using Trakers.Debug;

namespace Trakers.Tracking.Gestures
{
    public class JumpDetector : GestureDetector
    {
        static int n = 0;

        public JumpDetector(DebugWindow _debug) : base(_debug)
        {
            gesturePeriod = (float)1.5;
            indexesPerSecond = 30;
            indexesToCheck = (int)(gesturePeriod * indexesPerSecond);
        }

        /*
         * Lit les noeuds de l'historique du squelette afin de détecter un Jump.
         * Règles :
         * .
         */
        public bool CheckForJump()
        {
            //Crée un historique de squelette local, puisque l'historique est mis à jour toutes les ~1/30 s.
            List<List<Joint>> localHistory = new List<List<Joint>>(history);
            
            //Si il n'y a pas assez de positions dans l'historique local pour vérifier le geste.
            if (localHistory.Count < indexesToCheck)
                return false;

            //La distance de référence est ici la distance entre les épaules et les hanches.
            refDistance = Math.Abs(localHistory[0][(int)JointType.ShoulderCenter].Position.Y - localHistory[0][(int)JointType.HipCenter].Position.Y);
            //On commence la position pour les indexesToCheck dernières postures (celle à l'index 0 étant la dernière).

            int beginIdx = localHistory.Count - indexesToCheck + 1;

            int middleIdx = 0;
            int endIdx = 0;
            bool topOfJump = false;
            bool probableJump = false;

            //De la position p1 à pn, on suit l'algorithme.
            for (int i = beginIdx ; i < localHistory.Count ; i++)
            {
                if (localHistory[i][(int)JointType.Spine].Position.Y + refDistance / 2 < localHistory[beginIdx][(int)JointType.Spine].Position.Y)
                {
                    topOfJump = true;
                    middleIdx = i;
                }
            }

            if (topOfJump)
            {
                for (int i = middleIdx; i < localHistory.Count; i++)
                {
                    if (Math.Abs(localHistory[beginIdx][(int)JointType.Spine].Position.Y - localHistory[i][(int)JointType.Spine].Position.Y) < refDistance / 5 ||
                        localHistory[beginIdx][(int)JointType.Spine].Position.Y < localHistory[i][(int)JointType.Spine].Position.Y)
                    {
                        probableJump = true;
                        endIdx = i;
                    }
                }
            }
            else
            {
                //On supprime l'historique local.
                localHistory.Clear();
                return false;
            }

            if (probableJump)
            {
                if (Math.Abs(localHistory[beginIdx][(int)JointType.Spine].Position.Z - localHistory[endIdx][(int)JointType.Spine].Position.Z) < 0.10)
                {
                    //On supprime l'historique local.
                    localHistory.Clear();
                    return true;
                }
            }
            else
            {
                //On supprime l'historique local.
                localHistory.Clear();
                return false;
            }
            
            //On supprime l'historique local.
            localHistory.Clear();
            return false;
        }
    }
}
