﻿/*
* This file is part of the TraKERS\Middleware package.
*
* (c) IRI <http://www.iri.centrepompidou.fr/>
*
* For the full copyright and license information, please view the LICENSE_MIDDLEWARE
* file that was distributed with this source code.
*/

/*
 * Projet : TraKERS
 * Module : MIDDLEWARE
 * Sous-Module : Communication
 * Classe : Server
 * 
 * Auteur : alexandre.bastien@iri.centrepompidou.fr
 * 
 * Fonctionnalités : Reçoit des notifications du module sous-module Tracking.
 * Traduit les notifications sous forme de messages OSC et les envoie au Front Atelier.
 * Forme des messages :
 * - Notification de main dans le champ de recherche : Point3D indiquant la position de la main dans l'espace.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Threading;
using Tuio;
using System.Windows;
using Microsoft.Kinect;
using System.Windows.Media.Media3D;
using System.Timers;
using System.Resources;
using System.Reflection;

namespace Trakers.Communication
{
    public class Server
    {
        //Serveur TUIO, provenant de la DLL TuioServer créé par Bespoke.
        private TuioServer server;

        //Permet de savoir si un curseur pour la main gauche/droite a été créé.
        private bool leftHandCursorCreated;
        private bool rightHandCursorCreated;
        private bool messageCreated;
        private bool gestureLocked, modLocked;
        //Intervalle minimum entre les gestures.
        private int timerElapsing;
        //Timer.
        private System.Timers.Timer _timer;
        //Dernier code envoyé.
        private String lastCode;

        /*
        * Constructeur : On initialise le serveur avec une adresse et un port, au début les curseurs
        * ne sont pas créés et on indique au ThreadPool une fonction de callback de manière à vérifier
        * s'il reçoit des notifications.
        */
        public Server(String host, int port, int _timerElapsing)
        {
            //Au départ, aucune main n'est dans le champ de recherche et aucune gesture n'est détectée.
            leftHandCursorCreated = false;
            rightHandCursorCreated = false;
            messageCreated = false;
            gestureLocked = false;
            modLocked = false;
            lastCode = "";

            timerElapsing = _timerElapsing;

            //On démarre le serveur TUIO.
            server = new TuioServer(host, port);
            //On initialise le threadPool (appelé toutes les N ms).
            ThreadPool.QueueUserWorkItem(ThreadPoolCallback);

            //On instancie le timer à N ms.
            _timer = new System.Timers.Timer(timerElapsing);
            //Dès que le timer est expiré, on appelle _timer_Elapsed.
            _timer.Elapsed += new ElapsedEventHandler(_timer_Elapsed);
        }

        /*
        * Getter du serveur.
        */
        public TuioServer getServer()
        {
            return server;
        }

        /*
         * Méthode appelée à l'expiration du timer pour les gestures et modes.
         */
        public void _timer_Elapsed(object sender, ElapsedEventArgs e)
        {
            //On débloque la détection de gesture.
            gestureLocked = false;
            //On débloque la notification de nouveau mode.
            modLocked = false;
            //On arrête le timer.
            _timer.Stop();
        }

        /*
        * Méthode appelée lors d'une notification de type : main gauche entrée dans le champ.
        */
        public void LeftHandTracked(Point3D pt)
        {
            //Si le curseur de la main gauche n'est pas créé, alors on le crée.
            if (!leftHandCursorCreated)
            {
                server.AddTuioCursor(0, pt);
                leftHandCursorCreated = true;
            }
            //S'il existe, on le met simplement à jour.
            else
                server.UpdateTuioCursor(0, pt);
        }

        /*
        * Méthode appelée lors d'une notification de type : main droite entrée dans le champ.
        */
        public void RightHandTracked(Point3D pt)
        {
            //Si le curseur de la main droite n'est pas créé, alors on le crée.
            if (!rightHandCursorCreated)
            {
                server.AddTuioCursor(1, pt);
                rightHandCursorCreated = true;
            }
            //S'il existe, on le met simplement à jour.
            else
                server.UpdateTuioCursor(1, pt);
        }

        /*
        * Méthode appelée lors d'une notification de type : main gauche sortie du champ.
        */
        public void LeftHandQuit()
        {
            //Si le curseur de la main gauche existe, alors on le supprime.
            if (leftHandCursorCreated)
            {
                server.DeleteTuioCursor(0);
                leftHandCursorCreated = false;
            }
        }

        /*
        * Méthode appelée lors d'une notification de type : main droite sortie du champ.
        */
        public void RightHandQuit()
        {
            //Si le curseur de la main droite existe, alors on le supprime.
            if (rightHandCursorCreated)
            {
                server.DeleteTuioCursor(1);
                rightHandCursorCreated = false;
            }
        }

        /*
        * Méthode appelée lorsqu'une gesture a été détectée et que l'événement approprié a été lancé.
        */
        public void GesturePerformed(String code)
        {
            //Si le code vient d'être envoyé, on passe.
            if (lastCode.Equals(code))
                return;
            lastCode = code;
            //Si une gesture a été effectuée, on bloque un certain temps.
            if (!gestureLocked)
            {
                gestureLocked = true;
                
                //On crée un message contenant le code à envoyer.
                if (!messageCreated)
                {
                    messageCreated = true;
                    server.AddTuioString(2, code);
                    //On démarre le timer.
                    _timer.Start();
                }
            }
        }

        /*
        * Méthode appelée lorsqu'on doit entrer dans un autre mode.
        */
        public void ModeNotification(String code)
        {
            //Si le code vient d'être envoyé, on passe.
            if (lastCode.Equals(code))
                return;
            lastCode = code;
            //Si on a été notifié.
            if (!modLocked)
            {
                modLocked = true;

                //On crée un message contenant le code à envoyer.
                if (!messageCreated)
                {
                    messageCreated = true;
                    server.AddTuioString(2, code);
                    //On démarre le timer.
                    _timer.Start();
                }
            }
        }

        /*
        * Méthode de callback vérifiant toutes les 25 ms les nouvelles notifications.
        * Il est à noter que si le temps de rafraîchissement des trop rapide, les messages n'ont pas
        * le temps d'être envoyés.
        */
        private void ThreadPoolCallback(Object threadContext)
        {
            while (true)
            {
                //On initialise le message OSC.
                server.InitFrame();
                //On l'envoie au client (au host et au port spécifiés dans le constructeur).
                server.CommitFrame();
                //On attend 25 ms.
                Thread.Sleep(25);

                //Si une gesture a été effectuée et que le délai d'attente est expiré.
                if (messageCreated && !gestureLocked && !modLocked)
                {
                    //On débloque la détection de gesture et on supprime l'objet envoyant les messages OSC de gesture.
                    messageCreated = false;
                    server.DeleteTuioString(2);
                }
            }
        }

        /*
         * Getters et Setters
         */
        public void setTimerElapsing(int _timerElapsing)
        {
            timerElapsing = _timerElapsing;
        }

        public int getTimerElapsing()
        {
            return timerElapsing;
        }
    }
}
