import json

from django.contrib.auth import get_user_model
from django.core.urlresolvers import reverse
from renkanmanager.models import Renkan, Revision
from rest_framework import status
from rest_framework.test import APITestCase


User = get_user_model()

class RenkanTests(APITestCase):
    
    def setUp(self):
        User.objects.create_superuser('blop', 'blop@blop.com', 'blop')
        self.user = User.objects.get(username='blop')
        _ = self.client.login(username='blop', password='blop')
        User.objects.create_user('blip', 'blip@blip.com', 'blip')
        self.unauthorized_user = User.objects.get(username='blip')
        
        # POSTing test workspace
        post_url = reverse('workspace_list')
        test_title = 'bleh_title'
        data = {'title': test_title}
        post_response = self.client.post(post_url, data, format='json')
        self.test_workspace_guid = json.loads(post_response.content).get('workspace_guid')
        
    def test_create_renkan(self):
        
        ###################################################
        # POSTing new Renkan
        ###################################################
        
        post_url = reverse('renkan_list')
        test_title = 'bleh_title'
        test_content = json.dumps({'title': 'bleh_title', 'some_random_stuff': 'wow'})
        data = {'title': test_title, 'content': test_content}
        
        post_response = self.client.post(post_url, data, format='json')
        self.assertEqual(post_response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(Renkan.objects.count(), 1)
        self.assertEqual(Revision.objects.count(), 1)
        
        # Checking POSTed Renkan data
        post_response_dict = json.loads(post_response.content)
        new_renkan_guid = post_response_dict.get('renkan_guid', '')
        new_renkan = Renkan.objects.get(renkan_guid=new_renkan_guid)
        # GUIDs and username
        self.assertEqual(self.user.username, post_response_dict.get('project_created_by', ''))
        self.assertEqual(self.user.username, post_response_dict.get('last_updated_by', ''))
        self.assertEqual(self.user.username, new_renkan.creator.username)
        # Content and title extracted from revision
        self.assertEqual(new_renkan.title, test_title)
        self.assertEqual(new_renkan.content, test_content)
        # Revision count
        self.assertEqual(new_renkan.revision_count, 1)
        self.assertEqual(post_response_dict.get('revision_count', ''), 1)
        
        # Checking POSTed Renkan revision data
        new_renkan_revision_guid = post_response_dict.get('current_revision_guid', '')
        new_renkan_revision = Revision.objects.get(revision_guid = new_renkan_revision_guid)
        # GUIDs and username
        self.assertEqual(self.user.username, new_renkan_revision.creator.username)
        self.assertEqual(self.user.username, new_renkan_revision.last_updated_by.username)
        # Title and content
        #    in the response
        self.assertEqual(post_response_dict.get('title', ''), test_title)
        self.assertEqual(post_response_dict.get('content', ''), test_content)
        #    in the database
        self.assertEqual(new_renkan_revision.title, test_title)
        self.assertEqual(new_renkan_revision.content, test_content)
        
        # Checking permission
        self.assertTrue(self.user.has_perm('view_renkan', new_renkan))
        
        ###################################################
        # POSTing new Renkan into workspace
        ###################################################
        
        post_url = reverse('renkan_list_workspace', kwargs={'workspace_guid': self.test_workspace_guid})
        test_title = 'bleh_title_2'
        test_content = json.dumps({'title': 'bleh_title_2', 'some_random_stuff': 'wow_2'})
        data = {'title': test_title, 'content': test_content}
        
        post_response = self.client.post(post_url, data, format='json')
        self.assertEqual(post_response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(Renkan.objects.count(), 2)
        self.assertEqual(Revision.objects.count(), 2)
        
        # Checking POSTed Renkan data
        post_response_dict = json.loads(post_response.content)
        new_renkan_guid = post_response_dict.get('renkan_guid', '')
        new_renkan = Renkan.objects.get(renkan_guid=new_renkan_guid)
        # GUIDs and username
        self.assertEqual(self.test_workspace_guid, post_response_dict.get('workspace_guid', ''))
        self.assertEqual(self.test_workspace_guid, new_renkan.workspace_guid)
        self.assertEqual(self.user.username, post_response_dict.get('project_created_by', ''))
        self.assertEqual(self.user.username, post_response_dict.get('last_updated_by', ''))
        self.assertEqual(self.user.username, new_renkan.creator.username)
        # Content and title extracted from revision
        self.assertEqual(new_renkan.title, test_title)
        self.assertEqual(new_renkan.content, test_content)
        # Revision count
        self.assertEqual(new_renkan.revision_count, 1)
        self.assertEqual(post_response_dict.get('revision_count', ''), 1)
        
        ###################################################
        # POSTing with wrong workspace_guid
        ###################################################
        
        post_url = reverse('renkan_list_workspace', kwargs={'workspace_guid': 'bleh-bad-workspace-guid'})
        test_title = 'bleh_title_3'
        test_content = json.dumps({'title': 'bleh_title_3', 'some_random_stuff': 'wow_3'})
        data = {'title': test_title, 'content': test_content}
        
        post_response = self.client.post(post_url, data, format='json')
        self.assertEqual(post_response.status_code, status.HTTP_404_NOT_FOUND)
        
        ###################################################
        # try POSTing with unauthorized user 
        ###################################################
        
        _ = self.client.login(username='blip', password='blip')
        post_url = reverse('renkan_list_workspace', kwargs={'workspace_guid': self.test_workspace_guid})
        test_title = 'shouldnt_work'
        test_content = json.dumps({'title': 'shouldnt_work', 'some_random_stuff': 'test'})
        data = {'title': test_title, 'content': test_content}
        post_response = self.client.post(post_url, data, format='json')
        self.assertEqual(post_response.status_code, status.HTTP_403_FORBIDDEN)
        
    def test_get_renkan(self):
        
        ###################################################
        # POSTing new Renkan
        ###################################################
        
        post_url = reverse('renkan_list')
        data = {}
        post_response = self.client.post(post_url, data, format='json')
        post_response_dict = json.loads(post_response.content)
        new_renkan_guid = post_response_dict.get('renkan_guid', '')
        new_revision_guid = post_response_dict.get('current_revision_guid', '')
        
        ###################################################
        # GETting posted Renkan
        ###################################################
        
        get_url = reverse('renkan_detail', kwargs={'renkan_guid':new_renkan_guid})
        get_response = self.client.get(get_url, format='json')
        self.assertEqual(get_response.status_code, status.HTTP_200_OK)
        
        # Checking GET data
        get_response_dict = json.loads(get_response.content)
        self.assertEqual(new_renkan_guid, get_response_dict.get('renkan_guid', ''))
        self.assertEqual(new_revision_guid, get_response_dict.get('current_revision_guid', ''))
        self.assertEqual(self.user.username, get_response_dict.get('project_created_by', ''))
        self.assertEqual(self.user.username, get_response_dict.get('last_updated_by', ''))
        
        ###################################################
        # GETting with wrong guid
        ###################################################
        
        get_url = reverse('renkan_detail', kwargs={'renkan_guid': 'bad-id'})
        get_response = self.client.get(get_url, format='json')
        self.assertEqual(get_response.status_code, status.HTTP_404_NOT_FOUND)
        
        ###################################################
        # try GETting with unauthorized user 
        ###################################################
        
        _ = self.client.login(username='blip', password='blip')
        get_url = reverse('renkan_detail', kwargs={'renkan_guid': new_renkan_guid})
        get_response = self.client.get(get_url, format='json')
        self.assertEqual(get_response.status_code, status.HTTP_403_FORBIDDEN)
    
    def test_get_renkan_list(self):
        
        ###################################################
        # POSTing new Renkans
        ###################################################
        
        post_url = reverse('renkan_list')
        first_test_title = 'bleh_title'
        first_test_content = json.dumps({'title': 'bleh_title', 'some_random_stuff': 'wow'})
        second_test_title = 'bleh_title_2'
        second_test_content = json.dumps({'title': 'bleh_title_2', 'some_random_stuff': 'such'})
        first_data = {'title': first_test_title, 'content': first_test_content}
        second_data = {'title': second_test_title, 'content': second_test_content}
        self.client.post(post_url, first_data, format='json')
        self.client.post(post_url, second_data, format='json')
        
        ###################################################
        # GETting posted Renkans
        ###################################################
        
        get_url = post_url
        get_response = self.client.get(get_url, format='json')
        self.assertEqual(get_response.status_code, status.HTTP_200_OK)
        
        get_response_content = json.loads(get_response.content) # Should be a list
        self.assertTrue(isinstance(get_response_content, list))
        self.assertEqual(len(get_response_content), 2)
        
        ###################################################
        # POSTing new Renkan into workspace
        ###################################################
        
        post_url = reverse('renkan_list_workspace', kwargs={'workspace_guid': self.test_workspace_guid})
        third_test_title = 'bleh_title'
        third_test_content = json.dumps({'title': 'bleh_title_3rd', 'some_random_stuff': 'wow much'})
        third_data = {'title': third_test_title, 'content': third_test_content}
        self.client.post(post_url, third_data, format='json')
        
        ###################################################
        # GETting posted Renkans
        ###################################################
        
        get_url = post_url
        get_response = self.client.get(get_url, format='json')
        self.assertEqual(get_response.status_code, status.HTTP_200_OK)
        
        get_response_content = json.loads(get_response.content) # Should be a list
        self.assertTrue(isinstance(get_response_content, list))
        self.assertEqual(len(get_response_content), 1)
        
    
    def test_update_renkan(self):

        ###################################################
        # POSTing new Renkan
        ###################################################
        
        post_url = reverse('renkan_list')
        data = {}
        post_response = self.client.post(post_url, data, format='json')
        post_response_dict = json.loads(post_response.content)
        renkan_guid = post_response_dict.get('renkan_guid', '')
        revision_guid = post_response_dict.get('current_revision_guid', '')
        
        ###################################################
        # Init PUT data
        ###################################################
        first_put_title = 'edited_title'
        first_put_content = json.dumps({'title': 'edited_title', 'nodes': 1})
        second_put_title = 'edited_title_2'
        second_put_content = json.dumps({'title': 'edited_title_2', 'nodes': 2})

        ###################################################
        # PUTting by updating current revision
        ###################################################
        
        put_url = reverse('renkan_detail', kwargs={'renkan_guid': renkan_guid})
        put_data = {
            'title' : first_put_title,
            'content' : first_put_content,
            'create_new_revision': False
        }
        put_response = self.client.put(put_url, put_data, format='json')
        self.assertEqual(put_response.status_code, status.HTTP_200_OK)
        put_response_dict = json.loads(put_response.content)
        # checking we're still on the same revision as the initial one
        updated_project = Renkan.objects.get(renkan_guid=renkan_guid)
        self.assertEqual(revision_guid, put_response_dict.get('current_revision_guid', ''))
        self.assertEqual(put_response_dict.get('revision_count', ''), 1)
        self.assertEqual(updated_project.revision_count, 1)
        
        self.assertEqual(revision_guid, updated_project.current_revision_guid)
        # checking data was updated
        #     in the reponse
        self.assertEqual(put_response_dict.get('title', ''), first_put_title)
        self.assertEqual(put_response_dict.get('content', ''), first_put_content)
        #     in the database
        updated_revision = Revision.objects.get(revision_guid=revision_guid)
        self.assertEqual(updated_revision.title, first_put_title)
        self.assertEqual(updated_revision.content, first_put_content)
        
        ###################################################
        # PUTting by creating a new revision
        ###################################################
        
        put_data = {
            'title' : second_put_title,
            'content' : second_put_content,
            'create_new_revision': True
        }
        put_response = self.client.put(put_url, put_data, format='json')
        self.assertEqual(put_response.status_code, status.HTTP_200_OK)  
        put_response_dict = json.loads(put_response.content)
        
        # checking we created a new revision
        updated_project = Renkan.objects.get(renkan_guid=renkan_guid)
        created_revision_guid = put_response_dict.get('current_revision_guid', '')
        self.assertNotEqual(created_revision_guid, revision_guid)
        self.assertEqual(Renkan.objects.count(), 1)
        self.assertEqual(Revision.objects.count(), 2)
        self.assertEqual(put_response_dict.get('revision_count', ''), 2)
        self.assertEqual(updated_project.revision_count, 2)
        
        # checking project now points towards new revision
        self.assertEqual(updated_project.current_revision_guid, created_revision_guid)
        # checking data was updated
        #     in the reponse
        self.assertEqual(put_response_dict.get('title', ''), second_put_title)
        self.assertEqual(put_response_dict.get('content', ''), second_put_content)
        #     in the database
        updated_revision = Revision.objects.get(revision_guid=created_revision_guid)
        self.assertEqual(updated_revision.title, second_put_title)
        self.assertEqual(updated_revision.content, second_put_content)
        
        ###################################################
        # try to update source_revision_guid or workspace_guid
        ###################################################
        
        put_data = {
            'title' : second_put_title,
            'content' : second_put_content,
            'workspace_guid': 'bleh-workspace-guid' # should fail without even checking the id
        }
        put_response = self.client.put(put_url, put_data, format='json')
        self.assertEqual(put_response.status_code, status.HTTP_400_BAD_REQUEST)
        
        put_data = {
            'title' : second_put_title,
            'content' : second_put_content,
            'source_revision_guid' : 'bleh-revision-guid' # should fail without even checking the id
        }
        put_response = self.client.put(put_url, put_data, format='json')
        self.assertEqual(put_response.status_code, status.HTTP_400_BAD_REQUEST)
        
        ###################################################
        # PUTting with wrong guid
        ###################################################
        
        put_url = reverse('renkan_detail', kwargs={'renkan_guid': 'bad-id'})
        put_response = self.client.put(put_url, {}, format='json')
        self.assertEqual(put_response.status_code, status.HTTP_404_NOT_FOUND)
        
        ###################################################
        # try PUTting with unauthorized user 
        ###################################################
        
        _ = self.client.login(username='blip', password='blip')
        put_url = reverse('renkan_detail', kwargs={'renkan_guid': renkan_guid})
        put_data = {
            'title' : 'wont_work',
            'content' : json.dumps({'title': 'wont_work', 'nodes': 0}),
            'create_new_revision': False
        }
        put_response = self.client.put(put_url, put_data, format='json')
        self.assertEqual(put_response.status_code, status.HTTP_403_FORBIDDEN)
        
    def test_copy_renkan(self):
        
        ###################################################
        # POSTing new Renkan
        ###################################################
        
        post_url = reverse('renkan_list')
        test_title = 'bleh_title'
        test_content = json.dumps({'title': 'bleh_title', 'some_random_stuff': 'wow'})
        data = {'title': test_title, 'content': test_content}
        post_response = self.client.post(post_url, data, format='json')
        post_response_dict = json.loads(post_response.content)
        source_renkan_guid = post_response_dict.get('renkan_guid', '')
        source_revision_guid = post_response_dict.get('current_revision_guid', '')
        
        ###################################################
        # POSTing copy with query arg
        ###################################################
        
        first_copy_data = {}
        first_copy_url = post_url+'?source='+source_revision_guid
        first_copy_response = self.client.post(first_copy_url, first_copy_data, format='json')
        first_copy_response_dict = json.loads(first_copy_response.content)
        self.assertEqual(first_copy_response.status_code, status.HTTP_201_CREATED)  
        self.assertEqual(Renkan.objects.count(), 2)
        # Checking data
        #     in the response
        self.assertNotEqual(first_copy_response_dict.get('renkan_guid'), source_renkan_guid)
        self.assertEqual(first_copy_response_dict.get('source_revision_guid'), source_revision_guid)
        self.assertEqual(first_copy_response_dict.get('title', ''), test_title)
        self.assertEqual(first_copy_response_dict.get('content', ''), test_content)
        self.assertEqual(first_copy_response_dict.get('project_created_by', ''), self.user.username)
        self.assertEqual(first_copy_response_dict.get('last_updated_by', ''), self.user.username)
        #     in the database
        first_copy = Renkan.objects.get(renkan_guid=first_copy_response_dict.get('renkan_guid', ''))
        first_copy_revision = Revision.objects.get(revision_guid=first_copy_response_dict.get('current_revision_guid', ''))
        self.assertEqual(first_copy.title, test_title)
        self.assertEqual(first_copy.content, test_content)
        self.assertEqual(first_copy_revision.title, test_title)
        self.assertEqual(first_copy_revision.content, test_content)
        self.assertEqual(first_copy.creator.username, self.user.username)
        self.assertEqual(first_copy_revision.creator.username, self.user.username)
        self.assertEqual(first_copy_revision.last_updated_by.username, self.user.username)
        
        ###################################################
        # POSTing copy with source_revision_guid in data
        ###################################################
        
        second_copy_data = {'source_revision_guid': source_revision_guid}
        second_copy_url = post_url+'?source='+source_revision_guid
        second_copy_response = self.client.post(second_copy_url, second_copy_data, format='json')
        second_copy_response_dict = json.loads(second_copy_response.content)
        self.assertEqual(second_copy_response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(Renkan.objects.count(), 3)
        # Checking data
        #     in the response
        self.assertNotEqual(second_copy_response_dict.get('renkan_guid'), source_renkan_guid)
        self.assertEqual(first_copy_response_dict.get('source_revision_guid'), source_revision_guid)
        self.assertEqual(second_copy_response_dict.get('title', ''), test_title)
        self.assertEqual(second_copy_response_dict.get('content', ''), test_content)
        self.assertEqual(second_copy_response_dict.get('project_created_by', ''), self.user.username)
        self.assertEqual(second_copy_response_dict.get('last_updated_by', ''), self.user.username)
        #     in the database
        second_copy = Renkan.objects.get(renkan_guid=second_copy_response_dict.get('renkan_guid', ''))
        second_copy_revision = Revision.objects.get(revision_guid=second_copy_response_dict.get('current_revision_guid', ''))
        self.assertEqual(second_copy.title, test_title)
        self.assertEqual(second_copy.content, test_content)
        self.assertEqual(second_copy_revision.title, test_title)
        self.assertEqual(second_copy_revision.content, test_content)
        self.assertEqual(second_copy.creator.username, self.user.username)
        self.assertEqual(second_copy_revision.creator.username, self.user.username)
        self.assertEqual(second_copy_revision.last_updated_by.username, self.user.username)
        
        
        ###################################################
        # try POSTing copy with bad query arg guid and with bad data guid
        ###################################################
        
        bad_copy_data = {'source_revision_guid': 'bleh_bad_id'}
        #    with query arg
        qarg_bad_copy_url = post_url+'?source=bleh_bad_id'
        qarg_bad_copy_response = self.client.post(qarg_bad_copy_url, {}, format='json')
        self.assertEqual(qarg_bad_copy_response.status_code, status.HTTP_404_NOT_FOUND)
        #    with data
        data_bad_copy_url = post_url
        data_bad_copy_response = self.client.post(data_bad_copy_url, bad_copy_data, format='json')
        self.assertEqual(data_bad_copy_response.status_code, status.HTTP_404_NOT_FOUND)
        
        ###################################################
        # try POSTing with unauthorized user 
        ###################################################
        
        _ = self.client.login(username='blip', password='blip')
        post_url = reverse('renkan_list')+'?source='+source_revision_guid
        post_response = self.client.post(post_url, {}, format='json')
        self.assertEqual(post_response.status_code, status.HTTP_403_FORBIDDEN)
        
    def test_delete_renkan(self):
        
        ###################################################
        # POSTing new Renkan
        ###################################################
        
        post_url = reverse('renkan_list')
        test_title = 'bleh_title'
        test_content = json.dumps({'title': 'bleh_title', 'some_random_stuff': 'wow'})
        data = {'title': test_title, 'content': test_content}
        post_response = self.client.post(post_url, data, format='json')
        post_response_dict = json.loads(post_response.content)
        to_delete_renkan_guid = post_response_dict.get('renkan_guid', '')
        to_delete_revision_guid = post_response_dict.get('current_revision_guid', '')
        
        ###################################################
        # POSTing copy
        ###################################################
        data = {}
        post_response = self.client.post(post_url+'?source='+to_delete_revision_guid, data, format='json')
        post_response_dict = json.loads(post_response.content)
        copy_renkan_guid = post_response_dict.get('renkan_guid', '')
        
        # Adding another revision
        put_data = {'title': test_title+'_edited!', 'create_new_revision': True}
        put_url = reverse('renkan_detail', kwargs={'renkan_guid':to_delete_renkan_guid})
        put_response = self.client.put(put_url, put_data, format='json')
        self.assertEqual(Renkan.objects.count(), 2)
        self.assertEqual(Revision.objects.count(), 3)
        
        ###################################################
        # try to DELETE with unauthorized user 
        ###################################################
        
        _ = self.client.login(username='blip', password='blip')
        delete_url = reverse('renkan_detail', kwargs={'renkan_guid':to_delete_renkan_guid})
        delete_response = self.client.delete(delete_url, format='json')
        self.assertEqual(delete_response.status_code, status.HTTP_403_FORBIDDEN)
        
        # Restoring rightful user
        _ = self.client.login(username='blop', password='blop')
        
        ###################################################
        # DELETE initial renkan
        ###################################################
        
        delete_url = reverse('renkan_detail', kwargs={'renkan_guid':to_delete_renkan_guid})
        delete_response = self.client.delete(delete_url, format='json')
        self.assertEqual(delete_response.status_code, status.HTTP_204_NO_CONTENT)
        self.assertEqual(Renkan.objects.count(), 1) # Only the copy remains
        self.assertEqual(Revision.objects.count(), 1) # Only the copy initial revision remains
        
        ###################################################
        # Try to GET deleted renkan
        ###################################################
        
        get_url = reverse('renkan_detail', kwargs={'renkan_guid':to_delete_renkan_guid})
        get_response = self.client.get(get_url, format='json')
        self.assertEqual(get_response.status_code, status.HTTP_404_NOT_FOUND)
        
        ###################################################
        # Check that remaining renkan doesn't have reference to deleted renkan
        ###################################################
        copy_renkan = Renkan.objects.get(renkan_guid=copy_renkan_guid)
        self.assertEqual(copy_renkan.source_revision_guid, '')
        
        ###################################################
        # Try to DELETE renkan with wrong guid
        ###################################################
        
        delete_url = reverse('renkan_detail', kwargs={'renkan_guid': 'bad-id'})
        delete_response = self.client.delete(delete_url, format='json')
        self.assertEqual(delete_response.status_code, status.HTTP_404_NOT_FOUND)
    