from django.core.urlresolvers import reverse
from rest_framework.test import APITestCase
from rest_framework import status
from renkanmanager.models import Renkan, Workspace
from django.contrib.auth import get_user_model
import json

User = get_user_model()

class WorkspaceTests(APITestCase):
    
    def setUp(self):
        User.objects.create_superuser('blop', 'blop@blop.com', 'blop')
        self.user = User.objects.get(username='blop')
        _ = self.client.login(username='blop', password='blop')
        User.objects.create_user('blip', 'blip@blip.com', 'blip')
        self.unauthorized_user = User.objects.get(username='blip')
    
    def test_post_workspace(self):
        
        ###################################################
        # POSTing test Workspace
        ###################################################
        
        post_url = reverse('workspace_list')
        test_title = 'bleh_title'
        data = {'title': test_title}
        post_response = self.client.post(post_url, data, format='json')
        post_response_dict = json.loads(post_response.content)
        self.assertEqual(post_response.status_code, status.HTTP_201_CREATED)
        
        # Checking data
        #    in the response
        self.assertEqual(post_response_dict.get('workspace_created_by'), self.user.username)
        self.assertEqual(post_response_dict.get('title'), test_title)
        #    in the database
        created_workspace_guid = post_response_dict.get('workspace_guid')
        created_workspace = Workspace.objects.get(workspace_guid=created_workspace_guid)
        self.assertEqual(created_workspace.creator.username, self.user.username)
        self.assertEqual(created_workspace.title, test_title)
        
        # Checking permission
        self.assertTrue(self.user.has_perm('view_workspace', created_workspace))
        
        ###################################################
        # try POSTing with unauthorized user 
        ###################################################
        
        _ = self.client.login(username='blip', password='blip')
        post_url = reverse('workspace_list')
        data = {'title': 'wont_work'}
        post_response = self.client.post(post_url, data, format='json')
        self.assertEqual(post_response.status_code, status.HTTP_403_FORBIDDEN)
        
     
    def test_get_workspace_list(self):
        
        ###################################################
        # POSTing new Workspaces
        ###################################################
        
        post_url = reverse('workspace_list')
        first_test_title = 'bleh_title_2'
        second_test_title = 'bleh_title_3'
        first_data = {'title': first_test_title}
        second_data = {'title': second_test_title}
        self.client.post(post_url, first_data, format='json')
        self.client.post(post_url, second_data, format='json')
        
        ###################################################
        # GETting Workspaces
        ###################################################
        
        get_url = post_url
        get_response = self.client.get(get_url, format=json)
        self.assertEqual(get_response.status_code, status.HTTP_200_OK)
        
        # Check data
        get_response_content = json.loads(get_response.content) # Should be a list
        self.assertTrue(isinstance(get_response_content, list))
        self.assertEqual(len(get_response_content), 2)
    
    def test_get_workspace(self):
        
        ###################################################
        # POSTing test Workspace
        ###################################################
        
        workspace_post_url = reverse('workspace_list')
        test_title = 'bleh_title'
        data = {'title': test_title}
        post_response = self.client.post(workspace_post_url, data, format='json')
        post_response_dict = json.loads(post_response.content)
        test_workspace_guid = post_response_dict.get('workspace_guid')
        
        ###################################################
        # POSTing new Renkans into test Workspace
        ###################################################
        
        renkan_post_url = reverse('renkan_list_workspace', kwargs={'workspace_guid': test_workspace_guid})
        first_test_title = 'bleh_title'
        first_test_content = json.dumps({'title': 'bleh_title', 'some_random_stuff': 'wow'})
        second_test_title = 'bleh_title_2'
        second_test_content = json.dumps({'title': 'bleh_title_2', 'some_random_stuff': 'such'})
        first_data = {'title': first_test_title, 'content': first_test_content}
        second_data = {'title': second_test_title, 'content': second_test_content}
        self.client.post(renkan_post_url, first_data, format='json')
        self.client.post(renkan_post_url, second_data, format='json')
        
        ###################################################
        # GETting test Workspace
        ###################################################
        
        get_url = reverse('workspace_detail', kwargs={'workspace_guid': test_workspace_guid})
        get_response = self.client.get(get_url, format='json')
        self.assertEqual(get_response.status_code, status.HTTP_200_OK)
        
        # Checking GET data
        get_response_dict = json.loads(get_response.content)
        self.assertEqual(test_workspace_guid, get_response_dict.get('workspace_guid', ''))
        self.assertEqual(test_title, get_response_dict.get('title', ''))
        self.assertEqual(self.user.username, get_response_dict.get('workspace_created_by', ''))
        self.assertEqual(2, get_response_dict.get('renkan_count'))
        
        ###################################################
        # GETting wrong workspace_guid
        ###################################################
        
        get_url = reverse('workspace_detail', kwargs={'workspace_guid': 'bleh-bad-workspace-id'})
        get_response = self.client.get(get_url, format='json')
        self.assertEqual(get_response.status_code, status.HTTP_404_NOT_FOUND)
        
        ###################################################
        # try GETting with unauthorized user 
        ###################################################
        
        _ = self.client.login(username='blip', password='blip')
        get_url = reverse('workspace_detail', kwargs={'workspace_guid': test_workspace_guid})
        get_response = self.client.get(get_url, format='json')
        self.assertEqual(get_response.status_code, status.HTTP_403_FORBIDDEN)
        
    def test_update_workspace(self):
        
        ###################################################
        # POSTing test Workspace
        ###################################################
        
        workspace_post_url = reverse('workspace_list')
        test_title = 'bleh_title'
        data = {'title': test_title}
        post_response = self.client.post(workspace_post_url, data, format='json')
        post_response_dict = json.loads(post_response.content)
        test_workspace_guid = post_response_dict.get('workspace_guid')
        
        ###################################################
        # PUTting into test Workspace
        ###################################################
        
        put_url = reverse('workspace_detail', kwargs={'workspace_guid': test_workspace_guid})
        edited_title = 'bleh_title_2'
        put_data = {'title': edited_title}
        put_response = self.client.put(put_url, put_data, format='json')
        put_response_dict = json.loads(put_response.content)
        self.assertEqual(put_response.status_code, status.HTTP_200_OK)
        
        # Checking data
        #    in the response
        self.assertEqual(put_response_dict.get('workspace_created_by'), self.user.username)
        self.assertEqual(put_response_dict.get('title'), edited_title)
        self.assertEqual(put_response_dict.get('renkan_count'), 0)
        #    in the database
        edited_workspace_guid = put_response_dict.get('workspace_guid')
        edited_workspace = Workspace.objects.get(workspace_guid=edited_workspace_guid)
        self.assertEqual(edited_workspace.creator.username, self.user.username)
        self.assertEqual(edited_workspace.title, edited_title)
        self.assertEqual(edited_workspace.renkan_count, 0)
        
        ###################################################
        # try PUTting with unauthorized user 
        ###################################################
        
        _ = self.client.login(username='blip', password='blip')
        put_url = reverse('workspace_detail', kwargs={'workspace_guid': test_workspace_guid})
        put_data = {
            'title' : 'wont_work'
        }
        put_response = self.client.put(put_url, put_data, format='json')
        self.assertEqual(put_response.status_code, status.HTTP_403_FORBIDDEN)
        
    def test_delete_workspace(self):
        
        ###################################################
        # POSTing test Workspace
        ###################################################
        
        workspace_post_url = reverse('workspace_list')
        test_title = 'bleh_title'
        data = {'title': test_title}
        post_response = self.client.post(workspace_post_url, data, format='json')
        post_response_dict = json.loads(post_response.content)
        to_delete_workspace_guid = post_response_dict.get('workspace_guid')
        
        ###################################################
        # POSTing new Renkan into test Workspace
        ###################################################
        
        renkan_post_url = reverse('renkan_list_workspace', kwargs={'workspace_guid': to_delete_workspace_guid})
        renkan_test_title = 'bleh_title'
        renkan_test_content = json.dumps({'title': 'bleh_title', 'some_random_stuff': 'wow'})
        renkan_test_data = {'title': renkan_test_title, 'content': renkan_test_content}
        post_response = self.client.post(renkan_post_url, renkan_test_data, format='json')
        to_delete_renkan_guid = json.loads(post_response.content).get('renkan_guid', '')
        
        ###################################################
        # Try to DELETE test Workspace (should fail, it still contains a Renkan)
        ###################################################
        
        delete_url = reverse('workspace_detail', kwargs={'workspace_guid':to_delete_workspace_guid})
        delete_response = self.client.delete(delete_url, {}, format='json')
        self.assertEqual(delete_response.status_code, status.HTTP_400_BAD_REQUEST)
        
        ###################################################
        # DELETE posted Renkan
        ###################################################
        
        delete_renkan_url = reverse('renkan_detail', kwargs={'renkan_guid':to_delete_renkan_guid})
        delete_renkan_response = self.client.delete(delete_renkan_url, format='json')
        self.assertEqual(delete_renkan_response.status_code, status.HTTP_204_NO_CONTENT)
        
        ###################################################
        # try to DELETE with unauthorized user 
        ###################################################
        
        _ = self.client.login(username='blip', password='blip')
        delete_url = reverse('workspace_detail', kwargs={'workspace_guid':to_delete_workspace_guid})
        delete_response = self.client.delete(delete_url, {}, format='json')
        self.assertEqual(delete_response.status_code, status.HTTP_403_FORBIDDEN)
        
        # Restoring rightful user
        _ = self.client.login(username='blop', password='blop')
        
        ###################################################
        # DELETE test Workspace 
        ###################################################
        
        delete_url = reverse('workspace_detail', kwargs={'workspace_guid':to_delete_workspace_guid})
        delete_response = self.client.delete(delete_url, {}, format='json')
        self.assertEqual(delete_response.status_code, status.HTTP_204_NO_CONTENT)
        
        ###################################################
        # Try to DELETE workspace with wrong guid
        ###################################################
        
        delete_url = reverse('workspace_detail', kwargs={'workspace_guid': 'bad-workspace-guid'})
        delete_response = self.client.delete(delete_url, format=json)
        self.assertEqual(delete_response.status_code, status.HTTP_404_NOT_FOUND)
        