import uuid

from guardian.shortcuts import assign_perm
from models import Renkan, Workspace, Revision
from rest_framework import serializers


class RenkanSerializer(serializers.Serializer):
    renkan_guid = serializers.ReadOnlyField()
    current_revision_guid = serializers.ReadOnlyField()
    workspace_guid = serializers.CharField(required=False)
    source_revision_guid = serializers.CharField(required=False)
    revision_count = serializers.ReadOnlyField()
    project_created_by = serializers.StringRelatedField(source="creator")
    last_updated_by = serializers.SerializerMethodField("get_current_revision_last_updator")
    title = serializers.CharField(required=False)
    content = serializers.CharField(required=False)
    creation_date = serializers.ReadOnlyField()
    modification_date = serializers.SerializerMethodField("get_current_revision_modification_date")
    create_new_revision = serializers.BooleanField(write_only=True, required=False) # only used for updating
    
    # ADD ERROR HANDLING
    def get_current_revision_last_updator(self, renkan):
        current_revision = Revision.objects.get(revision_guid = renkan.current_revision_guid)
        return current_revision.last_updated_by.username
    
    
    def get_current_revision_modification_date(self, renkan):
        current_revision = Revision.objects.get(revision_guid = renkan.current_revision_guid)
        return current_revision.modification_date
    
    def create(self, validated_data):
        """
            Method to create a new Renkan (and its first revision)
        """
        creator = validated_data.get('creator')
        renkan = Renkan.objects.create()
        initial_revision = Revision.objects.create()
        initial_revision.title = validated_data.get('title', '')
        initial_revision.content = validated_data.get('content', '{}')
        initial_revision.parent_renkan_guid = renkan.renkan_guid
        initial_revision.creator = creator
        initial_revision.last_updated_by = validated_data.get('creator', '')
        renkan.current_revision_guid = initial_revision.revision_guid
        renkan.workspace_guid = validated_data.get('workspace_guid', '')
        renkan.source_revision_guid = validated_data.get('source_revision_guid', '')
        renkan.creator = creator
        initial_revision.save()
        renkan.save()
        assign_perm('view_renkan', creator, renkan)
        assign_perm('change_renkan', creator, renkan)
        assign_perm('delete_renkan', creator, renkan)
        assign_perm('view_revision', creator, initial_revision)
        assign_perm('delete_revision', creator, initial_revision)
        return renkan
    
    def update(self, renkan, validated_data):
        """
            Method to update a Renkan object. Creates a new revision if needed
        """
        updator = validated_data.get('updator')
        if validated_data.get("create_new_revision", False):
            revision_to_update = Revision.objects.create()
            revision_to_update.parent_renkan_guid = renkan.renkan_guid
            revision_to_update.creator = updator
            renkan.current_revision_guid = revision_to_update.revision_guid
        else:
            revision_to_update = Revision.objects.get(revision_guid=renkan.current_revision_guid)   
        
        revision_to_update.title = validated_data.get('title', '')
        revision_to_update.content = validated_data.get('content', '{}')
        revision_to_update.last_updated_by = updator
        revision_to_update.save()
        if not updator.has_perm('view_revision', revision_to_update):
            assign_perm('view_revision', updator, revision_to_update)
            assign_perm('delete_revision', updator, revision_to_update)
        renkan.save()
        return renkan
    
    def validate_workspace_guid(self, value):
        if self.instance is not None:
            raise serializers.ValidationError("You cannot update workspace_guid")
        return value
    
    def validate_source_revision_guid(self, value):
        if self.instance is not None:
            raise serializers.ValidationError("You cannot update source_revision_guid")
        return value


class RevisionSerializer(serializers.Serializer):
    revision_guid = serializers.ReadOnlyField()
    parent_renkan_guid = serializers.ReadOnlyField()
    workspace_guid = serializers.SerializerMethodField("get_related_workspace_guid")
    title = serializers.ReadOnlyField()
    content = serializers.ReadOnlyField()
    project_created_by = serializers.SerializerMethodField("get_related_project_creator")
    project_creation_date = serializers.SerializerMethodField("get_related_project_creation_date")
    revision_created_by = serializers.StringRelatedField(source="creator")
    revision_last_updated_by = serializers.StringRelatedField(source="last_updated_by")
    revision_modification_date = serializers.ReadOnlyField(source="modification_date")
    
    def get_related_workspace_guid(self, revision):
        parent_project = Renkan.objects.get(renkan_guid = revision.parent_renkan_guid)
        return parent_project.workspace_guid
    
    def get_related_project_creator(self, revision):
        parent_project = Renkan.objects.get(renkan_guid = revision.parent_renkan_guid)
        return parent_project.creator.username
    
    def get_related_project_creation_date(self, revision):
        parent_project = Renkan.objects.get(renkan_guid = revision.parent_renkan_guid)
        return parent_project.creation_date
    

class WorkspaceSerializer(serializers.Serializer):
    workspace_guid = serializers.ReadOnlyField()
    workspace_created_by = serializers.StringRelatedField(source="creator")
    creation_date = serializers.ReadOnlyField()
    renkan_count = serializers.ReadOnlyField()
    title = serializers.CharField()
    
    def create(self, validated_data):
        creator = validated_data.get('creator')
        workspace = Workspace.objects.create()
        workspace.title = validated_data.get('title', '')
        workspace.creator = creator
        workspace.save()
        assign_perm('view_workspace', creator, workspace)
        assign_perm('change_workspace', creator, workspace)
        assign_perm('delete_workspace', creator, workspace)
        return workspace
    
    def update(self, workspace, validated_data):
        workspace.title = validated_data.get('title', '')
        workspace.save()
        return workspace