# -*- coding: utf-8 -*-
'''
Created on Jul 01, 2014

@author: tc
'''
from datetime import datetime
import json
import logging
import uuid

from django.core.urlresolvers import reverse
from django.http import Http404
from django.http.response import HttpResponse, HttpResponseBadRequest
from django.shortcuts import get_object_or_404, redirect
from django.views.decorators.csrf import csrf_exempt
from django.views.generic import View
from renkanmanager.models import Renkan, Revision, Workspace
from renkanmanager.permissions import CanEditRenkan, CanDeleteRenkan, CanCreateRenkan, \
                                      CanEditWorkspace, CanDeleteWorkspace, CanCreateWorkspace, \
                                      CanDeleteRevision
from renkanmanager.serializers import RenkanSerializer, RevisionSerializer, WorkspaceSerializer
from renkanmanager.utils import LineNodePlacer, renkan_copier, renkan_deleter
from rest_framework import permissions, status
from rest_framework.response import Response
from rest_framework.views import APIView


logger = logging.getLogger(__name__)
            
class RenkanList(APIView):
    """
        View for listing renkans or posting new renkan
    """
    permission_classes = (
        permissions.IsAuthenticatedOrReadOnly,
        CanCreateRenkan,
    )
    
    def get(self, request, workspace_guid='', format=None):
        if workspace_guid == '':
            renkans = Renkan.objects.all()
        else:
            renkans = Renkan.objects.filter(workspace_guid=workspace_guid)
        serializer = RenkanSerializer(renkans, many=True)
        return Response(serializer.data, status=status.HTTP_200_OK)    
    
    def post(self, request, workspace_guid='', format=None):
        create_data = request.data
        # We check if the POST request is a copy request, either with ?source="id" query arg or with source_revision_guid present in the data
        source_revision_guid = request.GET.get("source", None) if request.GET.get("source", None) is not None else request.data.get("source_revision_guid", None)
        if source_revision_guid is not None:
            try:
                source_revision=Revision.objects.get(revision_guid=source_revision_guid)
            except Revision.DoesNotExist:
                return Response({'detail': 'Source revision '+source_revision_guid+' does not exist'}, status=status.HTTP_404_NOT_FOUND)
            create_data["source_revision_guid"] = source_revision.revision_guid
            create_data["title"] = source_revision.title
            create_data["content"] = source_revision.content
        if workspace_guid:
            try:
                workspace = Workspace.objects.get(workspace_guid=workspace_guid)
            except Workspace.DoesNotExist:
                return Response({'detail': 'Workspace '+workspace_guid+' does not exist'}, status=status.HTTP_404_NOT_FOUND)
            create_data["workspace_guid"] = workspace_guid
        serializer = RenkanSerializer(data=create_data)
        if serializer.is_valid():
            serializer.save(creator=request.user)       
            return Response(serializer.data, status=status.HTTP_201_CREATED)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
        

class RenkanDetail(APIView):
    """
        View for retrieving, updating or deleting a single renkan
    """
    lookup_field = "renkan_guid"
    permission_classes = (
        permissions.IsAuthenticatedOrReadOnly, 
        CanEditRenkan,
        CanDeleteRenkan,
    )
    
    def get_object(self, renkan_guid):
        return Renkan.objects.get(renkan_guid=renkan_guid)
    
    def get(self, request, renkan_guid, format=None):
        try:
            renkan = self.get_object(renkan_guid=renkan_guid)
        except Renkan.DoesNotExist:
            return Response({'detail': 'Renkan project '+renkan_guid+' does not exist'}, status=status.HTTP_404_NOT_FOUND)
        self.check_object_permissions(request, renkan)
        serializer = RenkanSerializer(renkan)
        return Response(serializer.data, status=status.HTTP_200_OK)
    
    def put(self, request, renkan_guid, format=None):
        try:
            renkan = self.get_object(renkan_guid=renkan_guid)
        except Renkan.DoesNotExist:
            return Response({'detail': 'Renkan project '+renkan_guid+' does not exist'}, status=status.HTTP_404_NOT_FOUND)
        self.check_object_permissions(request, renkan)
        serializer = RenkanSerializer(renkan, data=request.data)
        if serializer.is_valid():
            serializer.save(updator=request.user)
            return Response(serializer.data, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    
    def delete(self, request, renkan_guid, format=None):
        try:
            to_delete_renkan = self.get_object(renkan_guid=renkan_guid)
        except Renkan.DoesNotExist:
            return Response({'detail': 'Renkan project '+renkan_guid+' does not exist'}, status=status.HTTP_404_NOT_FOUND)
        self.check_object_permissions(request, to_delete_renkan)
        renkan_revisions = Revision.objects.filter(parent_renkan_guid = to_delete_renkan.renkan_guid)
        for child_revision in renkan_revisions:
            # Deleting reference to revision in renkans copied from this revision
            for related_renkan in Renkan.objects.filter(source_revision_guid=child_revision.revision_guid):
                related_renkan.source_revision_guid = ''
                related_renkan.save()
            child_revision.delete()
        to_delete_renkan.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)

class WorkspaceList(APIView):
    """
        View for listing workspaces or creating new workspace
    """
    permission_classes = (
        permissions.IsAuthenticatedOrReadOnly,
        CanCreateWorkspace,
    )
    
    def get(self, request, format=None):
        workspaces = Workspace.objects.all()
        serializer = WorkspaceSerializer(workspaces, many=True)
        return Response(serializer.data)
    
    def post(self, request, format=None):
        serializer = WorkspaceSerializer(data=request.data)
        if serializer.is_valid():
            serializer.save(creator=request.user)       
            return Response(serializer.data, status=status.HTTP_201_CREATED)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    
class WorkspaceDetail(APIView):
    """
        View for retrieving, updating or deleting a single workspace
    """
    permission_classes = (
        permissions.IsAuthenticatedOrReadOnly,
        CanEditWorkspace,
        CanDeleteWorkspace,
    )
    
    def get_object(self, workspace_guid):
        return Workspace.objects.get(workspace_guid=workspace_guid)
    
    def get(self, request, workspace_guid, format=None):
        try:
            workspace = Workspace.objects.get(workspace_guid=workspace_guid)
        except Workspace.DoesNotExist:
            return Response({'detail': 'Workspace '+workspace_guid+' does not exist.'}, status=status.HTTP_404_NOT_FOUND)
        self.check_object_permissions(request, workspace)
        serializer = WorkspaceSerializer(workspace)
        return Response(serializer.data, status=status.HTTP_200_OK)
    
    def put(self, request, workspace_guid, format=None):
        try:
            workspace = Workspace.objects.get(workspace_guid=workspace_guid)
        except Workspace.DoesNotExist:
            return Response({'detail': 'Workspace '+workspace_guid+' does not exist.'}, status=status.HTTP_404_NOT_FOUND)
        self.check_object_permissions(request, workspace)
        serializer = WorkspaceSerializer(workspace, data=request.data)
        if serializer.is_valid():
            serializer.save()       
            return Response(serializer.data, status=status.HTTP_200_OK)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)
    
    def delete(self, request, workspace_guid, format=None):
        try:
            to_delete_workspace = self.get_object(workspace_guid=workspace_guid)
        except Workspace.DoesNotExist:
            return Response({'detail': 'Workspace '+workspace_guid+' does not exist'}, status=status.HTTP_404_NOT_FOUND)
        self.check_object_permissions(request, to_delete_workspace)
        if to_delete_workspace.renkan_count != 0:
            return Response({'detail': 'Workspace '+workspace_guid+' cannot be deleted because it is not empty'}, status=status.HTTP_400_BAD_REQUEST)
        to_delete_workspace.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)

class RevisionList(APIView):
    """
        View for listing revisions from a given renkan
    """
    permission_classes = (permissions.IsAuthenticatedOrReadOnly,)
    
    def get_queryset(self, renkan_guid):
        return Revision.objects.filter(parent_renkan_guid=renkan_guid)
    
    def get(self, request, renkan_guid, format=None):
        revisions = self.get_queryset(renkan_guid)
        if not revisions:
            return Response({'detail': 'Renkan project '+renkan_guid+' does not exist'}, status=status.HTTP_404_NOT_FOUND)
        serializer = RevisionSerializer(revisions.all(), many=True)
        return Response(serializer.data, status=status.HTTP_200_OK)

class RevisionDetail(APIView):
    """
        View for retrieving or deleting a single revision from a given renkan
    """
    permission_classes = (
        permissions.IsAuthenticatedOrReadOnly,
        CanDeleteRevision,
    )
    
    def get_queryset(self, renkan_guid):
        return Revision.objects.filter(parent_renkan_guid=renkan_guid)
    
    def get(self, request, renkan_guid, revision_guid, format=None):
        revisions = self.get_queryset(renkan_guid)
        if not revisions:
            return Response({'detail': 'Renkan project '+renkan_guid+' does not exist'}, status=status.HTTP_404_NOT_FOUND)
        try:
            revision = revisions.get(revision_guid=revision_guid)
        except Revision.DoesNotExist:
            return Response({'detail': 'Revision '+revision_guid+' does not exist'}, status=status.HTTP_404_NOT_FOUND)
        self.check_object_permissions(request, revision)
        serializer = RevisionSerializer(revision)
        return Response(serializer.data, status=status.HTTP_200_OK)
    
    def delete(self, request, renkan_guid, revision_guid, format=None):
        revisions = self.get_queryset(renkan_guid)
        if not revisions:
            return Response({'detail': 'Renkan project '+renkan_guid+' does not exist'}, status=status.HTTP_404_NOT_FOUND)
        try:
            revision = revisions.get(revision_guid=revision_guid)
        except Revision.DoesNotExist:
            return Response({'detail': 'Revision '+revision_guid+' does not exist'}, status=status.HTTP_404_NOT_FOUND)
        self.check_object_permissions(request, revision)
        if revisions.count() == 1:
            return Response({'detail': 'You cannot delete the last remaining revision of a renkan from the Revision API. Try deleting the parent Renkan with the Renkan API'}, status=status.HTTP_400_BAD_REQUEST)
        if revision.is_current_revision:
            return Response({'detail': 'You cannot delete the current revision of a renkan from the Revision API.'}, status=status.HTTP_400_BAD_REQUEST)
        
        # Deleting reference to revision in renkans copied from this revision
        for related_renkan in Renkan.objects.filter(source_revision_guid=revision_guid):
            related_renkan.source_revision_guid = ''
            related_renkan.save()
        revision.delete()
        return Response(status=status.HTTP_204_NO_CONTENT)