import json

from django.contrib.auth import get_user_model
from django.core.urlresolvers import reverse
from renkanmanager.models import Renkan, Revision
from rest_framework import status
from rest_framework.test import APITestCase


User = get_user_model()

class RenkanTests(APITestCase):
    
    def setUp(self):
        User.objects.create_superuser("blop", "blop@blop.com", "blop")
        self.user = User.objects.get_by_natural_key(username="blop")
        _ = self.client.login(username="blop", password="blop")
        User.objects.create_user("blip", "blip@blip.com", "blip")
        self.unauthorized_user = User.objects.get_by_natural_key(username="blip")
        
        # POSTing test workspace
        post_url = reverse("v1.0:workspace_list")
        test_title = "bleh_title"
        data = {"title": test_title}
        post_response = self.client.post(post_url, data, format="json")
        self.test_workspace_guid = json.loads(post_response.content.decode()).get("id")
        self.first_test_title = "test_title_1"
        self.first_test_content = json.dumps({
            "id": "",
            "title": "test_title_1",
            "description": "test_description_1",
            "created": "",
            "updated": "",
            "edges": [],
            "nodes": [],
            "users": [],
            "space_id": "",
            "views": []
        })
        self.second_test_title = "test_title_1"
        self.second_test_content = json.dumps({
            "id": "",
            "title": "test_title_2",
            "description": "test_description_2",
            "created": "",
            "updated": "",
            "edges": [],
            "nodes": [],
            "users": [],
            "space_id": "",
            "views": []
        })
        self.third_test_title = "test_title_1"
        self.third_test_content = json.dumps({
            "id": "",
            "title": "test_title_3",
            "description": "test_description_3",
            "created": "",
            "updated": "",
            "edges": [],
            "nodes": [],
            "users": [],
            "space_id": "",
            "views": []
        })
        
    def test_create_renkan(self):
        
        ###################################################
        # POSTing new Renkan
        ###################################################
        
        post_url = reverse("v1.0:renkan_list")
        data = {"title": self.first_test_title, "content": self.first_test_content}
        
        post_response = self.client.post(post_url, data, format="json")
        self.assertEqual(post_response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(Renkan.objects.count(), 1)
        self.assertEqual(Revision.objects.count(), 1)
        
        # Checking POSTed Renkan data
        post_response_dict = json.loads(post_response.content.decode())
        new_renkan_guid = post_response_dict.get("id", "")
        new_renkan = Renkan.objects.get(renkan_guid=new_renkan_guid)
        # GUIDs and username
        self.assertEqual(getattr(self.user, User.USERNAME_FIELD), post_response_dict.get("created_by", ""))
        self.assertEqual(getattr(self.user, User.USERNAME_FIELD), post_response_dict.get("last_updated_by", ""))
        self.assertEqual(getattr(self.user, User.USERNAME_FIELD), getattr(new_renkan.creator, User.USERNAME_FIELD))
        # Content and title extracted from revision
        self.assertEqual(new_renkan.title, self.first_test_title)
        self.assertEqual(new_renkan.content, self.first_test_content)
        # Revision count
        self.assertEqual(new_renkan.revision_count, 1)
        self.assertEqual(post_response_dict.get("revision_count", ""), 1)
        
        # Checking POSTed Renkan revision data
        new_renkan_revision_guid = post_response_dict.get("current_revision_id", "")
        new_renkan_revision = Revision.objects.get(revision_guid = new_renkan_revision_guid)
        # GUIDs and username
        self.assertEqual(getattr(self.user, User.USERNAME_FIELD), getattr(new_renkan_revision.creator, User.USERNAME_FIELD))
        self.assertEqual(getattr(self.user, User.USERNAME_FIELD), getattr(new_renkan_revision.last_updated_by, User.USERNAME_FIELD))
        # Title and content
        #    in the response
        self.assertEqual(post_response_dict.get("title", ""), self.first_test_title)
        self.assertEqual(post_response_dict.get("content", ""), self.first_test_content)
        #    in the database
        self.assertEqual(new_renkan_revision.title, self.first_test_title)
        self.assertEqual(new_renkan_revision.content, self.first_test_content)
        
        # Checking permission
        self.assertTrue(self.user.has_perm("view_renkan", new_renkan))
        
        ###################################################
        # POSTing new Renkan into workspace
        ###################################################
        
        post_url = reverse("v1.0:renkan_list_workspace", kwargs={"workspace_guid": self.test_workspace_guid})
        data = {"title": self.second_test_title, "content": self.second_test_content}
        
        post_response = self.client.post(post_url, data, format="json")
        self.assertEqual(post_response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(Renkan.objects.count(), 2)
        self.assertEqual(Revision.objects.count(), 2)
        
        # Checking POSTed Renkan data
        post_response_dict = json.loads(post_response.content.decode())
        new_renkan_guid = post_response_dict.get("id", "")
        new_renkan = Renkan.objects.get(renkan_guid=new_renkan_guid)
        # GUIDs and username
        self.assertEqual(self.test_workspace_guid, post_response_dict.get("workspace_id", ""))
        self.assertEqual(self.test_workspace_guid, new_renkan.workspace_guid)
        self.assertEqual(getattr(self.user, User.USERNAME_FIELD), post_response_dict.get("created_by", ""))
        self.assertEqual(getattr(self.user, User.USERNAME_FIELD), post_response_dict.get("last_updated_by", ""))
        self.assertEqual(getattr(self.user, User.USERNAME_FIELD), getattr(new_renkan.creator, User.USERNAME_FIELD))
        # Content and title extracted from revision
        self.assertEqual(new_renkan.title, self.second_test_title)
        self.assertEqual(new_renkan.content, self.second_test_content)
        # Revision count
        self.assertEqual(new_renkan.revision_count, 1)
        self.assertEqual(post_response_dict.get("revision_count", ""), 1)
        
        ###################################################
        # POSTing new Renkan with no content
        ###################################################
        post_url = reverse("v1.0:renkan_list_workspace", kwargs={"workspace_guid": self.test_workspace_guid})
        test_title = "bleh_title_2"
        data = {"title": test_title}
        post_response = self.client.post(post_url, data, format="json")
        self.assertEqual(post_response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(Renkan.objects.count(), 3)
        self.assertEqual(Revision.objects.count(), 3)
        
        ###################################################
        # POSTing with wrong workspace_guid
        ###################################################
        
        post_url = reverse("v1.0:renkan_list_workspace", kwargs={"workspace_guid": "bleh-bad-workspace-guid"})
        data = {"title": self.third_test_title, "content": self.third_test_content}
        
        post_response = self.client.post(post_url, data, format="json")
        self.assertEqual(post_response.status_code, status.HTTP_404_NOT_FOUND)
        
        ###################################################
        # try POSTing with unauthorized user 
        ###################################################
        
        _ = self.client.login(username="blip", password="blip")
        post_url = reverse("v1.0:renkan_list_workspace", kwargs={"workspace_guid": self.test_workspace_guid})
        data = {"title": self.third_test_title, "content": self.third_test_content}
        post_response = self.client.post(post_url, data, format="json")
        self.assertEqual(post_response.status_code, status.HTTP_403_FORBIDDEN)
        
    def test_get_renkan(self):
        
        ###################################################
        # POSTing new Renkan
        ###################################################
        
        post_url = reverse("v1.0:renkan_list")
        data = {}
        post_response = self.client.post(post_url, data, format="json")
        post_response_dict = json.loads(post_response.content.decode())
        new_renkan_guid = post_response_dict.get("id", "")
        new_revision_guid = post_response_dict.get("current_revision_id", "")
        
        ###################################################
        # GETting posted Renkan
        ###################################################
        
        get_url = reverse("v1.0:renkan_detail", kwargs={"renkan_guid":new_renkan_guid})
        get_response = self.client.get(get_url, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_200_OK)
        
        # Checking GET data
        get_response_dict = json.loads(get_response.content.decode())
        self.assertEqual(new_renkan_guid, get_response_dict.get("id", ""))
        self.assertEqual(new_revision_guid, get_response_dict.get("current_revision_id", ""))
        self.assertEqual(getattr(self.user, User.USERNAME_FIELD), get_response_dict.get("created_by", ""))
        self.assertEqual(getattr(self.user, User.USERNAME_FIELD), get_response_dict.get("last_updated_by", ""))
        
        ###################################################
        # GETting with wrong guid
        ###################################################
        
        get_url = reverse("v1.0:renkan_detail", kwargs={"renkan_guid": "bad-id"})
        get_response = self.client.get(get_url, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_404_NOT_FOUND)
        
        ###################################################
        # try GETting with unauthorized user 
        ###################################################
        
        _ = self.client.login(username="blip", password="blip")
        get_url = reverse("v1.0:renkan_detail", kwargs={"renkan_guid": new_renkan_guid})
        get_response = self.client.get(get_url, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_403_FORBIDDEN)
    
    def test_get_renkan_list(self):
        
        ###################################################
        # POSTing new Renkans
        ###################################################
        
        post_url = reverse("v1.0:renkan_list")
        first_data = {"title": self.first_test_title, "content": self.first_test_content}
        second_data = {"title": self.second_test_title, "content": self.second_test_content}
        self.client.post(post_url, first_data, format="json")
        self.client.post(post_url, second_data, format="json")
        
        ###################################################
        # GETting posted Renkans
        ###################################################
        
        get_url = post_url
        get_response = self.client.get(get_url, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_200_OK)
        
        get_response_content = json.loads(get_response.content.decode()) # Should be a list
        self.assertTrue(isinstance(get_response_content, list))
        self.assertEqual(len(get_response_content), 2)
        
        ###################################################
        # POSTing new Renkan into workspace
        ###################################################
        
        post_url = reverse("v1.0:renkan_list_workspace", kwargs={"workspace_guid": self.test_workspace_guid})
        third_data = {"title": self.third_test_title, "content": self.third_test_content}
        self.client.post(post_url, third_data, format="json")
        
        ###################################################
        # GETting posted Renkans
        ###################################################
        
        get_url = post_url
        get_response = self.client.get(get_url, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_200_OK)
        
        get_response_content = json.loads(get_response.content.decode()) # Should be a list
        self.assertTrue(isinstance(get_response_content, list))
        self.assertEqual(len(get_response_content), 1)
        
    
    def test_update_renkan(self):

        ###################################################
        # POSTing new Renkan
        ###################################################
        
        post_url = reverse("v1.0:renkan_list")
        data = {}
        post_response = self.client.post(post_url, data, format="json")
        post_response_dict = json.loads(post_response.content.decode())
        renkan_guid = post_response_dict.get("id", "")
        revision_guid = post_response_dict.get("current_revision_id", "")

        ###################################################
        # PUTting by updating current revision
        ###################################################
        
        put_url = reverse("v1.0:renkan_detail", kwargs={"renkan_guid": renkan_guid})
        put_data = {
            "title" : self.first_test_title,
            "content" : self.first_test_content,
            "create_new_revision": False
        }
        put_response = self.client.put(put_url, put_data, format="json")
        self.assertEqual(put_response.status_code, status.HTTP_200_OK)
        put_response_dict = json.loads(put_response.content.decode())
        # checking we"re still on the same revision as the initial one
        updated_project = Renkan.objects.get(renkan_guid=renkan_guid)
        self.assertEqual(revision_guid, put_response_dict.get("current_revision_id", ""))
        self.assertEqual(put_response_dict.get("revision_count", ""), 1)
        self.assertEqual(updated_project.revision_count, 1)
        
        self.assertEqual(revision_guid, updated_project.current_revision_guid)
        # checking data was updated
        #     in the reponse
        self.assertEqual(put_response_dict.get("title", ""), self.first_test_title)
        self.assertEqual(put_response_dict.get("content", ""), self.first_test_content)
        #     in the database
        updated_revision = Revision.objects.get(revision_guid=revision_guid)
        self.assertEqual(updated_revision.title, self.first_test_title)
        self.assertEqual(updated_revision.content, self.first_test_content)
        
        ###################################################
        # PUTting by creating a new revision
        ###################################################
        
        put_data = {
            "title" : self.second_test_title,
            "content" : self.second_test_content,
            "create_new_revision": True
        }
        put_response = self.client.put(put_url, put_data, format="json")
        self.assertEqual(put_response.status_code, status.HTTP_200_OK)  
        put_response_dict = json.loads(put_response.content.decode())
        
        # checking we created a new revision
        updated_project = Renkan.objects.get(renkan_guid=renkan_guid)
        created_revision_guid = put_response_dict.get("current_revision_id", "")
        self.assertNotEqual(created_revision_guid, revision_guid)
        self.assertEqual(Renkan.objects.count(), 1)
        self.assertEqual(Revision.objects.count(), 2)
        self.assertEqual(put_response_dict.get("revision_count", ""), 2)
        self.assertEqual(updated_project.revision_count, 2)
        
        # checking project now points towards new revision
        self.assertEqual(updated_project.current_revision_guid, created_revision_guid)
        # checking data was updated
        #     in the reponse
        self.assertEqual(put_response_dict.get("title", ""), self.second_test_title)
        self.assertEqual(put_response_dict.get("content", ""), self.second_test_content)
        #     in the database
        updated_revision = Revision.objects.get(revision_guid=created_revision_guid)
        self.assertEqual(updated_revision.title, self.second_test_title)
        self.assertEqual(updated_revision.content, self.second_test_content)
        
        ###################################################
        # try to update source_revision_guid or workspace_guid
        ###################################################
        
        put_data = {
            "title" : self.third_test_title,
            "content" : self.third_test_content,
            "workspace_id": "bleh-workspace-guid" # should fail without even checking the id
        }
        put_response = self.client.put(put_url, put_data, format="json")
        self.assertEqual(put_response.status_code, status.HTTP_400_BAD_REQUEST)
        
        put_data = {
            "title" : self.third_test_title,
            "content" : self.third_test_content,
            "source_revision_id" : "bleh-revision-guid" # should fail without even checking the id
        }
        put_response = self.client.put(put_url, put_data, format="json")
        self.assertEqual(put_response.status_code, status.HTTP_400_BAD_REQUEST)
        
        ###################################################
        # PUTting with wrong guid
        ###################################################
        
        put_url = reverse("v1.0:renkan_detail", kwargs={"renkan_guid": "bad-id"})
        put_response = self.client.put(put_url, {}, format="json")
        self.assertEqual(put_response.status_code, status.HTTP_404_NOT_FOUND)
        
        ###################################################
        # try PUTting with unauthorized user 
        ###################################################
        
        _ = self.client.login(username="blip", password="blip")
        put_url = reverse("v1.0:renkan_detail", kwargs={"renkan_guid": renkan_guid})
        put_data = {
            "title" : self.third_test_title,
            "content" : self.third_test_content,
            "create_new_revision": False
        }
        put_response = self.client.put(put_url, put_data, format="json")
        self.assertEqual(put_response.status_code, status.HTTP_403_FORBIDDEN)
        
    def test_copy_renkan(self):
        
        ###################################################
        # POSTing new Renkan
        ###################################################
        
        post_url = reverse("v1.0:renkan_list")
        data = {"title": self.first_test_title, "content": self.first_test_content}
        post_response = self.client.post(post_url, data, format="json")
        post_response_dict = json.loads(post_response.content.decode())
        source_renkan_guid = post_response_dict.get("id", "")
        source_revision_guid = post_response_dict.get("current_revision_id", "")
        
        ###################################################
        # POSTing copy with query arg for RENKAN guid
        ###################################################
        
        first_copy_data = {"title": "new_title_copy_1"}
        first_copy_url = post_url+"?source_renkan_id="+source_renkan_guid
        first_copy_response = self.client.post(first_copy_url, first_copy_data, format="json")
        first_copy_response_dict = json.loads(first_copy_response.content.decode())
        self.assertEqual(first_copy_response.status_code, status.HTTP_201_CREATED)  
        self.assertEqual(Renkan.objects.count(), 2)
        # Checking data
        #     in the response
        self.assertNotEqual(first_copy_response_dict.get("id"), source_renkan_guid)
        self.assertEqual(first_copy_response_dict.get("source_revision_id"), source_revision_guid)
        self.assertEqual(first_copy_response_dict.get("title", ""), "new_title_copy_1")
        self.assertEqual(first_copy_response_dict.get("content", ""), self.first_test_content)
        self.assertEqual(first_copy_response_dict.get("created_by", ""), getattr(self.user, User.USERNAME_FIELD))
        self.assertEqual(first_copy_response_dict.get("last_updated_by", ""), getattr(self.user, User.USERNAME_FIELD))
        #     in the database
        first_copy = Renkan.objects.get(renkan_guid=first_copy_response_dict.get("id", ""))
        first_copy_revision = Revision.objects.get(revision_guid=first_copy_response_dict.get("current_revision_id", ""))
        self.assertEqual(first_copy.title, "new_title_copy_1")
        self.assertEqual(first_copy.content, self.first_test_content)
        self.assertEqual(first_copy_revision.title, "new_title_copy_1")
        self.assertEqual(first_copy_revision.content, self.first_test_content)
        self.assertEqual(getattr(first_copy.creator, User.USERNAME_FIELD), getattr(self.user, User.USERNAME_FIELD))
        self.assertEqual(getattr(first_copy_revision.creator, User.USERNAME_FIELD), getattr(self.user, User.USERNAME_FIELD))
        self.assertEqual(getattr(first_copy_revision.last_updated_by, User.USERNAME_FIELD), getattr(self.user, User.USERNAME_FIELD))
        
        ###################################################
        # POSTing copy with source_renkan_guid in data
        ###################################################
        
        second_copy_data = {"source_renkan_id": source_renkan_guid, "title": "new_title_copy_2"}
        second_copy_url = post_url
        second_copy_response = self.client.post(second_copy_url, second_copy_data, format="json")
        second_copy_response_dict = json.loads(second_copy_response.content.decode())
        self.assertEqual(second_copy_response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(Renkan.objects.count(), 3)
        # Checking data
        #     in the response
        self.assertNotEqual(second_copy_response_dict.get("id"), source_renkan_guid)
        self.assertEqual(second_copy_response_dict.get("source_revision_id"), source_revision_guid)
        self.assertEqual(second_copy_response_dict.get("title", ""), "new_title_copy_2")
        self.assertEqual(second_copy_response_dict.get("content", ""), self.first_test_content)
        self.assertEqual(second_copy_response_dict.get("created_by", ""), getattr(self.user, User.USERNAME_FIELD))
        self.assertEqual(second_copy_response_dict.get("last_updated_by", ""), getattr(self.user, User.USERNAME_FIELD))
        #     in the database
        second_copy = Renkan.objects.get(renkan_guid=second_copy_response_dict.get("id", ""))
        second_copy_revision = Revision.objects.get(revision_guid=second_copy_response_dict.get("current_revision_id", ""))
        self.assertEqual(second_copy.title, "new_title_copy_2")
        self.assertEqual(second_copy.content, self.first_test_content)
        self.assertEqual(second_copy_revision.title, "new_title_copy_2")
        self.assertEqual(second_copy_revision.content, self.first_test_content)
        self.assertEqual(getattr(second_copy.creator, User.USERNAME_FIELD), getattr(self.user, User.USERNAME_FIELD))
        self.assertEqual(getattr(second_copy_revision.creator, User.USERNAME_FIELD), getattr(self.user, User.USERNAME_FIELD))
        self.assertEqual(getattr(second_copy_revision.last_updated_by, User.USERNAME_FIELD), getattr(self.user, User.USERNAME_FIELD))
        
        ###################################################
        # POSTing copy with query arg for REVISION guid
        ###################################################
        
        third_copy_data = {"title": "new_title_copy_3"}
        third_copy_url = post_url+"?source_revision_id="+source_revision_guid
        third_copy_response = self.client.post(third_copy_url, third_copy_data, format="json")
        third_copy_response_dict = json.loads(third_copy_response.content.decode())
        self.assertEqual(third_copy_response.status_code, status.HTTP_201_CREATED)  
        self.assertEqual(Renkan.objects.count(), 4)
        # Checking data
        #     in the response
        self.assertNotEqual(third_copy_response_dict.get("id"), source_renkan_guid)
        self.assertEqual(third_copy_response_dict.get("source_revision_id"), source_revision_guid)
        self.assertEqual(third_copy_response_dict.get("title", ""), "new_title_copy_3")
        self.assertEqual(third_copy_response_dict.get("content", ""), self.first_test_content)
        self.assertEqual(third_copy_response_dict.get("created_by", ""), getattr(self.user, User.USERNAME_FIELD))
        self.assertEqual(third_copy_response_dict.get("last_updated_by", ""), getattr(self.user, User.USERNAME_FIELD))
        #     in the database
        third_copy = Renkan.objects.get(renkan_guid=third_copy_response_dict.get("id", ""))
        third_copy_revision = Revision.objects.get(revision_guid=third_copy_response_dict.get("current_revision_id", ""))
        self.assertEqual(third_copy.title, "new_title_copy_3")
        self.assertEqual(third_copy.content, self.first_test_content)
        self.assertEqual(third_copy_revision.title, "new_title_copy_3")
        self.assertEqual(third_copy_revision.content, self.first_test_content)
        self.assertEqual(getattr(third_copy.creator, User.USERNAME_FIELD), getattr(self.user, User.USERNAME_FIELD))
        self.assertEqual(getattr(third_copy_revision.creator, User.USERNAME_FIELD), getattr(self.user, User.USERNAME_FIELD))
        self.assertEqual(getattr(third_copy_revision.last_updated_by, User.USERNAME_FIELD), getattr(self.user, User.USERNAME_FIELD))
        
        ###################################################
        # POSTing copy with source_revision_guid in data
        ###################################################
        
        fourth_copy_data = {"source_revision_id": source_revision_guid}
        fourth_copy_url = post_url
        fourth_copy_response = self.client.post(fourth_copy_url, fourth_copy_data, format="json")
        fourth_copy_response_dict = json.loads(fourth_copy_response.content.decode())
        self.assertEqual(fourth_copy_response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(Renkan.objects.count(), 5)
        # Checking data
        #     in the response
        self.assertNotEqual(fourth_copy_response_dict.get("id"), source_renkan_guid)
        self.assertEqual(fourth_copy_response_dict.get("source_revision_id"), source_revision_guid)
        self.assertEqual(fourth_copy_response_dict.get("title", ""), self.first_test_title)
        self.assertEqual(fourth_copy_response_dict.get("content", ""), self.first_test_content)
        self.assertEqual(fourth_copy_response_dict.get("created_by", ""), getattr(self.user, User.USERNAME_FIELD))
        self.assertEqual(fourth_copy_response_dict.get("last_updated_by", ""), getattr(self.user, User.USERNAME_FIELD))
        #     in the database
        fourth_copy = Renkan.objects.get(renkan_guid=fourth_copy_response_dict.get("id", ""))
        fourth_copy_revision = Revision.objects.get(revision_guid=fourth_copy_response_dict.get("current_revision_id", ""))
        self.assertEqual(fourth_copy.title, self.first_test_title)
        self.assertEqual(fourth_copy.content, self.first_test_content)
        self.assertEqual(fourth_copy_revision.title, self.first_test_title)
        self.assertEqual(fourth_copy_revision.content, self.first_test_content)
        self.assertEqual(getattr(fourth_copy.creator, User.USERNAME_FIELD), getattr(self.user, User.USERNAME_FIELD))
        self.assertEqual(getattr(fourth_copy_revision.creator, User.USERNAME_FIELD), getattr(self.user, User.USERNAME_FIELD))
        self.assertEqual(getattr(fourth_copy_revision.last_updated_by, User.USERNAME_FIELD), getattr(self.user, User.USERNAME_FIELD))
        
        
        ###################################################
        # try POSTing copy with bad query arg guid and with bad data guid
        ###################################################
        
        bad_copy_data = {"source_revision_id": "bleh_bad_id"}
        #    with query arg
        qarg_bad_copy_url = post_url+"?source_revision_id=bleh_bad_id"
        qarg_bad_copy_response = self.client.post(qarg_bad_copy_url, {}, format="json")
        self.assertEqual(qarg_bad_copy_response.status_code, status.HTTP_404_NOT_FOUND)
        #    with data
        data_bad_copy_url = post_url
        data_bad_copy_response = self.client.post(data_bad_copy_url, bad_copy_data, format="json")
        self.assertEqual(data_bad_copy_response.status_code, status.HTTP_404_NOT_FOUND)
        
        ###################################################
        # try POSTing with unauthorized user 
        ###################################################
        
        _ = self.client.login(username="blip", password="blip")
        post_url = reverse("v1.0:renkan_list")+"?source_revision_id="+source_revision_guid
        post_response = self.client.post(post_url, {}, format="json")
        self.assertEqual(post_response.status_code, status.HTTP_403_FORBIDDEN)
        
    def test_delete_renkan(self):
        
        ###################################################
        # POSTing new Renkan
        ###################################################
        
        post_url = reverse("v1.0:renkan_list")
        data = {"title": self.first_test_title, "content": self.first_test_content}
        post_response = self.client.post(post_url, data, format="json")
        post_response_dict = json.loads(post_response.content.decode())
        to_delete_renkan_guid = post_response_dict.get("id", "")
        to_delete_revision_guid = post_response_dict.get("current_revision_id", "")
        
        ###################################################
        # POSTing copy
        ###################################################
        data = {}
        post_response = self.client.post(post_url+"?source="+to_delete_renkan_guid, data, format="json")
        post_response_dict = json.loads(post_response.content.decode())
        copy_renkan_guid = post_response_dict.get("id", "")
        
        # Adding another revision
        put_data = {"title": self.first_test_title+"_edited!", "create_new_revision": True}
        put_url = reverse("v1.0:renkan_detail", kwargs={"renkan_guid":to_delete_renkan_guid})
        put_response = self.client.put(put_url, put_data, format="json")
        self.assertEqual(Renkan.objects.count(), 2)
        self.assertEqual(Revision.objects.count(), 3)
        
        ###################################################
        # try to DELETE with unauthorized user 
        ###################################################
        
        _ = self.client.login(username="blip", password="blip")
        delete_url = reverse("v1.0:renkan_detail", kwargs={"renkan_guid":to_delete_renkan_guid})
        delete_response = self.client.delete(delete_url, format="json")
        self.assertEqual(delete_response.status_code, status.HTTP_403_FORBIDDEN)
        
        # Restoring rightful user
        _ = self.client.login(username="blop", password="blop")
        
        ###################################################
        # DELETE initial renkan
        ###################################################
        
        delete_url = reverse("v1.0:renkan_detail", kwargs={"renkan_guid":to_delete_renkan_guid})
        delete_response = self.client.delete(delete_url, format="json")
        self.assertEqual(delete_response.status_code, status.HTTP_204_NO_CONTENT)
        self.assertEqual(Renkan.objects.count(), 1) # Only the copy remains
        self.assertEqual(Revision.objects.count(), 1) # Only the copy initial revision remains
        
        ###################################################
        # Try to GET deleted renkan
        ###################################################
        
        get_url = reverse("v1.0:renkan_detail", kwargs={"renkan_guid":to_delete_renkan_guid})
        get_response = self.client.get(get_url, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_404_NOT_FOUND)
        
        ###################################################
        # Check that remaining renkan doesn"t have reference to deleted renkan
        ###################################################
        copy_renkan = Renkan.objects.get(renkan_guid=copy_renkan_guid)
        self.assertEqual(copy_renkan.source_revision_guid, "")
        
        ###################################################
        # Try to DELETE renkan with wrong guid
        ###################################################
        
        delete_url = reverse("v1.0:renkan_detail", kwargs={"renkan_guid": "bad-id"})
        delete_response = self.client.delete(delete_url, format="json")
        self.assertEqual(delete_response.status_code, status.HTTP_404_NOT_FOUND)
    