# -*- coding: utf-8 -*-
from django import forms
from django.contrib import admin
from django.core.urlresolvers import reverse
from .models import Renkan, Revision, Workspace
from renkanmanager import settings

class RenkanAdmin(admin.ModelAdmin):
    list_display = ('renkan_guid', 'current_revision_link', 'title', 'creator_display_field', 'creation_date')
    ordering = ('-creation_date',)

    readonly_fields = ('current_revision_link', 'title', 'content', 'is_copy', 'revision_count', 'creation_date', 'renkan_guid')

    fieldsets = (
        (None, {'fields': ('renkan_guid',)}),
        ("Révision courante", {'fields': ('current_revision_link', 'title', 'content')}),
        ("Copie", {'fields': ( 'is_copy', 'source_revision',)}),
        ("Création", {'fields': ('creator', 'creation_date')}),
    )

    search_fields = ['renkan_guid', 'current_revision__title', 'creator__'+settings.RENKAN_USER_DISPLAY_FIELD]

    def creator_display_field(self, obj):
        return getattr(obj.creator, settings.RENKAN_USER_DISPLAY_FIELD) if obj.creator else None
    creator_display_field.short_description = 'creator'

    def get_queryset(self, request):
       queryset = super(RenkanAdmin, self).get_queryset(request)
       return queryset.select_related('current_revision', 'creator')

    def current_revision_link(self, obj):
        return '<a href="%s">%s</a>' % (reverse("admin:renkanmanager_revision_change", args=(obj.current_revision.id,)), obj.current_revision.revision_guid)
    current_revision_link.allow_tags = True
    current_revision_link.short_description = "Révision courante"

class RevisionAdmin(admin.ModelAdmin):
    list_display = ('revision_guid', 'parent_renkan_link', 'creator_display_field', 'title', 'modification_date')
    ordering = ('-modification_date',)
    readonly_fields = ('parent_renkan_link', 'creation_date', 'modification_date','revision_guid', 'parent_renkan')

    search_fields = ('revision_guid', 'parent_renkan__renkan_guid', 'title', 'creator__'+settings.RENKAN_USER_DISPLAY_FIELD)

    fieldsets = (
        (None, {'fields': ('revision_guid', 'parent_renkan',)}),
        ("Contenu", {'fields': ('title', 'content',)}),
        ("Création", {'fields': ('creator', 'creation_date')}),
        ("Edition", {'fields': ('last_updated_by', 'modification_date')})
    )

    def creator_display_field(self, obj):
        return getattr(obj.creator, settings.RENKAN_USER_DISPLAY_FIELD) if obj.creator else None
    creator_display_field.short_description = 'creator'


    def parent_renkan_link(self, obj):
        return '<a href="%s">%s</a>' % (reverse("admin:renkanmanager_renkan_change", args=(obj.parent_renkan.id,)), obj.parent_renkan.renkan_guid)

    parent_renkan_link.allow_tags = True
    parent_renkan_link.short_description = "Renkan associé"

    def get_queryset(self, request):
       queryset = super(RevisionAdmin, self).get_queryset(request)
       return queryset.select_related('parent_renkan', 'creator')


class WorkspaceAdmin(admin.ModelAdmin):
    pass

admin.site.register(Renkan, RenkanAdmin)
admin.site.register(Revision, RevisionAdmin)
admin.site.register(Workspace, WorkspaceAdmin)
