import json

from django.contrib.auth import get_user_model
from django.core.urlresolvers import reverse
from renkanmanager.models import Renkan, Revision
from rest_framework import status
from rest_framework.test import APITestCase


User = get_user_model()

class RevisionTests(APITestCase):
    
    def setUp(self):
        User.objects.create_superuser("blop", "blop@blop.com", "blop")
        self.user = User.objects.get(username="blop")
        _ = self.client.login(username="blop", password="blop")
        User.objects.create_user("blip", "blip@blip.com", "blip")
        self.unauthorized_user = User.objects.get(username="blip")
        
        ###################################################
        # POSTing new Renkan
        ###################################################
        
        post_url = reverse("renkan_list")
        self.test_title = "test_title_1"
        self.test_content = json.dumps({
            "id": "",
            "title": "test_title_1",
            "description": "test_description_1",
            "created": "",
            "updated": "",
            "edges": [],
            "nodes": [],
            "users": [],
            "space_id": "",
            "views": []
        })
        self.test_data = {"title": self.test_title, "content": self.test_content}
        post_response = self.client.post(post_url, self.test_data, format="json")
        post_response_dict = json.loads(post_response.content.decode())
        self.test_renkan_guid = post_response_dict.get("renkan_guid", "")
        self.test_initial_revision_guid = post_response_dict.get("current_revision_guid", "")
    
    def test_get_revision(self):
        
        ###################################################
        # GETting revision
        ###################################################
        
        get_url = reverse("revision_detail", kwargs={"renkan_guid" : self.test_renkan_guid, "revision_guid": self.test_initial_revision_guid})
        get_response = self.client.get(get_url, {}, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_200_OK)
        
        # Checking data
        get_response_dict = json.loads(get_response.content.decode()) 
        self.assertEqual(get_response_dict.get("revision_guid", ""), self.test_initial_revision_guid)
        self.assertEqual(get_response_dict.get("parent_renkan_guid", ""), self.test_renkan_guid)
        self.assertEqual(get_response_dict.get("revision_created_by", ""), self.user.username)
        self.assertEqual(get_response_dict.get("revision_last_updated_by", ""), self.user.username)
        self.assertEqual(get_response_dict.get("title", ""), self.test_data.get("title", ""))
        self.assertEqual(get_response_dict.get("content", ""), self.test_data.get("content", ""))
        
        ###################################################
        # GETting with wrong renkan guid
        ###################################################
        
        get_url = reverse("revision_detail", kwargs={"renkan_guid" : "bad-renkan-guid", "revision_guid": self.test_initial_revision_guid})
        get_response = self.client.get(get_url, {}, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_404_NOT_FOUND)
        
        ###################################################
        # GETting with wrong revision guid
        ###################################################
        
        get_url = reverse("revision_detail", kwargs={"renkan_guid" : self.test_renkan_guid, "revision_guid": "bad-revision-guid"})
        get_response = self.client.get(get_url, {}, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_404_NOT_FOUND)
        
        ###################################################
        # try GETting with unauthorized user 
        ###################################################
        
        _ = self.client.login(username="blip", password="blip")
        get_url = reverse("revision_detail", kwargs={"renkan_guid" : self.test_renkan_guid, "revision_guid": self.test_initial_revision_guid})
        get_response = self.client.get(get_url, {}, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_403_FORBIDDEN)
        
    def test_get_revision_list(self):
        
        ###################################################
        # PUTting another revision in test renkan
        ###################################################
        
        put_url = reverse("renkan_detail", kwargs={"renkan_guid" : self.test_renkan_guid})
        put_title = "edited_title"
        put_data = {
            "title" : put_title,
            "create_new_revision": True
        }
        put_response = self.client.put(put_url, put_data, format="json")
        
        ###################################################
        # GETting revision lists
        ###################################################
        
        get_url = reverse("revision_list", kwargs={"renkan_guid" : self.test_renkan_guid})
        get_response = self.client.get(get_url, {}, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_200_OK)
        
        # Checking data
        get_response_content = json.loads(get_response.content.decode()) # Should be a list
        self.assertTrue(isinstance(get_response_content, list))
        self.assertEqual(len(get_response_content), 2)
        
        ###################################################
        # GETting with wrong renkan guid
        ###################################################
        
        _ = self.client.login(username="blip", password="blip")
        get_url = reverse("revision_detail", kwargs={"renkan_guid" : self.test_renkan_guid, "revision_guid": self.test_initial_revision_guid})
        get_response = self.client.get(get_url, {}, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_403_FORBIDDEN)
    
    def test_delete_revision(self):
        
        ###################################################
        # Try to DELETE the initial revision (last revision, should 400)
        ###################################################
        delete_url = reverse("revision_detail", kwargs={"renkan_guid" : self.test_renkan_guid, "revision_guid": self.test_initial_revision_guid})
        delete_response = self.client.delete(delete_url, {}, format="json")
        self.assertEqual(delete_response.status_code, status.HTTP_400_BAD_REQUEST)
        
        ###################################################
        # PUTting another revision in test renkan
        ###################################################
        
        put_url = reverse("renkan_detail", kwargs={"renkan_guid" : self.test_renkan_guid})
        put_title = "edited_title"
        put_data = {
            "title" : put_title,
            "create_new_revision": True
        }
        put_response = self.client.put(put_url, put_data, format="json")
        put_response_dict = json.loads(put_response.content.decode())
        added_revision_guid = put_response_dict.get("current_revision_guid", "")
        
        ###################################################
        # Try to DELETE the added revision (current revision, should 400)
        ###################################################
        
        delete_url = reverse("revision_detail", kwargs={"renkan_guid" : self.test_renkan_guid, "revision_guid": added_revision_guid})
        delete_response = self.client.delete(delete_url, {}, format="json")
        self.assertEqual(delete_response.status_code, status.HTTP_400_BAD_REQUEST)
        
        
        ###################################################
        # try to DELETE with unauthorized user 
        ###################################################
        
        _ = self.client.login(username="blip", password="blip")
        delete_url = reverse("revision_detail", kwargs={"renkan_guid" : self.test_renkan_guid, "revision_guid": self.test_initial_revision_guid})
        delete_response = self.client.delete(delete_url, {}, format="json")
        self.assertEqual(delete_response.status_code, status.HTTP_403_FORBIDDEN)
        
        # Restoring rightful user
        _ = self.client.login(username="blop", password="blop")
        
        ###################################################
        # Try to DELETE the initial revision (should 204 now that we added a revision)
        ###################################################
        
        delete_url = reverse("revision_detail", kwargs={"renkan_guid" : self.test_renkan_guid, "revision_guid": self.test_initial_revision_guid})
        delete_response = self.client.delete(delete_url, {}, format="json")
        self.assertEqual(delete_response.status_code, status.HTTP_204_NO_CONTENT)
        
        ###################################################
        # Try to DELETE with wrong renkan guid
        ###################################################
        
        delete_url = reverse("revision_detail", kwargs={"renkan_guid" : "bad-renkan-guid", "revision_guid": self.test_initial_revision_guid})
        delete_response = self.client.delete(delete_url, {}, format="json")
        self.assertEqual(delete_response.status_code, status.HTTP_404_NOT_FOUND)
        
        ###################################################
        # Try to DELETE with wrong revision guid
        ###################################################
        
        delete_url = reverse("revision_detail", kwargs={"renkan_guid" : self.test_renkan_guid, "revision_guid": "bad-revision-guid"})
        delete_response = self.client.delete(delete_url, {}, format="json")
        self.assertEqual(delete_response.status_code, status.HTTP_404_NOT_FOUND)