package org.iri_research.renkan.controller;

import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;

import org.iri_research.renkan.Constants;
import org.iri_research.renkan.RenkanException;
import org.iri_research.renkan.forms.SpaceForm;
import org.iri_research.renkan.forms.SpaceFormValidator;
import org.iri_research.renkan.models.Space;
import org.iri_research.renkan.repositories.SpacesRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort.Direction;
import org.springframework.data.web.PageableDefaults;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.client.HttpClientErrorException;

@Controller
@RequestMapping("/admin")
public class AdminController {

	private final Logger logger = LoggerFactory.getLogger(AdminController.class);
	
	
	@Autowired
	private SpacesRepository spacesRepository;
	
    @InitBinder(value={"space"})
    protected void initBinder(WebDataBinder binder) {
        binder.setValidator(new SpaceFormValidator());
    }
	
	@RequestMapping(value="", method = RequestMethod.GET, produces={"text/html;charset=UTF-8"})
	public String adminIndex() {
				
		return "admin/adminIndex";
	}
	
	@RequestMapping(value="/spaces", method = RequestMethod.GET, produces={"text/html;charset=UTF-8"})
	public String spacesList(
			Model model,
			@PageableDefaults(sort={"created"}, sortDir=Direction.DESC, pageNumber=0, value=Constants.PAGINATION_SIZE) Pageable p,
			HttpServletRequest request) {
		
		Page<Space> page = this.spacesRepository.findAll(p);
		
		model.addAttribute("page", page);
		model.addAttribute("baseUrl", Utils.buildBaseUrl(request));
		
		return "admin/spacesList";
	}
	
	@RequestMapping(value="/spaces/edit/", method = RequestMethod.GET, produces={"text/html;charset=UTF-8"})
	public String editSpace(Model model) {
		return editSpace(model, null);
	}
	
	@RequestMapping(value="/spaces/edit/{spaceId}", method = RequestMethod.GET, produces={"text/html;charset=UTF-8"})
	public String editSpace(Model model, @PathVariable(value="spaceId") String spaceId) {
		
		SpaceForm spaceForm = null;
		
		if(spaceId == null || spaceId.length() == 0 || "_".equals(spaceId)) {
			spaceForm = new SpaceForm();
		}
		else {
			Space space = this.spacesRepository.findOne(spaceId);		
			if(space == null) {
				throw new HttpClientErrorException(HttpStatus.NOT_FOUND, "space " + spaceId + " not found");			
			}
			spaceForm = new SpaceForm(space);
		}
		
		model.addAttribute("space", spaceForm);
		
		return "admin/spaceEdit";
	}
	
	
	@RequestMapping(value="/spaces/save", method = RequestMethod.POST)
	public String saveSpace(Model model, @ModelAttribute("space") @Valid SpaceForm spaceForm, BindingResult bindingResult) {

		logger.debug("space title " + spaceForm.getTitle());
		logger.debug("space description " + spaceForm.getDescription());
		
		if(bindingResult.hasErrors()) {
			return "admin/spaceEdit";
		}
		
		spaceForm.setSpacesRepository(spacesRepository);
		
		try {
			spaceForm.save();
		} catch (RenkanException e) {
			throw new HttpClientErrorException(HttpStatus.NOT_FOUND, "space " + spaceForm.getId() + " not found");
		}
		
		return "redirect:";
	}
	
}
