package org.iri_research.renkan.test.controller;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.iri_research.renkan.models.Space;
import org.iri_research.renkan.repositories.SpacesRepository;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;
import org.springframework.test.context.web.WebAppConfiguration;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.MvcResult;
import org.springframework.test.web.servlet.request.MockHttpServletRequestBuilder;
import org.springframework.test.web.servlet.request.MockMvcRequestBuilders;
import org.springframework.test.web.servlet.result.MockMvcResultMatchers;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;
import org.springframework.web.context.WebApplicationContext;

@RunWith(SpringJUnit4ClassRunner.class)
@WebAppConfiguration
@ContextConfiguration(locations={"controller-context.xml", "file:src/main/webapp/WEB-INF/spring-servlet.xml"})
public class AdminControllerTest {
	
	private final static int SPACE_NB = 3;
	
	private Logger logger = LoggerFactory.getLogger(AdminControllerTest.class);
	
	@Autowired
	private SpacesRepository spacesRepository;
	
	private Map<String, Space> spacesList = new HashMap<String, Space>(SPACE_NB);
	private List<String> spacesUuids = new ArrayList<>(SPACE_NB);
	
	@Autowired
	private WebApplicationContext context;
	private MockMvc mvc;
	
	
	@Before
	public void setup() {
		
		logger.debug("Setup");
		spacesRepository.deleteAll();
		for(int i=0;i<SPACE_NB;i++) {
			Date creationDate = new Date();
			String uuid = UUID.randomUUID().toString();
			spacesUuids.add(uuid);
			Space testSpace = new Space(uuid, "test " + i, "Test space " + 1, "{}", "http://ldt.iri.centrepompidou.fr", "#ababab", "test_user", "http://ldt.iri.centrepompidou.fr", creationDate);
			testSpace = spacesRepository.save(testSpace);
			this.spacesList.put(uuid, testSpace);
			try {
				Thread.sleep(1);
			} catch (InterruptedException e) {
				e.printStackTrace();
			}
		}
		
		this.mvc = MockMvcBuilders.webAppContextSetup(this.context).build();
	}
	
	@Test
	public void testContext() throws Exception {
		MockHttpServletRequestBuilder get = MockMvcRequestBuilders.get("/");
		MvcResult res = this.mvc.perform(get)
				.andExpect(MockMvcResultMatchers.status().isOk())
				.andReturn();
		
		logger.debug("testContext resp : " + res.getResponse().getContentAsString());
		
	}
	
	@Test
	public void testSpacePostUpdate() throws Exception {
		
		MockHttpServletRequestBuilder post = MockMvcRequestBuilders.post("/admin/spaces/save");
		post = post.param("id", this.spacesUuids.get(0));
		post = post.param("title", "New title");
		post = post.param("description", "New description");
		post = post.param("uri", "http://ldt.iri.centrepompidou.fr/new/uri");
		post = post.param("color", "#ffffff");
		post = post.param("binConfig", "{}");
		
		this.mvc.perform(post)
				.andExpect(MockMvcResultMatchers.status().isMovedTemporarily())
				.andExpect(MockMvcResultMatchers.redirectedUrl(""));
		
		Space sp = this.spacesRepository.findOne(this.spacesUuids.get(0));
		
		Assert.assertNotNull("Should find space", sp);
		Assert.assertEquals("Title equals", "New title", sp.getTitle());
		Assert.assertEquals("Description equals", "New description", sp.getDescription());
		Assert.assertEquals("Uri equals", "http://ldt.iri.centrepompidou.fr/new/uri", sp.getUri());
		Assert.assertEquals("Color equals", "#ffffff", sp.getColor());
		Assert.assertEquals("BinConfig equals", "{}", sp.getBinConfig());
		
	}

	@Test
	public void testSpacePostCreate() throws Exception {
		
		MockHttpServletRequestBuilder post = MockMvcRequestBuilders.post("/admin/spaces/save");
		post = post.param("title", "New title");
		post = post.param("description", "New description");
		post = post.param("uri", "http://ldt.iri.centrepompidou.fr/new/uri");
		post = post.param("color", "#ffffff");
		post = post.param("binConfig", "{}");
		
		this.mvc.perform(post)
				.andExpect(MockMvcResultMatchers.status().isMovedTemporarily())
				.andExpect(MockMvcResultMatchers.redirectedUrl(""));
		
		
		Assert.assertEquals("Must have one more space", SPACE_NB + 1, this.spacesRepository.count());
		
		for (Space sp : this.spacesRepository.findAll()) {
			if(this.spacesList.containsKey(sp.getId())) {
				continue;
			}
			else {
				Assert.assertNotNull("Should find space", sp);
				Assert.assertEquals("Title equals", "New title", sp.getTitle());
				Assert.assertEquals("Description equals", "New description", sp.getDescription());
				Assert.assertEquals("Uri equals", "http://ldt.iri.centrepompidou.fr/new/uri", sp.getUri());
				Assert.assertEquals("Color equals", "#ffffff", sp.getColor());
				Assert.assertEquals("BinConfig equals", "{}", sp.getBinConfig());
			}
		}
	}
	
	@Test
	public void testSpacePostUpdateEmptyTitle() throws Exception {
		
		MockHttpServletRequestBuilder post = MockMvcRequestBuilders.post("/admin/spaces/save");
		post = post.param("id", this.spacesUuids.get(0));
		post = post.param("title", "");
		post = post.param("description", "New description");
		post = post.param("uri", "http://ldt.iri.centrepompidou.fr/new/uri");
		post = post.param("color", "#ffffff");
		post = post.param("binConfig", "{}");
		
		this.mvc.perform(post)
				.andExpect(MockMvcResultMatchers.status().isOk())
				.andExpect(MockMvcResultMatchers.view().name("admin/spaceEdit"))
				.andExpect(MockMvcResultMatchers.model().hasErrors())
				.andExpect(MockMvcResultMatchers.model().errorCount(1))
				.andExpect(MockMvcResultMatchers.model().attributeHasErrors("space"))
				.andExpect(MockMvcResultMatchers.model().attributeHasFieldErrors("space", "title"));
		
		Space sp = this.spacesRepository.findOne(this.spacesUuids.get(0));
		
		Assert.assertNotNull("Should find space", sp);
		Assert.assertEquals("Title equals", "test 0", sp.getTitle());
		
	}
	
	@Test
	public void testSpacePostCreateEmptyTitle() throws Exception {
		
		MockHttpServletRequestBuilder post = MockMvcRequestBuilders.post("/admin/spaces/save");
		post = post.param("title", "");
		post = post.param("description", "New description");
		post = post.param("uri", "http://ldt.iri.centrepompidou.fr/new/uri");
		post = post.param("color", "#ffffff");
		post = post.param("binConfig", "{}");
		
		this.mvc.perform(post)
				.andExpect(MockMvcResultMatchers.status().isOk())
				.andExpect(MockMvcResultMatchers.view().name("admin/spaceNew"))
				.andExpect(MockMvcResultMatchers.model().hasErrors())
				.andExpect(MockMvcResultMatchers.model().errorCount(1))
				.andExpect(MockMvcResultMatchers.model().attributeHasErrors("space"))
				.andExpect(MockMvcResultMatchers.model().attributeHasFieldErrors("space", "title"));
				
		Assert.assertEquals("Must not have one more space", SPACE_NB, this.spacesRepository.count());
		
	}

	@Test
	public void testSpacePostUpdateBadJson() throws Exception {
		
		MockHttpServletRequestBuilder post = MockMvcRequestBuilders.post("/admin/spaces/save");
		post = post.param("id", this.spacesUuids.get(0));
		post = post.param("title", "New Title");
		post = post.param("description", "New description");
		post = post.param("uri", "http://ldt.iri.centrepompidou.fr/new/uri");
		post = post.param("color", "#ffffff");
		post = post.param("binConfig", "{");
		
		this.mvc.perform(post)
				.andExpect(MockMvcResultMatchers.status().isOk())
				.andExpect(MockMvcResultMatchers.view().name("admin/spaceEdit"))
				.andExpect(MockMvcResultMatchers.model().hasErrors())
				.andExpect(MockMvcResultMatchers.model().errorCount(1))
				.andExpect(MockMvcResultMatchers.model().attributeHasErrors("space"))
				.andExpect(MockMvcResultMatchers.model().attributeHasFieldErrors("space", "binConfig"));
		
		Space sp = this.spacesRepository.findOne(this.spacesUuids.get(0));
		
		Assert.assertNotNull("Should find space", sp);
		Assert.assertEquals("Bin config equals", "{}", sp.getBinConfig());
		
	}
	
	@Test
	public void testSpacePostCreateBadJson() throws Exception {
		
		MockHttpServletRequestBuilder post = MockMvcRequestBuilders.post("/admin/spaces/save");
		post = post.param("title", "New Title");
		post = post.param("description", "New description");
		post = post.param("uri", "http://ldt.iri.centrepompidou.fr/new/uri");
		post = post.param("color", "#ffffff");
		post = post.param("binConfig", "}");
		
		this.mvc.perform(post)
				.andExpect(MockMvcResultMatchers.status().isOk())
				.andExpect(MockMvcResultMatchers.view().name("admin/spaceNew"))
				.andExpect(MockMvcResultMatchers.model().hasErrors())
				.andExpect(MockMvcResultMatchers.model().errorCount(1))
				.andExpect(MockMvcResultMatchers.model().attributeHasErrors("space"))
				.andExpect(MockMvcResultMatchers.model().attributeHasFieldErrors("space", "binConfig"));
				
		Assert.assertEquals("Must not have one more space", SPACE_NB, this.spacesRepository.count());
		
	}

	@Test
	public void testSpacePostUpdateAllErrors() throws Exception {
		
		MockHttpServletRequestBuilder post = MockMvcRequestBuilders.post("/admin/spaces/save");
		post = post.param("id", this.spacesUuids.get(0));
		post = post.param("title", "");
		post = post.param("description", "New description");
		post = post.param("uri", "http://ldt.iri.centrepompidou.fr/new/uri");
		post = post.param("color", "#ffffff");
		post = post.param("binConfig", "{");
		
		this.mvc.perform(post)
				.andExpect(MockMvcResultMatchers.status().isOk())
				.andExpect(MockMvcResultMatchers.view().name("admin/spaceEdit"))
				.andExpect(MockMvcResultMatchers.model().hasErrors())
				.andExpect(MockMvcResultMatchers.model().errorCount(2))
				.andExpect(MockMvcResultMatchers.model().attributeHasErrors("space"))
				.andExpect(MockMvcResultMatchers.model().attributeHasFieldErrors("space", "title"))
				.andExpect(MockMvcResultMatchers.model().attributeHasFieldErrors("space", "binConfig"));
		
		Space sp = this.spacesRepository.findOne(this.spacesUuids.get(0));
		
		Assert.assertNotNull("Should find space", sp);
		Assert.assertEquals("Bin config equals", "{}", sp.getBinConfig());
		Assert.assertEquals("title equals", "test 0", sp.getTitle());
		
	}
	
	@Test
	public void testSpacePostCreateAllErrors() throws Exception {
		
		MockHttpServletRequestBuilder post = MockMvcRequestBuilders.post("/admin/spaces/save");
		post = post.param("title", "");
		post = post.param("description", "New description");
		post = post.param("uri", "http://ldt.iri.centrepompidou.fr/new/uri");
		post = post.param("color", "#ffffff");
		post = post.param("binConfig", "}");
		
		this.mvc.perform(post)
				.andExpect(MockMvcResultMatchers.status().isOk())
				.andExpect(MockMvcResultMatchers.view().name("admin/spaceNew"))
				.andExpect(MockMvcResultMatchers.model().hasErrors())
				.andExpect(MockMvcResultMatchers.model().errorCount(2))
				.andExpect(MockMvcResultMatchers.model().attributeHasErrors("space"))
				.andExpect(MockMvcResultMatchers.model().attributeHasFieldErrors("space", "title"))
				.andExpect(MockMvcResultMatchers.model().attributeHasFieldErrors("space", "binConfig"));
				
		Assert.assertEquals("Must not have one more space", SPACE_NB, this.spacesRepository.count());
		
	}
	
	
	@After
	public void teardown() {
		spacesRepository.deleteAll();
	}

}
