package org.iri_research.renkan.coweb.event;

import java.util.Map;

import javax.inject.Inject;

import org.iri_research.renkan.RenkanException;
import org.iri_research.renkan.models.Project;
import org.iri_research.renkan.models.ProjectSync;
import org.iri_research.renkan.repositories.ProjectSyncsRepository;
import org.iri_research.renkan.repositories.ProjectsRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public abstract class AbstractSyncEventManager implements ISyncEventManager {

	private final Logger logger = LoggerFactory.getLogger(AbstractSyncEventManager.class);
	
	@Inject
	private ProjectsRepository projectsRepository;
	
	@Inject
	private ProjectSyncsRepository projectSyncsRepository;
	
	@Override
	public ProjectsRepository getProjectsRepository() {
		return this.projectsRepository;
	}
		
	@Override
	public boolean dispatchEvent(Map<String, Object> data) {
		
		this.saveSyncEvent(data);

		String eventType = (String) data.get("type");
		
		if("null".equalsIgnoreCase(eventType)) {
			return this.nullOperation(data);
		}
		else if ("update".equalsIgnoreCase(eventType)) {
			return this.update(data);
		}
		else if ("insert".equalsIgnoreCase(eventType)) {
			return this.insert(data);
		}
		else if("delete".equalsIgnoreCase(eventType)) {
			return this.delete(data);
		}
		
		logger.warn(String.format("dispatchEvent : eventType unknown %s", eventType));
		return false;
	}

	private void saveSyncEvent(Map<String, Object> data) {
		
		String project_id = null;
	
		@SuppressWarnings("unchecked")
		Map<String, Object> values = (Map<String, Object>) data.get("value");
		
		if(values != null) {
			project_id = (String) values.get("project_id");
		}
		
		if (project_id == null || "".equals(project_id)) {
			logger.warn("saveSyncEvent : project id is null. Can not save sync event");
			return;
		}
		
		Project p = this.projectsRepository.findOne(project_id);
		
		if (p == null) {
			logger.warn("saveSyncEvent : project not found. Can not save sync event");
			return;
		}
		

		try {
			ProjectSync ps = this.projectSyncsRepository.getProjectSync(data.toString(), p);
			this.projectSyncsRepository.save(ps);
		} catch (RenkanException e) {
			logger.warn("saveSyncEvent : Error when getting Projectr syn object",e);
		}		
		
	}

	@Override
	public abstract boolean update(Map<String, Object> data);

	@Override
	public abstract boolean insert(Map<String, Object> data);

	@Override
	public abstract boolean delete(Map<String, Object> data);

	@Override
	public abstract boolean nullOperation(Map<String, Object> data);

}
