package org.iri_research.renkan.rest;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.UUID;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.UriInfo;

import org.iri_research.renkan.models.Project;
import org.iri_research.renkan.repositories.ProjectsRepository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.mongodb.BasicDBObject;
import com.mongodb.DBCollection;
import com.mongodb.DBCursor;
import com.mongodb.DBObject;
import com.mongodb.util.JSON;
import com.sun.jersey.spi.resource.Singleton;


@Singleton
@Path("projects")
@Component
public class ProjectsResource {
	
	private Logger logger = LoggerFactory.getLogger(ProjectsResource.class);
	
	@Autowired
	private ProjectsRepository projectRepository;
	
	@Context
	private UriInfo uriInfo;
		
	private DBCollection getCollection() {		
		return projectRepository.getCollection();
	}

	@GET
	@Path("{id : [a-zA-Z\\-0-9]+}")
	@Produces(MediaType.APPLICATION_JSON + ";charset=utf-8")
	public Project getProject(@PathParam("id") String projectId) {
		
		this.logger.debug("GetProject: " + projectId);
				
		Project project = this.projectRepository.findOne(projectId);
		
		if (null == project) {
			throw new WebApplicationException(Status.NOT_FOUND);
		}
		
		return project;				
	}
	
	@DELETE
	@Path("{id : [a-zA-Z\\-0-9]+}")
	@Produces(MediaType.TEXT_PLAIN + ";charset=utf-8")
	public Response deleteProject(@PathParam("id") String projectId) {
		
		this.logger.debug("DeleteProject : id " + projectId);
		
		this.projectRepository.delete(projectId);
		return Response.ok(this.uriInfo.getAbsolutePathBuilder().segment(projectId).build().toString() + " deleted").build();
		
	}


	/**
	 * test: curl -i -X PUT -H 'Content-Type: application/json' -d @test-data.json  http://localhost:8080/renkan/rest/projects/12eff140-e65c-11e1-aff1-0800200c9a66
	 * @param projectId
	 * @param projectContent
	 */
	@PUT
	@Path("{id : [a-zA-Z\\-0-9]+}")
	@Consumes(MediaType.APPLICATION_JSON + ";charset=utf-8")
	public Response putProject(@PathParam("id") String projectId, Project project) {

		if(!projectId.equals(project.getId())) {
			throw new WebApplicationException(Response.status(Status.BAD_REQUEST).entity("Id parameter and id in JSON do not match").build());
		}
		if(project.getCreated() == null) {
			project.setCreated(new Date());
		}
		this.projectRepository.save(project);
		return Response.noContent().build();
		
	}

	
	@POST
	@Consumes(MediaType.APPLICATION_JSON + ";charset=utf-8")
	public Response postProject(Project project) {
		if(project.getId() != null) {
			throw new WebApplicationException(Response.status(Status.BAD_REQUEST).entity("Id in JSON must be null").build());
		}
		
		project.setId(UUID.randomUUID().toString());
		if(project.getCreated() == null) {
			project.setCreated(new Date());
		}
		project = this.projectRepository.save(project);
		return Response.created(this.uriInfo.getAbsolutePathBuilder().segment(project.getId()).build()).entity(project).build();		
	}
		
	@GET
	@Produces(MediaType.APPLICATION_JSON + ";charset=utf-8")
	public String getProjectList() {
		
		DBCollection projectCollection = this.getCollection();
		BasicDBObject keys = new BasicDBObject();
		keys.put("description", 1);
		keys.put("title", 1);
		keys.put("uri", 1);
		keys.put("created", 1);
		DBCursor cursor = projectCollection.find(new BasicDBObject(), keys);
		
		List<DBObject> res = new ArrayList<DBObject>();
				
		try {
			while(cursor.hasNext()) {
				DBObject obj = cursor.next();
				DBObject links = new BasicDBObject();
				
				DBObject linkdef = new BasicDBObject();
				linkdef.put("href", this.uriInfo.getAbsolutePathBuilder().path(obj.get("_id").toString()).build().toString());
				linkdef.put("method", "get");
				linkdef.put("produces", MediaType.APPLICATION_JSON + ";charset=utf-8");				
				links.put("view", linkdef);
				
				linkdef = new BasicDBObject();
				linkdef.put("href", this.uriInfo.getAbsolutePathBuilder().path(obj.get("_id").toString()).build().toString());
				linkdef.put("method", "put");
				linkdef.put("consumes", MediaType.APPLICATION_JSON + ";charset=utf-8");				
				links.put("update", linkdef);

				linkdef = new BasicDBObject();
				linkdef.put("href", this.uriInfo.getAbsolutePathBuilder().path(obj.get("_id").toString()).build().toString());
				linkdef.put("method", "delete");
				links.put("delete", linkdef);				
				
				obj.put("__links",  links);
				res.add(obj);
			}
		}
		finally {
			cursor.close();
		}
		
		return JSON.serialize(res);
	}
		
}
