import json, logging, uuid, datetime, time

from django.db import transaction
from django.contrib.auth import get_user_model
from django.conf import settings
from django.utils import six
from django.core.exceptions import ValidationError
from renkanmanager import settings as renkan_settings
from renkanmanager.models import Renkan, Workspace, Revision
from rest_framework import serializers

logger = logging.getLogger(__name__)
RENKAN_USER_DISPLAY_FIELD = renkan_settings.RENKAN_USER_DISPLAY_FIELD

class RenkanSerializer(serializers.Serializer):
    id = serializers.ReadOnlyField(source="renkan_guid")
    current_revision_id = serializers.SlugRelatedField(source="current_revision", slug_field="revision_guid", read_only=True)
    workspace_id = serializers.SlugRelatedField(source="workspace", slug_field="workspace_guid", queryset=Workspace.objects, required=False)
    source_revision_id = serializers.SlugRelatedField(source="source_revision", slug_field="revision_guid", queryset=Revision.objects, required=False)
    revision_count = serializers.ReadOnlyField()
    created_by = serializers.SlugRelatedField(source="creator", slug_field=RENKAN_USER_DISPLAY_FIELD, read_only=True)
    last_updated_by = serializers.SerializerMethodField("get_current_revision_last_updator")
    title = serializers.CharField(required=False)
    content = serializers.JSONField(required=False)
    creation_date = serializers.ReadOnlyField()
    modification_date = serializers.SerializerMethodField()

    def __init__(self, instance=None, data=serializers.empty, **kwargs):

        self.validation_timestamp = kwargs.pop('validation_timestamp', None)
        super(RenkanSerializer, self).__init__(instance, data, **kwargs)


    def get_current_revision_last_updator(self, renkan):
        last_updator = renkan.current_revision.last_updated_by
        if last_updator:
            return getattr(last_updator, RENKAN_USER_DISPLAY_FIELD)
        else:
            return ''


    def get_modification_date(self, renkan):
        return renkan.current_revision.modification_date

    @transaction.atomic
    def create(self, validated_data):
        """
            Method to create a new Renkan (and its first revision)
        """
        logger.debug("RENKANSERIALIZER %r", validated_data)
        creator = validated_data.get('creator')
        workspace_obj = validated_data.get('workspace', None)
        source_revision_obj = validated_data.get('source_revision', None)
        try:
            return Renkan.objects.create_renkan(
                creator=creator,
                title=validated_data.get('title', ''),
                content=validated_data.get('content', ''),
                workspace=workspace_obj,
                source_revision=source_revision_obj
            )
        except ValidationError as ve:
            raise serializers.ValidationError(ve.args[0])

    @transaction.atomic
    def update(self, renkan, validated_data):
        """
            Method to update a Renkan object. Creates a new revision if needed
        """
        logger.debug('RENKAN SERIALIZER UPDATE %r', validated_data)
        updator = validated_data.get('updator')
        create_new_revision = validated_data.get("create_new_revision", False)
        title = validated_data.get('title', renkan.current_revision.title)
        content = validated_data.get('content', '')
        if not content:
            content = renkan.current_revision.content
        if title != json.loads(content).get("title", ""):
            content_dict = json.loads(content)
            content_dict["title"] = title
            content = json.dumps(content_dict)
        try:
            renkan.save_renkan(updator=updator, timestamp=self.validation_timestamp, title=title, content=content, create_new_revision=create_new_revision)
        except ValidationError as ve:
            raise serializers.ValidationError(str(ve.args[0]))
        # FORCE Renkan reload.
        # TODO: How to avoid the reload ???
        return Renkan.objects.get(id=renkan.id)

    def validate_workspace_id(self, value):
        if self.instance is not None:
            raise serializers.ValidationError("You cannot update workspace_guid")
        return value

    def validate_source_revision_id(self, value):
        if self.instance is not None:
            raise serializers.ValidationError("You cannot update source_revision_guid")
        return value

    def validate_content(self, value):
        try:
            loaded_json = json.loads(value)
        except ValueError:
            raise serializers.ValidationError("Content format is not a JSON-serializable")
        if (not "nodes" in loaded_json):
            raise serializers.ValidationError("Content requires a 'nodes' entry")
        if (not "edges" in loaded_json):
            raise serializers.ValidationError("Content requires a 'edges' entry")
        if (not "views" in loaded_json):
            raise serializers.ValidationError("Content requires a 'views' entry")
        return value

    def validate(self, data):
        if self.instance:
            validation_timestamp = self.validation_timestamp or data.get('content', {}).get('updated')
            if not validation_timestamp:
                raise serializers.ValidationError("No validation timestamp and no content json to extract it from")
            validation_timestamp = dateparse.parse_datetime(validation_timestamp) if isinstance(validation_timestamp, six.string_types) else validation_timestamp
            if validation_timestamp != self.instance.current_revision.modification_date:
                raise serializers.ValidationError("Invalid timestamp was provided")
        return data


class RevisionSerializer(serializers.Serializer):
    id = serializers.ReadOnlyField(source="revision_guid")
    parent_renkan_id = serializers.SlugRelatedField(source="parent_renkan", slug_field='renkan_guid', read_only=True)
    workspace_id = serializers.SerializerMethodField("get_related_workspace_guid")
    title = serializers.ReadOnlyField()
    content = serializers.JSONField(read_only=True)
    renkan_created_by = serializers.SerializerMethodField("get_related_renkan_creator")
    renkan_creation_date = serializers.SerializerMethodField("get_related_renkan_creation_date")
    revision_created_by = serializers.SlugRelatedField(source="creator", slug_field=RENKAN_USER_DISPLAY_FIELD, read_only=True)
    revision_last_updated_by = serializers.SlugRelatedField(source="last_updated_by", slug_field=RENKAN_USER_DISPLAY_FIELD, read_only=True)
    revision_modification_date = serializers.ReadOnlyField(source="modification_date")

    def get_related_workspace_guid(self, revision):
        if revision.parent_renkan.workspace:
            return revision.parent_renkan.workspace.workspace_guid
        else:
            return ''

    def get_related_renkan_creator(self, revision):
        return getattr(revision.parent_renkan.creator, RENKAN_USER_DISPLAY_FIELD)

    def get_related_renkan_creation_date(self, revision):
        return revision.parent_renkan.creation_date


class WorkspaceSerializer(serializers.Serializer):
    id = serializers.ReadOnlyField(source="workspace_guid")
    workspace_created_by = serializers.SlugRelatedField(source="creator", slug_field=RENKAN_USER_DISPLAY_FIELD, read_only=True)
    creation_date = serializers.ReadOnlyField()
    renkan_count = serializers.ReadOnlyField()
    title = serializers.CharField()

    def create(self, validated_data):
        creator = validated_data.get('creator')
        title = validated_data.get('title', '')
        workspace = Workspace.objects.create(creator=creator, title=title)
        workspace.save()
        return workspace

    def update(self, workspace, validated_data):
        workspace.title = validated_data.get('title', '')
        workspace.save()
        return workspace
