import json

from django.contrib.auth import get_user_model
from django.core.urlresolvers import reverse
from renkanmanager.models import Renkan, Revision
from rest_framework import status
from rest_framework.test import APITestCase


User = get_user_model()

class RevisionTests(APITestCase):

    def setUp(self):
        User.objects.create_superuser("blop", "blop@blop.com", "blop")
        self.user = User.objects.get_by_natural_key(username="blop")
        _ = self.client.login(username="blop", password="blop")

        ###################################################
        # POSTing new Renkan
        ###################################################

        post_url = reverse("v1.0:renkan_list")
        self.test_title = "test_title_1"
        test_content_dict_init = {
            "id": "",
            "title": "test_title_1",
            "description": "",
            "created": "2016-03-11 15:10:10.645193+00:00",
            "updated": "2016-03-11 15:10:10.645193+00:00",
            "edges": [],
            "nodes": [],
            "users": [],
            "space_id": "",
            "views": []
        }
        self.test_content = json.dumps(test_content_dict_init)
        self.test_data = {"title": self.test_title, "content": self.test_content}
        post_response = self.client.post(post_url, self.test_data, format="json")
        post_response_dict = json.loads(post_response.content.decode())
        self.test_content_dict = json.loads(post_response_dict.get("content", "{}"))
        self.post_response_content_dict = json.loads(post_response_dict.get("content", "{}"))
        self.test_renkan_guid = post_response_dict.get("id", "")
        self.test_initial_revision_guid = post_response_dict.get("current_revision_id", "")

    def test_get_revision(self):

        ###################################################
        # GETting revision
        ###################################################

        get_url = reverse("v1.0:revision_detail", kwargs={"renkan_guid" : self.test_renkan_guid, "revision_guid": self.test_initial_revision_guid})
        get_response = self.client.get(get_url, {}, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_200_OK)

        # Checking data
        get_response_dict = json.loads(get_response.content.decode())
        self.assertEqual(get_response_dict.get("id", ""), self.test_initial_revision_guid)
        self.assertEqual(get_response_dict.get("parent_renkan_id", ""), self.test_renkan_guid)
        self.assertEqual(get_response_dict.get("revision_created_by", ""), getattr(self.user, User.USERNAME_FIELD))
        self.assertEqual(get_response_dict.get("revision_last_updated_by", ""), getattr(self.user, User.USERNAME_FIELD))
        self.assertEqual(get_response_dict.get("title", ""), self.test_data.get("title", ""))

        ###################################################
        # GETting with wrong renkan guid
        ###################################################

        get_url = reverse("v1.0:revision_detail", kwargs={"renkan_guid" : "bad-renkan-guid", "revision_guid": self.test_initial_revision_guid})
        get_response = self.client.get(get_url, {}, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_404_NOT_FOUND)

        ###################################################
        # GETting with wrong revision guid
        ###################################################

        get_url = reverse("v1.0:revision_detail", kwargs={"renkan_guid" : self.test_renkan_guid, "revision_guid": "bad-revision-guid"})
        get_response = self.client.get(get_url, {}, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_404_NOT_FOUND)

    def test_get_revision_list(self):

        ###################################################
        # PUTting another revision in test renkan
        ###################################################

        put_url = reverse("v1.0:renkan_detail", kwargs={"renkan_guid" : self.test_renkan_guid})
        put_title = "edited_title"
        put_data = {
            "title" : put_title,
            "validation_timestamp": self.post_response_content_dict.get("updated", ""),
            "create_new_revision": True
        }
        put_response = self.client.put(put_url, put_data, format="json")

        ###################################################
        # GETting revision lists
        ###################################################

        get_url = reverse("v1.0:revision_list", kwargs={"renkan_guid" : self.test_renkan_guid})
        get_response = self.client.get(get_url, {}, format="json")
        self.assertEqual(get_response.status_code, status.HTTP_200_OK)

        # Checking data
        get_response_content = json.loads(get_response.content.decode()) # Should be a list
        self.assertTrue(isinstance(get_response_content, list))
        self.assertEqual(len(get_response_content), 2)


    def test_delete_revision(self):

        ###################################################
        # Try to DELETE the initial revision (last revision, should 400)
        ###################################################
        delete_url = reverse("v1.0:revision_detail", kwargs={"renkan_guid" : self.test_renkan_guid, "revision_guid": self.test_initial_revision_guid})
        delete_response = self.client.delete(delete_url, {}, format="json")
        self.assertEqual(delete_response.status_code, status.HTTP_400_BAD_REQUEST)

        ###################################################
        # PUTting another revision in test renkan
        ###################################################

        put_url = reverse("v1.0:renkan_detail", kwargs={"renkan_guid" : self.test_renkan_guid})
        put_title = "edited_title"
        put_data = {
            "title" : put_title,
            "validation_timestamp": self.post_response_content_dict.get("updated", ""),
            "create_new_revision": True
        }
        put_response = self.client.put(put_url, put_data, format="json")
        put_response_dict = json.loads(put_response.content.decode())
        added_revision_guid = put_response_dict.get("current_revision_id", "")

        ###################################################
        # Try to DELETE the added revision (current revision, should 400)
        ###################################################

        delete_url = reverse("v1.0:revision_detail", kwargs={"renkan_guid" : self.test_renkan_guid, "revision_guid": added_revision_guid})
        delete_response = self.client.delete(delete_url, {}, format="json")
        self.assertEqual(delete_response.status_code, status.HTTP_400_BAD_REQUEST)

        # Restoring rightful user
        _ = self.client.login(username="blop", password="blop")

        ###################################################
        # Try to DELETE the initial revision (should 204 now that we added a revision)
        ###################################################

        delete_url = reverse("v1.0:revision_detail", kwargs={"renkan_guid" : self.test_renkan_guid, "revision_guid": self.test_initial_revision_guid})
        delete_response = self.client.delete(delete_url, {}, format="json")
        self.assertEqual(delete_response.status_code, status.HTTP_204_NO_CONTENT)

        ###################################################
        # Try to DELETE with wrong renkan guid
        ###################################################

        delete_url = reverse("v1.0:revision_detail", kwargs={"renkan_guid" : "bad-renkan-guid", "revision_guid": self.test_initial_revision_guid})
        delete_response = self.client.delete(delete_url, {}, format="json")
        self.assertEqual(delete_response.status_code, status.HTTP_404_NOT_FOUND)

        ###################################################
        # Try to DELETE with wrong revision guid
        ###################################################

        delete_url = reverse("v1.0:revision_detail", kwargs={"renkan_guid" : self.test_renkan_guid, "revision_guid": "bad-revision-guid"})
        delete_response = self.client.delete(delete_url, {}, format="json")
        self.assertEqual(delete_response.status_code, status.HTTP_404_NOT_FOUND)
