from django.db import models
from django.utils.translation import ugettext_lazy as _
from django.conf import settings
from django.contrib.auth.models import User
from polemictweet.utils.unique_slug import unique_slugify
from sorl.thumbnail import ImageField
import logging


class Event(models.Model):
    
    STATE_CHOICES = (
    (1, 'Event to come'),
    (2, 'Live'),
    (3, 'Replay to come'),
    (4, 'Replay'),
    )
    title = models.CharField(max_length=1024, verbose_name=_('event.title'))
    description = models.TextField(null=True, blank=True, verbose_name=_('event.description'))
    program = models.TextField(null=True, blank=True, verbose_name=_('event.program'))
    date_begin = models.DateTimeField(null=True, verbose_name=_('event.date_begin'))
    date_end = models.DateTimeField(null=True, verbose_name=_('event.date_end'))
    hashtag = models.CharField(max_length=50, null=True, blank=True, verbose_name=_('event.hashtag'))
    slug = models.SlugField( null=True, blank=True, verbose_name=_('event.slug'))
    state = models.IntegerField(choices=STATE_CHOICES, default=1)
    creator = models.ForeignKey(User, blank=True, null=True, verbose_name=_('event.creator'))
    event_image = ImageField(upload_to="thumbnails/event/", max_length=200)
    event_logo = ImageField(upload_to="thumbnails/logo/", max_length=200)

    def save(self, *args, **kwargs):
        if not self.slug or self.slug == "" :
            slug_str = "%s" % (self.title)
            self.slug = unique_slugify(self, slug_str)    
        super(Event, self).save(*args, **kwargs)

class Live(models.Model):
    TYPE_CHOICES = (
    (1, 'Embed'),
    (2, 'Stream'),
    )
    provider = models.CharField(max_length=10, null=True, blank=True, default='rtmp', verbose_name=_('live.provider'))
    streamer = models.CharField(max_length=1024, null=True, blank=True, default='rtmp://media.iri.centrepompidou.fr/ddc-player', verbose_name=_('live.streamer'))
    file = models.CharField(max_length=1024, null=True, blank=True, default='livestream', verbose_name=_('live.file'))
    embed = models.TextField(null = True, blank = True, verbose_name=_('live.embed'))
    events =  models.ManyToManyField(Event)
    type = models.IntegerField(choices=TYPE_CHOICES, default=2)
    
    @staticmethod
    def create_or_change_live(type, provider, streamer, file, embed, id, events):
        if id is not None :
            live = Live.objects.get(id=int(id))
            live.type = type
            live.streamer = streamer
            live.provider = provider
            live.file = file
            live.embed = embed   
        else :
            if type == "1" :
                live = Live(type = type, embed=embed, provider='', streamer='', file='')   
            else :
                live = Live(provider=provider, streamer=streamer, file=file, embed='')    
            live.save()       
            for event in events:
                live.events.add(event)        
        live.save()
        return live

class Content(models.Model):
    display = models.CharField(max_length=1024, null=True, blank=True, verbose_name=_('content.display'))
    program = models.TextField(null=True, blank=True, verbose_name=_('content.program'))
    url = models.CharField(max_length=1024, null=True, blank=True, verbose_name=_('content.url'))
    events = models.ManyToManyField(Event)
    
    @staticmethod
    def create_content(display, program, url, events):
        content = Content(display = display, program=program, url=url)
        content.display_fr = display
        content.program_fr = program     
        content.save()       
        for event in events:
            content.events.add(event)        
        content.save()
        return content

class Group(models.Model):
    title = models.CharField(max_length=1024, null=True, blank=True, verbose_name=_('group.title'))
    description = models.TextField(null=True, blank=True, verbose_name=_('group.description'))
    image = ImageField(upload_to="thumbnails/group/", max_length=255)
    slug = models.SlugField( null=True, blank=True, verbose_name=_('group.slug'))
    events = models.ManyToManyField(Event)
    
    def save(self, *args, **kwargs):
        if self.slug =='' or not self.slug:
            slug_str = "%s" % (self.title) 
            self.slug = unique_slugify(self, slug_str)    
        super(Group, self).save(*args, **kwargs)

class Partner(models.Model):
    name = models.CharField(max_length=1024, verbose_name=_('partner.name'))
    url = models.CharField(max_length=1024, verbose_name=_('partner.url'))
    partner_image = ImageField(upload_to="thumbnails/partner/", max_length=200)

class Organisator(models.Model):
    event = models.ForeignKey('Event', verbose_name=_('organisator.event'))
    partner = models.ForeignKey('Partner', verbose_name=_('organisator.partner'))
    orga = models.BooleanField(verbose_name=_('organisator.orga'))

    
    
    
