from fabric.api import task, env, sudo
from fabric.colors import green
from fabric.tasks import Task
from ldt_fablib import (check_folder_access, syncdb, collectstatic, 
    export_version, do_sync_web, create_config, clean_export_folder, do_sync_comp, 
    sync_install_build, do_create_virtualenv, clean_rsync_folder, rsync_export, 
    get_comp_versions_dict)
import imp
import os.path

@task
def relaunch_server(do_collectstatic=True, do_syncdb=True):
    print("Relaunch server")
    check_folder_access()
    if do_syncdb:
        syncdb(env.remote_path['src'], env.remote_path['virtualenv'])
    if do_collectstatic:
        collectstatic(env.remote_path['src'], env.remote_path['virtualenv'], env.platform_web_module)
    sudo(env.web_relaunch_cmd, shell=False)

@task
def sync_web(version):
    print(green("sync web with version %s" % version))
    export_path = export_version(web=version)
    export_path_full = os.path.join(export_path,'web')
    do_sync_web(version, export_path_full)
    create_config(export_path_full)
    clean_export_folder(export_path)
    relaunch_server()
    
@task
def update_lib(version, package):
    print(green("update ldt with version %s" % version))
    export_path = export_version(web=version)
    export_path_full = os.path.join(export_path,'web')
    lib_path = os.path.join(export_path_full, "virtualenv", "res", "lib")
    
    f, pathname, description = imp.find_module("patch", [lib_path])
    imp.load_module("patch", f, pathname, description)
    f, pathname, description = imp.find_module("lib_create_env", [lib_path])
    lib_create_env = imp.load_module("lib_create_env", f, pathname, description)
    
    package_path_full = os.path.join(export_path_full, "virtualenv", "res", "src", lib_create_env.URLS[package]['local'])
    
    sync_install_build(package_path_full)
    clean_export_folder(export_path)
    relaunch_server()
    
@task
def sync_platform(version):
    print(green("sync platform with version web %s" % (version)))
    export_path = export_version(web=version)
    
    export_path_web = os.path.join(export_path,'web')
    do_sync_web(version, export_path_web)    

    comp_versions = get_comp_versions_dict(export_path_web)  

    for key in [k for k in env.repos if key != 'web']:
        export_path_key = export_version(**{key: comp_versions[key]})
        export_path_comp = os.path.join(export_path_key, key)
        do_sync_comp(key, export_path_comp)
        clean_export_folder(export_path_key)    
    
    create_config(export_path_web)
    clean_export_folder(export_path)
    relaunch_server()

@task
def create_virtualenv(version):
    print(green("create virtualenv with version %s" % version))
    export_path = export_version(web=version)
    export_path_web = os.path.join(export_path,'web')
    venv_remote_export_path = ""
    try:
        virtualenv_path = os.path.join(export_path_web, "virtualenv")
    
        venv_remote_export_path = os.path.join(env.remote_path['venv_export'], env.export_prefix, version,"virtualenv")
        rsync_export(virtualenv_path, venv_remote_export_path, env.rsync_filters['venv'])
        do_create_virtualenv(venv_remote_export_path, env.remote_path['virtualenv'])
    finally:
        clean_export_folder(export_path)
        if venv_remote_export_path:
            clean_rsync_folder(venv_remote_export_path)

class SyncComp(Task):
    
    def __init__(self, key):
        self.key = key

    def __get_name(self):
        return "sync_" + self.key
    
    name = property(__get_name)
    
    def run(self, version):
        print(green("sync ldt with version %s" % version))
        export_path_web = export_version(web=version)    
        export_path_web_full = os.path.join(export_path_web,'web')
        comp_versions = get_comp_versions_dict(export_path_web_full)

        export_path = export_version(**{self.key:comp_versions[self.key]})
        export_path_full = os.path.join(export_path,self.key)
        do_sync_comp(self.key, export_path_full)
        clean_export_folder(export_path)
        clean_export_folder(export_path_web)
        
        relaunch_server()

for sync_name in [key for key in env.repos if key != 'web']:
    globals()[sync_name] = SyncComp(sync_name)
