"""
This file demonstrates two different styles of tests (one doctest and one
unittest). These will both pass when you run "manage.py test".

Replace these with more appropriate tests for your application.
"""

import base64
import lxml.etree
import tempfile
import uuid

from django.contrib.auth import get_user_model

from ldt.ldt_utils.models import Project, Content
from ldt.ldt_utils.utils import (LdtUtils, LdtAnnotation, create_ldt,
    create_empty_iri, copy_ldt)
from ldt.test.client import Client
from ldt.test.testcases import TestCase


User = get_user_model()

class UtilsTest(TestCase):
    
    fixtures = ['testing_sites_data.json', 'user_data.json']
    
    def setUp(self):
        self.client = Client()
        User.objects.create_superuser('blop', 'blop@blop.com', 'blop')
        
        _ = self.client.login(username='blop', password='blop')
        
        self.user = User()
        self.user.username = 'blop'
        
        self.LU = LdtUtils()
        self.project = Project(title="titleproj1", owner=self.user)
        self.project.ldt = '<iri ldtversion="1.0.3" xmlns:dc="http://dublincore.org/documents/dcmi-namespace/"><project id="af3b99e4-b695-11df-bfde-00145ea4a2be" user="admin" title="CA:reponse a TC" abstract=""/> <medias> <media extra="" id="laurentcantet_entrelesmurs" pict="" src="http://amateur.iri.centrepompidou.fr//atelier/static/media/ldt/laurentcantet_entrelesmurs/laurentcantet_entrelesmurs.iri" video="rtmp://media.iri.centrepompidou.fr/ddc_player/video/regardssignes/"/> </medias> <annotations> <content id="laurentcantet_entrelesmurs"> <ensemble id="ens_perso" idProject="fe0d5d4c-2201-11df-8a24-00145ea4a2be" title="Decoupages personnels" author="perso" abstract=""> <decoupage id="c_EFC3FFE7-0204-A086-EBEC-D2A03A0E56CB" author="perso"> <title>CA: prof et admin</title> <abstract/> <elements> <element id="s_442AAB3A-42DA-F9BF-75E7-D2A0663FD5FF" begin="985690" dur="373222" author="" date="2010/09/02" color="16711680" src=""> <title/> <abstract/> <audio source=""/> <tags/> </element> <element id="s_0050F043-3AD2-0A7C-6699-D2A03A1EBA02" begin="5052858" dur="124407" author="" date="2010/09/02" color="10053375" src=""> <title>conseil de classe</title> <abstract>Reprise de la figure precedente</abstract> <audio source="undefined"/> <tags/> </element> </elements> </decoupage> <decoupage id="c_EEEF5C29-86E1-4AAE-E068-04EB5B00E492" author="perso"> <title>TC: prof et admin</title> <abstract/> <elements> <element id="s_880D9D4B-8BC0-BA43-5ECA-04EBA9FC9E59" begin="2426894" dur="141478" author="" date="2010/02/25" color="10053375" src=""> <title>Conseil de classe</title> <abstract/> <audio source=""/> <tags> <tag>Argumentation</tag> </tags> </element> <element id="s_D568A57C-7110-DED2-3165-04EC54387060" begin="5052858" dur="124407" author="" date="2010/02/25" color="10053375" src=""> <title>conseil de classe</title> <abstract>Reprise de la figure precedente</abstract> <audio source="undefined"/> <tags/> </element> </elements> </decoupage> </ensemble> </content> </annotations> <displays> <display id="0" title="Init view" idsel="laurentcantet_entrelesmurs" tc="2426424" zoom="0" scroll="0" infoBAB=""> <audio source=""/> <content id="laurentcantet_entrelesmurs"> <decoupage idens="en_2" id="de_PPP" tagsSelect=""/> <decoupage idens="laurentcantet_entrelesmurs" id="c_14A2E638-1936-97DC-E303-2DBA6A82A8B3" tagsSelect=""/> <decoupage idens="ens_perso" id="c_EEEF5C29-86E1-4AAE-E068-04EB5B00E492" tagsSelect=""/> <decoupage idens="ens_perso" id="c_EFC3FFE7-0204-A086-EBEC-D2A03A0E56CB" tagsSelect=""/> </content> </display> </displays> <edits> <editing id="0" tags=""> <title>Bout a bout 1</title> <abstract/> <edit id="edit1" tags=""> <eList/> <caption/> <audio/> <mList/> </edit> <edit id="edit2" tags=""> <eList/> <caption/> <audio/> <mList/> </edit> </editing> </edits> </iri>'
        self.project.id = "11321"
        self.project.ldt_id = str(uuid.uuid1())
        self.project.description = "proj1description"
        self.project.save()
        
        self.projectcopy = Project(title="the2ndproject")
        self.projectcopy.id = "21"
    
    def tearDown(self):
        self.project.delete()
        self.projectcopy.delete()
  
    def test_generate_ldt(self):
        self.cont1 = Content(iriurl="cont1_id/iriurl1", duration=123)
        self.cont1.iri_id = "cont1_id"
        self.cont1.save()
        
        self.cont2 = Content(iriurl="cont2_id/iriurl2", duration=100)
        self.cont2.iri_id = "cont2_id"
        self.cont2.save()
        
        self.project.contents.add(self.cont1, self.cont2)
    
        f = tempfile.TemporaryFile(mode='r+')
        doc = self.LU.generate_ldt(Content.objects.all())
        doc.write(f, pretty_print=True)
        f.seek(0)
        ldoc = lxml.etree.parse(f)
        self.assertEqual(ldoc.xpath("/iri/displays/display/content")[0].get("id"), self.cont1.iri_id)
        self.assertEqual(ldoc.xpath("/iri/medias/media")[0].get("id"), self.cont1.iri_id)
        
        self.assertEqual(ldoc.xpath("/iri/displays/display/content")[1].get("id"), self.cont2.iri_id)
        self.assertEqual(ldoc.xpath("/iri/medias/media")[1].get("id"), self.cont2.iri_id)
        f.close()
        
    def test_generate_init(self):
        self.cont3 = Content(iriurl="id3/iriurl3", duration=111)
        self.cont3.iri_id = "id3"
        self.cont3.save()
        
        self.cont4 = Content(iriurl="id4/iriurl4", duration=111)
        self.cont4.iri_id = "id4"
        self.cont4.save()
        
        search=''
        field='all'
        query = base64.urlsafe_b64encode(search.encode('utf8'))

        self.project.contents.add(self.cont3, self.cont4)
        ldoc = self.LU.generate_init({'field':field, 'query':query}, 'ldt.ldt_utils.views.lignesdetemps.search_ldt', 'ldt.ldt_utils.views.lignesdetemps.search_segments')
        self.assertEqual(ldoc.xpath("/iri/files/init")[0].tag, "init")
        self.assertEqual(ldoc.xpath("/iri/files/library")[0].tag, "library")    
    def test_create_ldt(self):
        self.cont5 = Content(iriurl="id5/iriurl5", duration=111)
        self.cont5.iri_id = "id5"
        self.cont5.save()
        
        self.cont6 = Content(iriurl="id6/iriurl6", duration=111)
        self.cont6.iri_id = "id6"
        self.cont6.save()
        
        self.project.contents.add(self.cont5, self.cont6)
        self.project.ldt = ""
        create_ldt(self.project, self.user)
        ldt = lxml.etree.fromstring(self.project.ldt_encoded)
        self.assertEqual(ldt.xpath("/iri")[0].tag, "iri")
        self.assertEqual(ldt.xpath("/iri/project")[0].get("title"), self.project.title)
        self.assertEqual(ldt.xpath("/iri/medias/media")[0].get("src"), self.cont5.relative_iri_url())
        self.assertEqual(ldt.xpath("/iri/medias/media")[1].get("id"), self.cont6.iri_id)

    def test_copy_ldt(self):
        self.cont7 = Content(iriurl="id7/iriurl7", duration=111)
        self.cont7.iri_id = "id7"
        self.cont7.save()
        
        self.cont8 = Content(iriurl="id8/iriurl8", duration=111)
        self.cont8.iri_id = "id8"
        self.cont8.save()
        
        self.project.contents.add(self.cont7, self.cont8)
        copy_ldt(self.project, self.projectcopy, self.user)
        ldt1 = lxml.etree.fromstring(self.project.ldt_encoded)
        ldt2 = lxml.etree.fromstring(self.projectcopy.ldt_encoded)
        self.assertTrue(ldt1.xpath("/iri/project")[0].get("id") != ldt2.xpath("/iri/project")[0].get("id"))
        self.assertEqual(ldt1.xpath("/iri/medias/media")[0].get("id"), ldt2.xpath("/iri/medias/media")[0].get("id"))
        self.assertEqual(ldt1.xpath("/iri/annotations/content/ensemble")[0].get("title"), ldt2.xpath("/iri/annotations/content/ensemble")[0].get("title"))
        self.assertEqual(ldt1.xpath("/iri/annotations/content/ensemble/decoupage")[0].get("id"), ldt2.xpath("/iri/annotations/content/ensemble/decoupage")[0].get("id"))
        self.assertEqual(ldt1.xpath("/iri/annotations/content/ensemble/decoupage/title")[1].text, ldt2.xpath("/iri/annotations/content/ensemble/decoupage/title")[1].text.strip("\n\t"))

    def test_create_empty_iri(self):
        self.cont9 = Content(iriurl="id9/iriurl9", duration=111)
        self.cont9.iri_id = "id9"
        self.cont9.save()
        
        self.cont10 = Content(iriurl="id10/iriurl10", duration=111)
        self.cont10.iri_id = "id10"
        self.cont10.save()
        
        self.project.contents.add(self.cont9, self.cont10)
        tmp = tempfile.TemporaryFile(mode='r+')
        create_empty_iri(tmp, self.cont9, "admin")
        tmp.seek(0)
        ldoc = lxml.etree.parse(tmp)
        self.assertEqual(ldoc.xpath("/iri/head/meta")[0].get("content"), self.cont9.iri_id)
        self.assertEqual(ldoc.xpath("/iri/body/medias/media/video")[0].get("id"), self.cont9.iri_id)
        tmp.close()
   
    def test_add_annotation(self):
        
        self.cont11 = Content(iriurl="id11/iriurl11", duration=111)
        self.cont11.iri_id = "id11"
        self.cont11.save()
        
        self.project.contents.add(self.cont11)
        self.project.ldt = ""
        create_ldt(self.project, self.user)
        
        self.LA = LdtAnnotation(self.project)

        self.LA.add("id11", "cutting_id", "cutting_title", "title", "text", ["tag1", "tag2"], "800",
                    "10000", "jdoe", "2011-09-10T09:12:58")
        self.LA.save()
        ldt = lxml.etree.fromstring(self.project.ldt)
        ann = ldt.xpath('/iri/annotations/content[@id="id11"]/ensemble/decoupage/elements/element')[0]
        title = ann.xpath('title')[0].text
        abstract = ann.xpath('abstract')[0].text

        self.assertEqual(ann.get("author"), "jdoe")
        self.assertEqual(title, "title")
        self.assertEqual(abstract, "text")
        