from django.conf import settings
from django.contrib.auth import models as auth_models
from django.contrib.auth.models import (AbstractUser, UserManager, Group,
    BaseUserManager, Permission)
from django.contrib.contenttypes.models import ContentType
from django.db import models
from django.db.models.signals import post_save
from django.utils import timezone
from sorl.thumbnail import ImageField
from south.signals import post_migrate


class LdtUserManager(BaseUserManager):

    def create_user(self, username, email=None, password=None, **extra_fields):
        """
        Creates and saves a User with the given username, email and password.
        """
        now = timezone.now()
        if not username:
            raise ValueError('The given username must be set')
        email = UserManager.normalize_email(email)
        user = self.model(username=username, email=email,
                          is_staff=False, is_active=True, is_superuser=False,
                          last_login=now, date_joined=now, **extra_fields)

        user.set_password(password)
        user.save(using=self._db)
        everyone, _ = Group.objects.get_or_create(name=settings.PUBLIC_GROUP_NAME)
        user.groups.add(everyone)
        return user

    def create_superuser(self, username, email, password, **extra_fields):
        u = self.create_user(username, email, password, **extra_fields)
        u.is_staff = True
        u.is_active = True
        u.is_superuser = True
        u.save(using=self._db)
        return u

# This method was added to stabilize migrations
def get_group_image_upload_to(instance, filename):
    return settings.MEDIA_ROOT+"thumbnails/groups/"

def get_group_image_default():
    return settings.DEFAULT_GROUP_ICON

class GroupProfile(models.Model):
    group = models.OneToOneField(Group, primary_key=False, related_name='profile')
    description = models.TextField()
    image = ImageField(upload_to=get_group_image_upload_to, default=get_group_image_default, max_length=200)
    
    @staticmethod
    def create_group_profile(sender, instance, created, **kwargs):
        if created:
            profile = GroupProfile()
            profile.group_id = instance.id
            profile.save()
            admins = LdtUser.objects.filter(is_superuser=True)
            if admins:
                admin_user = admins[0] 
                admin_user.groups.add(instance)
                admin_user.save()

def get_ldtuser_image_default():
    return settings.DEFAULT_USER_ICON

def get_ldtuser_language_default():
    if hasattr(settings, 'LANGUAGE_CODE'):
        return settings.LANGUAGE_CODE[:2]
    else :
        return 'en'

class LdtUser(AbstractUser):
    language = models.CharField(max_length=2, default=get_ldtuser_language_default)
    image = ImageField(upload_to="thumbnails/users/", default=get_ldtuser_image_default, max_length=200)
    
    objects = LdtUserManager()


def get_profile(self):
    return self.profile

Group.add_to_class("get_profile", get_profile)  # @UndefinedVariable
post_save.connect(GroupProfile.create_group_profile, sender=Group)


def add_group_permissions(sender, **kwargs):
    ct = ContentType.objects.get(app_label='auth', model='group')
    Permission.objects.get_or_create(codename='is_owner_group', name='Can add/delete admins to the group',content_type=ct)

post_migrate.connect(add_group_permissions, sender=auth_models)

