from __future__ import with_statement
import os
from django.contrib.auth.models import User
from django.contrib.auth.forms import (UserCreationForm, AuthenticationForm,
    PasswordChangeForm, SetPasswordForm, UserChangeForm, PasswordResetForm)
from django.core import mail
from django.forms.fields import Field, EmailField
from django.test import TestCase
from django.test.utils import override_settings
from django.utils.encoding import force_unicode
from django.utils import translation


class UserCreationFormTest(TestCase):

    fixtures = ['authtestdata.json']

    def test_user_already_exists(self):
        data = {
            'username': 'testclient',
            'password1': 'test123',
            'password2': 'test123',
            }
        form = UserCreationForm(data)
        self.assertFalse(form.is_valid())
        self.assertEqual(form["username"].errors,
                         [force_unicode(form.error_messages['duplicate_username'])])

    def test_invalid_data(self):
        data = {
            'username': 'jsmith!',
            'password1': 'test123',
            'password2': 'test123',
            }
        form = UserCreationForm(data)
        self.assertFalse(form.is_valid())
        self.assertEqual(form["username"].errors,
                         [force_unicode(form.fields['username'].error_messages['invalid'])])

    def test_password_verification(self):
        # The verification password is incorrect.
        data = {
            'username': 'jsmith',
            'password1': 'test123',
            'password2': 'test',
            }
        form = UserCreationForm(data)
        self.assertFalse(form.is_valid())
        self.assertEqual(form["password2"].errors,
                         [force_unicode(form.error_messages['password_mismatch'])])

    def test_both_passwords(self):
        # One (or both) passwords weren't given
        data = {'username': 'jsmith'}
        form = UserCreationForm(data)
        required_error = [force_unicode(Field.default_error_messages['required'])]
        self.assertFalse(form.is_valid())
        self.assertEqual(form['password1'].errors, required_error)
        self.assertEqual(form['password2'].errors, required_error)

        data['password2'] = 'test123'
        form = UserCreationForm(data)
        self.assertFalse(form.is_valid())
        self.assertEqual(form['password1'].errors, required_error)

    def test_success(self):
        # The success case.
        data = {
            'username': 'jsmith@example.com',
            'password1': 'test123',
            'password2': 'test123',
            }
        form = UserCreationForm(data)
        self.assertTrue(form.is_valid())
        u = form.save()
        self.assertEqual(repr(u), '<User: jsmith@example.com>')

UserCreationFormTest = override_settings(USE_TZ=False)(UserCreationFormTest)


class AuthenticationFormTest(TestCase):

    fixtures = ['authtestdata.json']

    def test_invalid_username(self):
        # The user submits an invalid username.

        data = {
            'username': 'jsmith_does_not_exist',
            'password': 'test123',
            }
        form = AuthenticationForm(None, data)
        self.assertFalse(form.is_valid())
        self.assertEqual(form.non_field_errors(),
                         [force_unicode(form.error_messages['invalid_login'])])

    def test_inactive_user(self):
        # The user is inactive.
        data = {
            'username': 'inactive',
            'password': 'password',
            }
        form = AuthenticationForm(None, data)
        self.assertFalse(form.is_valid())
        self.assertEqual(form.non_field_errors(),
                         [force_unicode(form.error_messages['inactive'])])

    def test_inactive_user_i18n(self):
        with self.settings(USE_I18N=True):
            with translation.override('pt-br', deactivate=True):
                # The user is inactive.
                data = {
                    'username': 'inactive',
                    'password': 'password',
                    }
                form = AuthenticationForm(None, data)
                self.assertFalse(form.is_valid())
                self.assertEqual(form.non_field_errors(),
                                 [force_unicode(form.error_messages['inactive'])])

    def test_success(self):
        # The success case
        data = {
            'username': 'testclient',
            'password': 'password',
            }
        form = AuthenticationForm(None, data)
        self.assertTrue(form.is_valid())
        self.assertEqual(form.non_field_errors(), [])

AuthenticationFormTest = override_settings(USE_TZ=False)(AuthenticationFormTest)


class SetPasswordFormTest(TestCase):

    fixtures = ['authtestdata.json']

    def test_password_verification(self):
        # The two new passwords do not match.
        user = User.objects.get(username='testclient')
        data = {
            'new_password1': 'abc123',
            'new_password2': 'abc',
            }
        form = SetPasswordForm(user, data)
        self.assertFalse(form.is_valid())
        self.assertEqual(form["new_password2"].errors,
                         [force_unicode(form.error_messages['password_mismatch'])])

    def test_success(self):
        user = User.objects.get(username='testclient')
        data = {
            'new_password1': 'abc123',
            'new_password2': 'abc123',
            }
        form = SetPasswordForm(user, data)
        self.assertTrue(form.is_valid())

SetPasswordFormTest = override_settings(USE_TZ=False)(SetPasswordFormTest)


class PasswordChangeFormTest(TestCase):

    fixtures = ['authtestdata.json']

    def test_incorrect_password(self):
        user = User.objects.get(username='testclient')
        data = {
            'old_password': 'test',
            'new_password1': 'abc123',
            'new_password2': 'abc123',
            }
        form = PasswordChangeForm(user, data)
        self.assertFalse(form.is_valid())
        self.assertEqual(form["old_password"].errors,
                         [force_unicode(form.error_messages['password_incorrect'])])

    def test_password_verification(self):
        # The two new passwords do not match.
        user = User.objects.get(username='testclient')
        data = {
            'old_password': 'password',
            'new_password1': 'abc123',
            'new_password2': 'abc',
            }
        form = PasswordChangeForm(user, data)
        self.assertFalse(form.is_valid())
        self.assertEqual(form["new_password2"].errors,
                         [force_unicode(form.error_messages['password_mismatch'])])

    def test_success(self):
        # The success case.
        user = User.objects.get(username='testclient')
        data = {
            'old_password': 'password',
            'new_password1': 'abc123',
            'new_password2': 'abc123',
            }
        form = PasswordChangeForm(user, data)
        self.assertTrue(form.is_valid())

    def test_field_order(self):
        # Regression test - check the order of fields:
        user = User.objects.get(username='testclient')
        self.assertEqual(PasswordChangeForm(user, {}).fields.keys(),
                         ['old_password', 'new_password1', 'new_password2'])

PasswordChangeFormTest = override_settings(USE_TZ=False)(PasswordChangeFormTest)


class UserChangeFormTest(TestCase):

    fixtures = ['authtestdata.json']

    def test_username_validity(self):
        user = User.objects.get(username='testclient')
        data = {'username': 'not valid'}
        form = UserChangeForm(data, instance=user)
        self.assertFalse(form.is_valid())
        self.assertEqual(form['username'].errors,
                         [force_unicode(form.fields['username'].error_messages['invalid'])])

    def test_bug_14242(self):
        # A regression test, introduce by adding an optimization for the
        # UserChangeForm.

        class MyUserForm(UserChangeForm):
            def __init__(self, *args, **kwargs):
                super(MyUserForm, self).__init__(*args, **kwargs)
                self.fields['groups'].help_text = 'These groups give users different permissions'

            class Meta(UserChangeForm.Meta):
                fields = ('groups',)

        # Just check we can create it
        form = MyUserForm({})

    def test_bug_17944_empty_password(self):
        user = User.objects.get(username='empty_password')
        form = UserChangeForm(instance=user)
        # Just check that no error is raised.
        form.as_table()

    def test_bug_17944_unmanageable_password(self):
        user = User.objects.get(username='unmanageable_password')
        form = UserChangeForm(instance=user)
        # Just check that no error is raised.
        form.as_table()

    def test_bug_17944_unknown_password_algorithm(self):
        user = User.objects.get(username='unknown_password')
        form = UserChangeForm(instance=user)
        # Just check that no error is raised.
        form.as_table()


UserChangeFormTest = override_settings(USE_TZ=False)(UserChangeFormTest)


class PasswordResetFormTest(TestCase):

    fixtures = ['authtestdata.json']

    def create_dummy_user(self):
        """creates a user and returns a tuple
        (user_object, username, email)
        """
        username = 'jsmith'
        email = 'jsmith@example.com'
        user = User.objects.create_user(username, email, 'test123')
        return (user, username, email)

    def test_invalid_email(self):
        data = {'email': 'not valid'}
        form = PasswordResetForm(data)
        self.assertFalse(form.is_valid())
        self.assertEqual(form['email'].errors,
                         [force_unicode(EmailField.default_error_messages['invalid'])])

    def test_nonexistant_email(self):
        # Test nonexistant email address
        data = {'email': 'foo@bar.com'}
        form = PasswordResetForm(data)
        self.assertFalse(form.is_valid())
        self.assertEqual(form.errors,
                         {'email': [force_unicode(form.error_messages['unknown'])]})

    def test_cleaned_data(self):
        # Regression test
        (user, username, email) = self.create_dummy_user()
        data = {'email': email}
        form = PasswordResetForm(data)
        self.assertTrue(form.is_valid())
        self.assertEqual(form.cleaned_data['email'], email)

    def test_custom_email_subject(self):
        template_path = os.path.join(os.path.dirname(__file__), 'templates')
        with self.settings(TEMPLATE_DIRS=(template_path,)):
            data = {'email': 'testclient@example.com'}
            form = PasswordResetForm(data)
            self.assertTrue(form.is_valid())
            # Since we're not providing a request object, we must provide a
            # domain_override to prevent the save operation from failing in the
            # potential case where contrib.sites is not installed. Refs #16412.
            form.save(domain_override='example.com')
            self.assertEqual(len(mail.outbox), 1)
            self.assertEqual(mail.outbox[0].subject, u'Custom password reset on example.com')

    def test_bug_5605(self):
        # bug #5605, preserve the case of the user name (before the @ in the
        # email address) when creating a user.
        user = User.objects.create_user('forms_test2', 'tesT@EXAMple.com', 'test')
        self.assertEqual(user.email, 'tesT@example.com')
        user = User.objects.create_user('forms_test3', 'tesT', 'test')
        self.assertEqual(user.email, 'tesT')

    def test_inactive_user(self):
        #tests that inactive user cannot
        #receive password reset email
        (user, username, email) = self.create_dummy_user()
        user.is_active = False
        user.save()
        form = PasswordResetForm({'email': email})
        self.assertFalse(form.is_valid())

    def test_unusable_password(self):
        user = User.objects.create_user('testuser', 'test@example.com', 'test')
        data = {"email": "test@example.com"}
        form = PasswordResetForm(data)
        self.assertTrue(form.is_valid())
        user.set_unusable_password()
        user.save()
        form = PasswordResetForm(data)
        self.assertFalse(form.is_valid())
        self.assertEqual(form["email"].errors,
                         [u"The user account associated with this e-mail address cannot reset the password."])

PasswordResetFormTest = override_settings(USE_TZ=False)(PasswordResetFormTest)
