from django.conf import settings
from django.conf.urls.defaults import url
from django.contrib.auth.models import Group
from guardian.shortcuts import assign
from ldt.ldt_utils.models import Project
from ldt.api.ldt.authentication import SessionAuthentication, MultiAuthentication, ApiKeyAuthentication
from ldt.api.ldt.serializers.cinelabserializer import CinelabSerializer
from ldt.api.ldt.resources import ContentResource
from ldt.api.ldt.resources.user import UserResource
from ldt.security import protect_models, unprotect_models
from tastypie.authorization import Authorization
from tastypie.resources import Bundle, ModelResource, ALL
from tastypie import fields
from tastypie.exceptions import BadRequest


class ProjectResource(ModelResource):
    contents = fields.ManyToManyField(ContentResource, 'contents')
    owner = fields.ForeignKey(UserResource, 'owner')
    class Meta:
        allowed_methods = ['get', 'post', 'put']
        authorization = Authorization() # BE CAREFUL WITH THAT, it's unsecure
        authentication = MultiAuthentication(ApiKeyAuthentication(), SessionAuthentication())
        resource_name = 'projects'
        queryset = Project.objects.all()
        serializer = CinelabSerializer()
        filtering = {
            'state' : ALL,
            'ldt_id' : ALL,
            'title' : ALL
        }
        # In the future version :
        # detail_uri_name = 'ldt_id'
    
    
    def get_object_list(self, request):
        return Project.safe_objects.all()
    
    def override_urls(self):
        # WARNING : in tastypie <= 1.0, override_urls is used instead of prepend_urls. From 1.0.0, prepend_urls will be prefered and override_urls deprecated 
        return [
            url(r"^(?P<resource_name>%s)/(?P<ldt_id>[\w\d_.-]+)/$" % self._meta.resource_name, self.wrap_view('dispatch_detail'), name="api_dispatch_detail"),
        ]
    
    def get_resource_uri(self, bundle_or_obj):
        kwargs = {
            'resource_name': self._meta.resource_name,
            'api_name': self._meta.api_name
        }
        if isinstance(bundle_or_obj, Bundle):
            kwargs['ldt_id'] = bundle_or_obj.obj.ldt_id
        else:
            kwargs['ldt_id'] = bundle_or_obj.ldt_id
        return self._build_reverse_url("api_dispatch_detail", kwargs=kwargs)
        
    def obj_create(self, bundle, request=None, **kwargs):
        unprotect_models()
        bundle = super(ProjectResource, self).obj_create(bundle, request)
        # Assign permission for the owner
        assign('view_project', request.user, bundle.obj)
        assign('change_project', request.user, bundle.obj)
        # Since the project is published by default, we assign permission for the everyone group
        everyone = Group.objects.get(name=settings.PUBLIC_GROUP_NAME)
        assign('ldt_utils.view_project', everyone, bundle.obj)
        protect_models()
        return bundle
    
    def obj_delete_list(self, request=None, **kwargs):
        raise BadRequest("PUT with a list of projects is forbidden.")
    