package org.apache.lucene.search;
/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import java.io.IOException;

import org.apache.lucene.util.LuceneTestCase;

import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.store.RAMDirectory;

public class TestBooleanOr extends LuceneTestCase {

  private static String FIELD_T = "T";
  private static String FIELD_C = "C";

  private TermQuery t1 = new TermQuery(new Term(FIELD_T, "files"));
  private TermQuery t2 = new TermQuery(new Term(FIELD_T, "deleting"));
  private TermQuery c1 = new TermQuery(new Term(FIELD_C, "production"));
  private TermQuery c2 = new TermQuery(new Term(FIELD_C, "optimize"));

  private IndexSearcher searcher = null;

  private int search(Query q) throws IOException {
    QueryUtils.check(q,searcher);
    return searcher.search(q, null, 1000).totalHits;
  }

  public void testElements() throws IOException {
    assertEquals(1, search(t1));
    assertEquals(1, search(t2));
    assertEquals(1, search(c1));
    assertEquals(1, search(c2));
  }

  /**
   * <code>T:files T:deleting C:production C:optimize </code>
   * it works.
   *
   * @throws IOException
   */
  public void testFlat() throws IOException {
    BooleanQuery q = new BooleanQuery();
    q.add(new BooleanClause(t1, BooleanClause.Occur.SHOULD));
    q.add(new BooleanClause(t2, BooleanClause.Occur.SHOULD));
    q.add(new BooleanClause(c1, BooleanClause.Occur.SHOULD));
    q.add(new BooleanClause(c2, BooleanClause.Occur.SHOULD));
    assertEquals(1, search(q));
  }

  /**
   * <code>(T:files T:deleting) (+C:production +C:optimize)</code>
   * it works.
   *
   * @throws IOException
   */
  public void testParenthesisMust() throws IOException {
    BooleanQuery q3 = new BooleanQuery();
    q3.add(new BooleanClause(t1, BooleanClause.Occur.SHOULD));
    q3.add(new BooleanClause(t2, BooleanClause.Occur.SHOULD));
    BooleanQuery q4 = new BooleanQuery();
    q4.add(new BooleanClause(c1, BooleanClause.Occur.MUST));
    q4.add(new BooleanClause(c2, BooleanClause.Occur.MUST));
    BooleanQuery q2 = new BooleanQuery();
    q2.add(q3, BooleanClause.Occur.SHOULD);
    q2.add(q4, BooleanClause.Occur.SHOULD);
    assertEquals(1, search(q2));
  }

  /**
   * <code>(T:files T:deleting) +(C:production C:optimize)</code>
   * not working. results NO HIT.
   *
   * @throws IOException
   */
  public void testParenthesisMust2() throws IOException {
    BooleanQuery q3 = new BooleanQuery();
    q3.add(new BooleanClause(t1, BooleanClause.Occur.SHOULD));
    q3.add(new BooleanClause(t2, BooleanClause.Occur.SHOULD));
    BooleanQuery q4 = new BooleanQuery();
    q4.add(new BooleanClause(c1, BooleanClause.Occur.SHOULD));
    q4.add(new BooleanClause(c2, BooleanClause.Occur.SHOULD));
    BooleanQuery q2 = new BooleanQuery();
    q2.add(q3, BooleanClause.Occur.SHOULD);
    q2.add(q4, BooleanClause.Occur.MUST);
    assertEquals(1, search(q2));
  }

  /**
   * <code>(T:files T:deleting) (C:production C:optimize)</code>
   * not working. results NO HIT.
   *
   * @throws IOException
   */
  public void testParenthesisShould() throws IOException {
    BooleanQuery q3 = new BooleanQuery();
    q3.add(new BooleanClause(t1, BooleanClause.Occur.SHOULD));
    q3.add(new BooleanClause(t2, BooleanClause.Occur.SHOULD));
    BooleanQuery q4 = new BooleanQuery();
    q4.add(new BooleanClause(c1, BooleanClause.Occur.SHOULD));
    q4.add(new BooleanClause(c2, BooleanClause.Occur.SHOULD));
    BooleanQuery q2 = new BooleanQuery();
    q2.add(q3, BooleanClause.Occur.SHOULD);
    q2.add(q4, BooleanClause.Occur.SHOULD);
    assertEquals(1, search(q2));
  }

  @Override
  protected void setUp() throws Exception {
    super.setUp();

    //
    RAMDirectory rd = new RAMDirectory();

    //
    IndexWriter writer = new IndexWriter(rd, new StandardAnalyzer(org.apache.lucene.util.Version.LUCENE_CURRENT), true, IndexWriter.MaxFieldLength.LIMITED);

    //
    Document d = new Document();
    d.add(new Field(
        FIELD_T,
        "Optimize not deleting all files",
        Field.Store.YES,
        Field.Index.ANALYZED));
    d.add(new Field(
        FIELD_C,
        "Deleted When I run an optimize in our production environment.",
        Field.Store.YES,
        Field.Index.ANALYZED));

    //
    writer.addDocument(d);
    writer.close();

    //
    searcher = new IndexSearcher(rd, true);
  }
}
