from django.http import HttpResponse, HttpResponseForbidden
from django.utils.translation import ugettext as _
from ldt.ldt_utils.models import Project
from ldt.ldt_utils.projectserializer import ProjectJsonSerializer
import ldt.auth as ldt_auth
import logging
import lxml.etree


def project_annotations_rdf(request, ldt_id):

    project = Project.safe_objects.get(ldt_id=ldt_id); #@UndefinedVariable
    
    if not ldt_auth.check_access(request.user, project):
        return HttpResponseForbidden(_("You can not access this project"))

    content_type = request.REQUEST.get("content_type")
    if content_type is None:
        content_type = "application/rdf+xml; charset=utf-8"
    else:
        content_type = content_type.encode("utf-8")
    if "charset" not in content_type:
        content_type += "; charset=utf-8" 
    resp = HttpResponse(content_type=content_type)
    resp['Cache-Control'] = 'no-cache, must-revalidate'
    resp['Pragma'] = 'no-cache'

    ps = ProjectJsonSerializer(project, from_contents=False, from_display=True) 
    annotations = ps.get_annotations(first_cutting=True)
    
    rdf_ns = u"http://www.w3.org/1999/02/22-rdf-syntax-ns#"
    dc_ns = u"http://purl.org/dc/elements/1.1/"
    rdf = u"{%s}" % rdf_ns
    dc = u"{%s}" % dc_ns
    nsmap = {u'rdf' : rdf_ns, u'dc':dc_ns}
    
    rdf_root = lxml.etree.Element(rdf + u"RDF", nsmap=nsmap)
    
    logging.debug("RDF annotations : " + repr(annotations)) #@UndefinedVariable
    
    for annotation in annotations:
        uri = u""
        if 'uri' in annotation and annotation['uri']:
            uri = unicode(annotation['uri'])
        annot_desc = lxml.etree.SubElement(rdf_root, rdf + u"Description")
        annot_desc.set(rdf + u'about', uri)
        if annotation['title']:
            lxml.etree.SubElement(annot_desc, dc + 'title').text = lxml.etree.CDATA(unicode(annotation['title']))
        if annotation['desc']:
            lxml.etree.SubElement(annot_desc, dc + 'description').text = lxml.etree.CDATA(unicode(annotation['desc']))
        if annotation['tags']:
            for tag in annotation['tags']:
                lxml.etree.SubElement(annot_desc, dc + 'subject').text = lxml.etree.CDATA(unicode(tag))
        lxml.etree.SubElement(annot_desc, dc + 'coverage').text = u"start=%s, duration=%s" % (annotation['begin'], annotation['duration'])
        
    resp.write(u"<?xml version=\"1.0\" encoding=\"utf-8\"?>\n")
    resp.write(u"<!DOCTYPE rdf:RDF PUBLIC \"-//DUBLIN CORE//DCMES DTD 2002/07/31//EN\" \"http://dublincore.org/documents/2002/07/31/dcmes-xml/dcmes-xml-dtd.dtd\">\n")
    
    resp.write(lxml.etree.tostring(rdf_root, xml_declaration=False, encoding="utf-8", pretty_print=True))

    return resp