from django.conf import settings
from django.contrib.auth.decorators import login_required
from django.core.urlresolvers import reverse
from django.db.models import Q
from django.forms.models import model_to_dict
from django.forms.util import ErrorList
from django.http import (HttpResponse, HttpResponseRedirect, 
    HttpResponseForbidden, HttpResponseServerError)
from django.shortcuts import (render_to_response, get_object_or_404, 
    get_list_or_404)
from django.template import RequestContext
from django.template.loader import render_to_string
from django.utils import simplejson
from django.utils.html import escape
from django.utils.translation import ugettext as _, ungettext
from forms import (LdtAddForm, SearchForm, AddProjectForm, CopyProjectForm, 
    ContentForm, MediaForm)
from ldt.core.models import Owner
from ldt.ldt_utils.models import Content
from ldt.ldt_utils.utils import boolean_convert, LdtUtils, LdtSearch
from lxml.html import fragment_fromstring
from models import Media, Project
from projectserializer import ProjectSerializer
from urllib2 import urlparse
import base64
import django.core.urlresolvers
import ldt.auth as ldt_auth
import ldt.utils.path as ldt_utils_path
import logging
import lxml.etree
import mimetypes
import os
import urllib2


@login_required
def workspace(request):

    # list of contents
    content_list = Content.objects.all() #@UndefinedVariable

    # get list of projects
    project_list = Project.objects.all() #@UndefinedVariable
    
    is_gecko = ((request.META['HTTP_USER_AGENT'].lower().find("firefox")) > -1);
    
    # render list
    return render_to_response("ldt/ldt_utils/workspace.html",
                              {'contents': content_list, 'projects': project_list,
                               'is_gecko': is_gecko},
                              context_instance=RequestContext(request))


def popup_embed(request):

    json_url = request.GET.get("json_url")
    player_id = request.GET.get("player_id")
    ldt_id = request.GET.get("ldt_id")


    project = Project.objects.get(ldt_id=ldt_id); #@UndefinedVariable

    stream_mode = project.stream_mode
    if stream_mode != "video":
        stream_mode = 'radio'

    player_width = 650
    player_height = 480
    
    if stream_mode == 'radio':
        player_height = 1
        
    if not ldt_auth.checkAccess(request.user, project):
        return HttpResponseForbidden(_("You can not access this project"))

    ps = ProjectSerializer(project, from_contents=False, from_display=True)
    annotations = ps.getAnnotations(first_cutting=True)
    
    rend_dict = {'json_url':json_url, 'player_id':player_id, 'annotations':annotations, 'ldt_id': ldt_id, 'stream_mode': stream_mode, 'player_width': player_width, 'player_height': player_height}

    embed_rendered = dict((typestr,
                           (lambda s:escape(lxml.etree.tostring(fragment_fromstring(render_to_string("ldt/ldt_utils/partial/embed_%s.html" % (s), rend_dict, context_instance=RequestContext(request))), pretty_print=True)))(typestr))
                           for typestr in ('player', 'seo_body', 'seo_meta', 'links'))

    rend_dict['embed_rendered'] = embed_rendered

    return render_to_response("ldt/ldt_utils/embed_popup.html",
                              rend_dict,
                              context_instance=RequestContext(request))



@login_required
def projectsfilter(request, filter, is_owner=False, status=0):

    is_owner = boolean_convert(is_owner)
    status = int(status)
    query = Q()

    if is_owner:
        owner = None
        try:
            owner = Owner.objects.get(user=request.user) #@UndefinedVariable
        except:
            return HttpResponseServerError("<h1>User not found</h1>")
        query &= Q(owner=owner)

    if status > 0:
        query &= Q(state=status)

    if filter:
        if len(filter) > 0 and filter[0] == '_':
            filter = filter[1:]
        query &= Q(title__icontains=filter)

    project_list = Project.objects.filter(query) #@UndefinedVariable
    
    is_gecko = ((request.META['HTTP_USER_AGENT'].lower().find("firefox")) > -1);

    return render_to_response("ldt/ldt_utils/partial/projectslist.html",
                              {'projects': project_list, 'is_gecko': is_gecko},
                              context_instance=RequestContext(request))



@login_required
def contentsfilter(request, filter):
    if filter and len(filter) > 0 and filter[0] == '_':
        filter = filter[1:]

    if filter:
        content_list = Content.objects.filter(title__icontains=filter) #@UndefinedVariable
    else:
        content_list = Content.objects.all() #@UndefinedVariable

    return render_to_response("ldt/ldt_utils/partial/contentslist.html",
                              {'contents': content_list},
                              context_instance=RequestContext(request))


def searchForm(request):
    form = SearchForm()
    return render_to_response('ldt/ldt_utils/search_form.html', {'form': form} , context_instance=RequestContext(request))    

def searchIndex(request):
    
    sform = SearchForm(request.POST)
    if sform.is_valid():
        search = sform.cleaned_data["search"]
    
    
        queryStr = base64.urlsafe_b64encode(search.encode('utf8'))
        field = request.POST["field"]
        language_code = request.LANGUAGE_CODE[:2]
    
        url = settings.WEB_URL + django.core.urlresolvers.reverse("ldt.ldt_utils.views.searchInit", args=[field, queryStr])
        return render_to_response('ldt/ldt_utils/init_ldt.html', {'LDT_MEDIA_PREFIX': settings.LDT_MEDIA_PREFIX, 'colorurl': settings.LDT_MEDIA_PREFIX + 'swf/ldt/pkg/color.xml', 'i18nurl': settings.LDT_MEDIA_PREFIX + 'swf/ldt/pkg/i18n', 'language': language_code, 'baseurl': settings.LDT_MEDIA_PREFIX + 'swf/ldt/', 'url': url}, context_instance=RequestContext(request))
    else:
        resp = HttpResponse()
        resp.write("<html><head></head><body>Error : No result</body></html>");

def searchIndexGet(request, field, query):

    language_code = request.LANGUAGE_CODE[:2]
    url = settings.WEB_URL + django.core.urlresolvers.reverse("ldt.ldt_utils.views.searchInit", args=[field, query])
    return render_to_response('ldt/ldt_utils/init_ldt.html', {'LDT_MEDIA_PREFIX': settings.LDT_MEDIA_PREFIX, 'colorurl': settings.LDT_MEDIA_PREFIX + 'swf/ldt/pkg/color.xml', 'i18nurl': settings.LDT_MEDIA_PREFIX + 'swf/ldt/pkg/i18n', 'language': language_code, 'baseurl': settings.LDT_MEDIA_PREFIX + 'swf/ldt/', 'url': url}, context_instance=RequestContext(request))

def searchInit(request, field, query):
    
    ldtgen = LdtUtils()
    
    doc = ldtgen.generateInit([field, query], 'ldt.ldt_utils.views.searchLdt', 'ldt.ldt_utils.views.searchSegments')
    
    resp = HttpResponse(mimetype="text/xml;charset=utf-8")
    doc.write(resp, pretty_print=True)
    return resp

def searchLdt(request, field, query, edition=None):
    
    contentList = []
    resp = HttpResponse(mimetype="text/xml")
    queryStr = ""

    if query and len(query) > 0:        
        queryStr = base64.urlsafe_b64decode(query.encode("ascii")).decode("utf8")
        searcher = LdtSearch()
        ids = {}
        
        for result in searcher.query(field, queryStr):
            ids[result["iri_id"]] = ""            

        id_list = ids.keys()
        
        #if edition is not None:
        #    ids_editions = map(lambda t:t[0], filter(lambda id: id[0] is not None, Speak.objects.filter(session__day__edition=edition).order_by("session__start_ts", "order").values_list("content__iri_id")))            
        #    id_list = filter(lambda id: id in id_list, ids_editions)
            
        contentList = Content.objects.filter(iri_id__in=id_list)        #@UndefinedVariable

            
    ldtgen = LdtUtils()
    ldtgen.generateLdt(contentList, file=resp, title=u"Recherche : " + queryStr)
    
    return resp


def searchSegments(request, field, query, edition=None):
    
    if query and len(query) > 0:
        searcher = LdtSearch()
        
        queryStr = base64.urlsafe_b64decode(query.encode("ascii")).decode("utf8")
        res = searcher.query(field, queryStr)
    else:
        res = []
        
    iri_ids = None
    
    #if edition is not None:
    #    iri_ids = map(lambda t:t[0], filter(lambda id: id[0] is not None, Speak.objects.filter(session__day__edition=edition).order_by("session__start_ts", "order").values_list("content__iri_id")))

    iri = lxml.etree.Element('iri')
    doc = lxml.etree.ElementTree(iri)	

    for resultMap in res:
        if iri_ids is None or resultMap['iri_id'] in iri_ids:
            seg = lxml.etree.SubElement(iri, 'seg')
            seg.set('idctt', resultMap['iri_id'])
            seg.set('idens', resultMap['ensemble_id'])
            seg.set('iddec', resultMap['decoupage_id'])
            seg.set('idseg', resultMap['element_id'])
            seg.set('idvue', "")
            seg.set('crit', "")

    return doc  
  
    return HttpResponse(lxml.etree.tostring(doc, pretty_print=True), mimetype="text/xml;charset=utf-8") 



@login_required         
def list_ldt(request):
    contents = Content.objects.all() #@UndefinedVariable
    try:
        owner = Owner.objects.get(user=request.user) #@UndefinedVariable
    except:
        return HttpResponseRedirect(settings.LOGIN_URL)
    ldtProjects = Project.objects.filter(owner=owner) #@UndefinedVariable
    context = {
    'contents': contents,
    'projects': ldtProjects.reverse(),
    }
    return render_to_response('ldt/ldt_utils/ldt_list.html', context, context_instance=RequestContext(request))

@login_required         
def list_content(request):
    contents = Content.objects.all() #@UndefinedVariable
    context = {
        'contents': contents,
    }
    return render_to_response('ldt/ldt_utils/content_list.html', context, context_instance=RequestContext(request))

@login_required
def create_ldt_view(request):
    if request.method == "POST" :
        form = LdtAddForm(request.POST)
        if form.is_valid():
            user = request.user
            Project.create_project(title=form.cleaned_data['title'], user=user, contents=form.cleaned_data['contents'])
            form_status = "saved"
            contents = []
            #return HttpResponseRedirect(reverse("ldt.ldt_utils.views.list_ldt"))
    else:
        form = LdtAddForm()
        contents = Content.objects.all() #@UndefinedVariable
        form_status = "none"
            
    return render_to_response('ldt/ldt_utils/create_ldt.html', {'contents': contents, 'form': form, 'form_status':form_status, 'create_project_action':reverse(create_ldt_view)}, context_instance=RequestContext(request))
     
def created_ldt(request):
    return render_to_response('ldt/ldt_utils/save_done.html', context_instance=RequestContext(request))
    
def indexProject(request, id):

    urlStr = settings.WEB_URL + reverse("space_ldt_init", args=['ldtProject', id])
    posturl = settings.WEB_URL + reverse("ldt.ldt_utils.views.save_ldtProject")
    language_code = request.LANGUAGE_CODE[:2]
    
    ldt = get_object_or_404(Project, ldt_id=id)
    if ldt.state == 2: #published
        readonly = 'true'
    else:
        readonly = 'false'
    
    return render_to_response('ldt/ldt_utils/init_ldt.html', {'LDT_MEDIA_PREFIX': settings.LDT_MEDIA_PREFIX, 'colorurl': settings.LDT_MEDIA_PREFIX + 'swf/ldt/pkg/color.xml', 'i18nurl': settings.LDT_MEDIA_PREFIX + 'swf/ldt/pkg/i18n', 'language': language_code, 'baseurl': settings.LDT_MEDIA_PREFIX + 'swf/ldt/', 'url': urlStr, 'posturl': posturl, 'id': id, 'readonly': readonly}, context_instance=RequestContext(request))
    
def indexProjectFull(request, id):

    urlStr = settings.WEB_URL + reverse("space_ldt_init", args=['ldtProject', id])
    posturl = settings.WEB_URL + reverse("ldt.ldt_utils.views.save_ldtProject")
    language_code = request.LANGUAGE_CODE[:2]
    
    ldt = get_object_or_404(Project, ldt_id=id)
    if ldt.state == 2: #published
        readonly = 'true'
    else:
        readonly = 'false'
    
    return render_to_response('ldt/ldt_utils/init_ldt_full.html', {'LDT_MEDIA_PREFIX': settings.LDT_MEDIA_PREFIX, 'colorurl': settings.LDT_MEDIA_PREFIX + 'swf/ldt/pkg/color.xml', 'i18nurl': settings.LDT_MEDIA_PREFIX + 'swf/ldt/pkg/i18n', 'language': language_code, 'baseurl': settings.LDT_MEDIA_PREFIX + 'swf/ldt/', 'url': urlStr, 'posturl': posturl, 'id': id, 'readonly': readonly}, context_instance=RequestContext(request))
    
def init(request, method, url):
    ldtgen = LdtUtils()

    doc = ldtgen.generateInit([url], 'ldt.ldt_utils.views.' + method, None)

    resp = HttpResponse(mimetype="text/xml")
    resp['Cache-Control'] = 'no-cache, must-revalidate'
    resp['Pragma'] = 'no-cache'

    resp.write(lxml.etree.tostring(doc, pretty_print=True, xml_declaration=True, encoding="utf-8")) 
    return resp
       
def ldtProject(request, id):
    resp = HttpResponse(mimetype="text/xml")
    resp['Cache-Control'] = 'no-cache, must-revalidate'
    resp['Pragma'] = 'no-cache'
    
    project = Project.objects.get(ldt_id=id) #@UndefinedVariable
    resp.write(project.ldt)
    return resp


def project_json_id(request, id):
    
    project = get_object_or_404(Project, ldt_id=id)

    return project_json(request, project, False)


def project_json_externalid(request, id):
        
    res_proj = get_list_or_404(Project.objects.order_by('-modification_date'), contents__external_id=id) #@UndefinedVariable
    
    return project_json(request, res_proj[0], False)



def project_json(request, project, serialize_contents=True):
    
    if not ldt_auth.checkAccess(request.user, project):
        return HttpResponseForbidden(_("You can not access this project"))
        
    mimetype = request.REQUEST.get("mimetype")
    if mimetype is None:
        mimetype = "application/json; charset=utf-8"
    else:
        mimetype = mimetype.encode("utf-8")
    if "charset" not in mimetype:
        mimetype += "; charset=utf-8" 
    resp = HttpResponse(mimetype=mimetype)
    resp['Cache-Control'] = 'no-cache, must-revalidate'
    resp['Pragma'] = 'no-cache'
    
    indent = request.REQUEST.get("indent")
    if indent is None:
        indent = settings.LDT_JSON_DEFAULT_INDENT
    else:
        indent = int(indent)
    
    callback = request.REQUEST.get("callback")
    escape_str = request.REQUEST.get("escape")
    escape_bool = False
    if escape_str:
        escape_bool = {'true': True, 'false': False, "0": False, "1": True}.get(escape_str.lower())
        
        
    ps = ProjectSerializer(project, serialize_contents)
    project_dict = ps.serialize_to_cinelab()
    
    json_str = simplejson.dumps(project_dict, ensure_ascii=False, indent=indent)
    
    if callback is not None:
        json_str = "%s(%s)" % (callback, json_str)
    
    if escape_bool:
        json_str = escape(json_str)
    
    resp.write(json_str)

    return resp

def project_annotations_rdf(request, ldt_id):

    project = Project.objects.get(ldt_id=ldt_id); #@UndefinedVariable
    
    if not ldt_auth.checkAccess(request.user, project):
        return HttpResponseForbidden(_("You can not access this project"))

    mimetype = request.REQUEST.get("mimetype")
    if mimetype is None:
        mimetype = "application/rdf+xml; charset=utf-8"
    else:
        mimetype = mimetype.encode("utf-8")
    if "charset" not in mimetype:
        mimetype += "; charset=utf-8" 
    resp = HttpResponse(mimetype=mimetype)
    resp['Cache-Control'] = 'no-cache, must-revalidate'
    resp['Pragma'] = 'no-cache'

    ps = ProjectSerializer(project, from_contents=False, from_display=True) 
    annotations = ps.getAnnotations(first_cutting=True)
    
    rdf_ns = u"http://www.w3.org/1999/02/22-rdf-syntax-ns#"
    dc_ns = u"http://purl.org/dc/elements/1.1/"
    rdf = u"{%s}" % rdf_ns
    dc = u"{%s}" % dc_ns
    nsmap = {u'rdf' : rdf_ns, u'dc':dc_ns}
    
    rdf_root = lxml.etree.Element(rdf + u"RDF", nsmap=nsmap)
    
    logging.debug("RDF annotations : " + repr(annotations)) #@UndefinedVariable
    
    for annotation in annotations:
        uri = u""
        if 'uri' in annotation and annotation['uri']:
            uri = unicode(annotation['uri'])
        annot_desc = lxml.etree.SubElement(rdf_root, rdf + u"Description")
        annot_desc.set(rdf + u'about', uri)
        if annotation['title']:
            lxml.etree.SubElement(annot_desc, dc + 'title').text = lxml.etree.CDATA(unicode(annotation['title']))
        if annotation['desc']:
            lxml.etree.SubElement(annot_desc, dc + 'description').text = lxml.etree.CDATA(unicode(annotation['desc']))
        if annotation['tags']:
            for tag in annotation['tags']:
                lxml.etree.SubElement(annot_desc, dc + 'subject').text = lxml.etree.CDATA(unicode(tag))
        lxml.etree.SubElement(annot_desc, dc + 'coverage').text = u"start=%s, duration=%s" % (annotation['begin'], annotation['duration'])
        
    resp.write(u"<?xml version=\"1.0\" encoding=\"utf-8\"?>\n")
    resp.write(u"<!DOCTYPE rdf:RDF PUBLIC \"-//DUBLIN CORE//DCMES DTD 2002/07/31//EN\" \"http://dublincore.org/documents/2002/07/31/dcmes-xml/dcmes-xml-dtd.dtd\">\n")
    
    resp.write(lxml.etree.tostring(rdf_root, xml_declaration=False, encoding="utf-8", pretty_print=True))

    return resp

def save_ldtProject(request):
    if request.method == "POST":
        ldt = request.POST['ldt']
        id = request.POST['id']
        ldtproject = Project.objects.get(ldt_id=id) #@UndefinedVariable

        #save xml ldt
        ldtproject.ldt = ldt


        doc = lxml.etree.fromstring(ldtproject.ldt.encode("utf-8"))
        result = doc.xpath("/iri/project")
        
        #set new title
        ldtproject.title = result[0].get("title")
        
        #get new content list
        new_contents = []
        result = doc.xpath("/iri/medias/media")
        for medianode in result:
            id = medianode.get("id")
            new_contents.append(id)
    
        #set new content list
        for c in ldtproject.contents.all():
            if not c.iri_id in new_contents:
                ldtproject.contents.remove(c)
    
        ldtproject.save()
    else:
        ldt = ''
        new_contents = []
    
    return render_to_response('ldt/ldt_utils/save_done.html', {'ldt': ldt, 'id':id, 'title':ldtproject.title, 'contents': new_contents}, context_instance=RequestContext(request))



@login_required
def publish(request, id, redirect=True):
    ldt = get_object_or_404(Project, ldt_id=id)
    ldt.state = 2 #published
    ldt.save()
    redirect = boolean_convert(redirect)
    if redirect:
        return HttpResponseRedirect(reverse("ldt.ldt_utils.views.list_ldt"))
    else:
        return HttpResponse(simplejson.dumps({'res':True, 'ldt': {'id': ldt.id, 'state':ldt.state, 'ldt_id': ldt.ldt_id}}, ensure_ascii=False), mimetype='application/json')
    
@login_required
def unpublish(request, id, redirect=True):
    ldt = get_object_or_404(Project, ldt_id=id)
    ldt.state = 1 #edition
    ldt.save()
    redirect = boolean_convert(redirect)
    if redirect:
        return HttpResponseRedirect(reverse("ldt.ldt_utils.views.list_ldt"))
    else:
        return HttpResponse(simplejson.dumps({'res':True, 'ldt': {'id': ldt.id, 'state':ldt.state, 'ldt_id': ldt.ldt_id}}, ensure_ascii=False), mimetype='application/json')
    

def index(request, url):
    
    urlStr = settings.WEB_URL + reverse("ldt_init", args=['ldt', url])
    language_code = request.LANGUAGE_CODE[:2]
    
    return render_to_response('ldt/ldt_utils/init_ldt.html', {'LDT_MEDIA_PREFIX': settings.LDT_MEDIA_PREFIX, 'colorurl': settings.LDT_MEDIA_PREFIX + 'swf/ldt/pkg/color.xml', 'i18nurl': settings.LDT_MEDIA_PREFIX + 'swf/ldt/pkg/i18n', 'language': language_code, 'baseurl': settings.LDT_MEDIA_PREFIX + 'swf/ldt/', 'url': urlStr, 'weburl':settings.WEB_URL + settings.BASE_URL}, context_instance=RequestContext(request))


def ldt(request, url, startSegment=None):
    
    resp = HttpResponse(mimetype="text/xml; charset=utf-8")
    resp['Cache-Control'] = 'no-cache'

    contentList = Content.objects.filter(iri_id=url) #@UndefinedVariable

    ldtgen = LdtUtils()
    ldtgen.generateLdt(contentList, file=resp, title=contentList[0].title, startSegment=startSegment)

    return resp


def loading(request):
    return render_to_response('ldt/ldt_utils/loading.html', context_instance=RequestContext(request))


@login_required
def create_project(request, iri_id):

    content = get_object_or_404(Content, iri_id=iri_id)
    contents = [ content, ]
    if request.method == "POST" :
        form = AddProjectForm(request.POST)
        if form.is_valid():
            user = request.user
            project = Project.create_project(title=form.cleaned_data['title'], user=user, contents=contents)
            # Modal window is not used with firefox
            is_gecko = ((request.META['HTTP_USER_AGENT'].lower().find("firefox")) > -1);
            if is_gecko :
                return HttpResponseRedirect(reverse('ldt.ldt_utils.views.indexProjectFull', args=[project.ldt_id]))
            else:
                return HttpResponseRedirect(reverse('ldt.ldt_utils.views.indexProject', args=[project.ldt_id]))
    else:
        form = AddProjectForm()
    # Modal window is not used with firefox, so we ask to submit the form in _parent in firefox case.
    target_parent = ((request.META['HTTP_USER_AGENT'].lower().find("firefox")) > -1);
    return render_to_response('ldt/ldt_utils/create_ldt.html', {'form':form, 'contents':contents, 'create_project_action':reverse("ldt.ldt_utils.views.create_project", args=[iri_id]), 'target_parent':target_parent}, context_instance=RequestContext(request))

@login_required
def update_project(request, ldt_id):

    project = get_object_or_404(Project, ldt_id=ldt_id)
    contents = project.contents.all()
    if request.method == "POST" :
        submit_action = request.REQUEST.get("submit_button", False)
        if submit_action == "prepare_delete":
            errors = []
            if project.state == 2:
                errors.append(_("the project %(title)s is published. please unpublish before deleting.") % {'title':project.title})
                message = _("can not delete the project. Please correct the following error")
                title = _('title error deleting project')
            else:
                message = _("please confirm deleting project %(title)s") % {'title':project.title}
                title = _("confirm deletion")
            return render_to_response('ldt/ldt_utils/error_confirm.html', {'errors':errors, 'message':message, 'title': title}, context_instance=RequestContext(request))
        elif submit_action == "delete":
            if project.state != 2:
                project.delete()
            form_status = 'deleted'
            form = AddProjectForm()
        else:
            form_status = 'saved'
            form = AddProjectForm(request.POST)
            if form.is_valid():
                if project.title != form.cleaned_data['title']:
                    project.title = form.cleaned_data['title']
                    ldt = lxml.etree.fromstring(project.ldt.encode("utf-8"))
                    res = ldt.xpath("/iri/project")
                    res[0].set("title", project.title)
                    project.ldt = lxml.etree.tostring(ldt, pretty_print=True)
                    project.save()
    else:
        form = AddProjectForm({'title':unicode(project.title)})
        form_status = 'none'
        
    return render_to_response('ldt/ldt_utils/create_ldt.html', {'form':form, 'form_status':form_status, 'ldt_id': ldt_id, 'contents':contents, 'create_project_action':reverse("ldt.ldt_utils.views.update_project", args=[ldt_id])}, context_instance=RequestContext(request))


@login_required
def copy_project(request, ldt_id):

    project = get_object_or_404(Project, ldt_id=ldt_id)
    if request.method == "POST" :
        form = CopyProjectForm(request.POST)
        if form.is_valid():
            user = request.user
            project = project.copy_project(title=request.POST['title'], user=user)
            return HttpResponseRedirect(reverse('ldt.ldt_utils.views.indexProject', args=[project.ldt_id]))
    else:
        form = CopyProjectForm
    return render_to_response('ldt/ldt_utils/copy_ldt.html', {'form':form, 'project':project}, context_instance=RequestContext(request))


def write_content_base(request, iri_id=None):

    if iri_id:        
        instance_content = Content.objects.get(iri_id=iri_id) #@UndefinedVariable
        instance_media = instance_content.media_obj
        logging.debug("write_content_base : valid form: for instance : media -> " + repr(instance_media) + " content : for instance : " + repr(instance_content)) #@UndefinedVariable
    else:
        logging.debug("No iri_id") #@UndefinedVariable
        instance_content = None
        instance_media = None
    
    form_status = 'none'        
    if request.method == "POST":
        
        if instance_content is not None:
            content_instance_val = model_to_dict(instance_content, exclude=ContentForm.Meta.exclude)
        else:
            content_instance_val = {}
        
        if instance_media is not None:
            media_instance_val = model_to_dict(instance_media, exclude=MediaForm.Meta.exclude)
        else:
            media_instance_val = {}
        #add prefix
        
        def add_prefix(dict, prefix):
            for key, value in dict.items():
                dict['%s-%s' % (prefix, key)] = value
                del(dict[key])
        
        add_prefix(content_instance_val, "content")
        add_prefix(media_instance_val, "media")        
        
        for k in request.POST.keys():
            value = request.POST.get(k)
            content_instance_val[k] = value
            media_instance_val[k] = value
        
        content_form = ContentForm(content_instance_val, prefix="content", instance=instance_content)
        media_form = MediaForm(media_instance_val, request.FILES, prefix="media", instance=instance_media)
        
        media_valid = media_form.is_valid()
        content_valid = content_form.is_valid()
        
        logging.debug("write_content_base : valid form: for instance : " + repr(instance_media) + " -> media " + str(media_valid) + " content : for instance : " + repr(instance_content) + " : " + str(content_valid)) #@UndefinedVariable
        
        if media_valid and content_valid :
            
            # see if media must be created
            cleaned_data = {}
            cleaned_data.update(media_form.cleaned_data)
            
            media_input_type = content_form.cleaned_data["media_input_type"]
            
            if media_input_type == "none":
                media = None
            elif media_input_type == "link":
                media = content_form.cleaned_data["media_obj"]
                created = False
            elif media_input_type == "create":
                del cleaned_data["media_file"]
                if not cleaned_data['videopath']:
                    cleaned_data['videopath'] = settings.STREAM_URL
                # if the source is already http:// or rtmp:// we don't have to add STREAM_URL
                if cleaned_data['src'].startswith("rtmp://") or cleaned_data['src'].startswith("http://"):
                    cleaned_data['videopath'] = ''
                media, created = Media.objects.get_or_create(src=cleaned_data['src'], defaults=cleaned_data) #@UndefinedVariable
            elif media_input_type == "url" or media_input_type == "upload" :                
                # copy file
                #complet src
                destination_file = None
                source_file = None
                try:
                    if media_input_type == "url":
                        url = cleaned_data["external_src_url"]
                        source_file = urllib2.urlopen(url)
                        source_filename = source_file.info().get('Content-Disposition', None)
                        if not source_filename:
                            source_filename = urlparse.urlparse(url).path.rstrip("/").split('/')[-1]
                    elif media_input_type == "upload":
                        #source_file = request.FILES['media-media_file']
                        # At this point the file has already be uploaded thanks to the upload view, and original file name is sent through a post var
                        source_filename = request.POST["media-local_file_name"]
                    
                    source_filename = ldt_utils_path.sanitize_filename(source_filename)
                    destination_filepath = os.path.join(settings.STREAM_PATH, source_filename)
                    base_source_filename = source_filename
                    extension = base_source_filename.split(".")[-1]
                    if extension == base_source_filename:
                        extension = ""
                        base_basename_filename = base_source_filename
                    else:
                        base_basename_filename = base_source_filename[:-1 * (len(extension) + 1)]
                    i = 0
                    init_path = destination_filepath
                    init_name = source_filename
                    while os.path.exists(destination_filepath):
                        base_source_filename = "%s.%d.%s" % (base_basename_filename, i, extension)
                        destination_filepath = os.path.join(settings.STREAM_PATH, base_source_filename)
                        i += 1
                    
                    if media_input_type == "url":
                        # we upload the file if we are in url case
                        destination_file = open(destination_filepath, "wb")
                        chunck = source_file.read(2048)
                        while chunck:
                            destination_file.write(chunck)
                            chunck = source_file.read(2048)
                        
                    elif media_input_type == "upload":
                        # Since the file names have been 1 iteration too far, we rebuild the existing file name
                        if i == 1 :
                            base_source_filename = init_name
                            destination_filepath = init_path
                        else :
                            i -= 2
                            base_source_filename = "%s.%d.%s" % (base_basename_filename, i, extension)
                            destination_filepath = os.path.join(settings.STREAM_PATH, base_source_filename)
                    
                    
                    src_prefix = settings.STREAM_SRC_PREFIX.rstrip("/")
                    if len(src_prefix) > 0:
                        cleaned_data["src"] = src_prefix + "/" + base_source_filename
                    else:
                        cleaned_data["src"] = base_source_filename
                    
                    
                except Exception as inst:
                    logging.debug("write_content_base : POST error when processing file:" + str(inst)) #@UndefinedVariable
                    form_status = "error"
                    #set error for form
                    if media_input_type == "url":
                        errors = media_form._errors.setdefault("external_src_url", ErrorList())
                        errors.append(_("Problem when downloading file from url : ") + url)
                    elif media_input_type == "upload":
                        errors = media_form._errors.setdefault("media_file", ErrorList())
                        errors.append(_("Problem when uploading file : ") + str(inst))
                finally:
                    if media_input_type == "url":
                        if destination_file:
                            destination_file.close()
                        if source_file:
                            source_file.close()
                
                if form_status != "error":
                    #try:
                    del cleaned_data["media_file"]
                    if not cleaned_data['videopath']:
                        cleaned_data['videopath'] = settings.STREAM_URL
                    mimetype = cleaned_data.get('mimetype_field', None)
                    if not mimetype:
                        mimetype = mimetypes.guess_type(cleaned_data['src'])
                    cleaned_data['mimetype_field'] = mimetype
                    media, created = Media.objects.get_or_create(src=cleaned_data['src'], defaults=cleaned_data) #@UndefinedVariable
                else:
                    media = None
                    

            if media and not created:
                for attribute in ('external_id', 'external_permalink', 'external_publication_url', 'external_src_url', 'media_creation_date', 'videopath', 'duration', 'description', 'title'):
                    setattr(media, attribute, cleaned_data.get(attribute))
                mimetype = cleaned_data.get('mimetype_field', None)
                if not mimetype:
                    mimetype = mimetypes.guess_type(media.src) 
                media.mimetype_field = mimetype
                    
                media.save()
            
            
            if form_status != "error":
                #try:
                content_defaults = {}
                content_defaults.update(content_form.cleaned_data)
                content_defaults['media_obj'] = media
                del content_defaults["media_input_type"]
                content, created = Content.objects.get_or_create(iri_id=content_form.cleaned_data['iri_id'], defaults=content_defaults) #@UndefinedVariable
                if not created:
                    
                    for attribute in ('iriurl', 'title', 'description', 'duration', 'content_creation_date', 'tags', 'media_obj'):
                        setattr(content, attribute, content_defaults[attribute])
                content.save()
                form_status = 'saved'
                media_form = MediaForm(instance=media, prefix="media")
                content_form = ContentForm(instance=content, prefix="content")
        else:
            form_status = 'error'
    else:
        form_status = 'empty'
        initial = { 'media_input_type':"link"}
        
        content_form = ContentForm(prefix="content", instance=instance_content, initial=initial)
        media_form = MediaForm(prefix="media", instance=instance_media)
        
        if instance_content is not None:
            content_form.media_input_type = "link"
    
    return content_form, media_form, form_status

@login_required
def write_content(request, iri_id=None):
    
    submit_action = request.REQUEST.get("submit_button", False) 

    if submit_action == "prepare_delete": 
        errors, titles = prepare_delete_content(request, iri_id)
        if errors and len(errors) > 0:
            message = ungettext("There is %(count)d error when deleting content", "There are %(count)d errors when deleting content", len(errors)) % { 'count': len(errors)}
            title_msg = _('title error deleting content')
        else:
            message = _("Confirm delete content %(titles)s") % { 'titles' : ",".join(titles) }
            title_msg = _("confirm delete content")
        return render_to_response('ldt/ldt_utils/error_confirm.html', {'errors':errors, 'message':message, 'title': title_msg}, context_instance=RequestContext(request))  
    elif submit_action == "delete":
        delete_content(request, iri_id)
        form_status = "deleted"
        content_form = ContentForm()
        media_form = MediaForm()
    else:
        content_form, media_form, form_status = write_content_base(request, iri_id)

    if iri_id:
        create_content_action = reverse('ldt.ldt_utils.views.write_content', kwargs={'iri_id':iri_id})
    else:
        create_content_action = reverse('ldt.ldt_utils.views.write_content')
    
    session_key = request.COOKIES[settings.SESSION_COOKIE_NAME]
    cookie_name = settings.SESSION_COOKIE_NAME
    
    return render_to_response('ldt/ldt_utils/create_content.html', {'content_form': content_form, 'media_form': media_form, 'form_status': form_status, 'create_content_action': create_content_action, 'iri_id': iri_id, 'session_key':session_key, 'cookie_name':cookie_name}, context_instance=RequestContext(request))

@login_required
def prepare_delete_content(request, iri_id=None):
    errors = []
    titles = []
    if not iri_id:
        iri_id = request.REQUEST.get("iri_id", None)
        
    if iri_id:
        for content in Content.objects.filter(iri_id=iri_id): #@UndefinedVariable
            titles.append(unicode(content.title))
            projects = content.project_set.all()
            projects_nb = len(projects)
            if projects_nb > 0:
                project_titles = map(lambda p: unicode(p.title), projects)
                errors.append(ungettext("Content '%(title)s' is referenced by this project : %(project_titles)s. Please delete it beforehand.", "Content '%(title)s' is referenced by %(count)d projects: %(project_titles)s. Please delete them beforehand.", projects_nb) % {'title':unicode(content.title), 'count':projects_nb, 'project_titles': ",".join(project_titles)})
 
        
    return errors, titles


@login_required
def delete_content(request, iri_id=None):
    if not iri_id:
        iri_id = request.REQUEST.get("iri_id", None)
        
    if iri_id:
        Content.objects.filter(iri_id=iri_id).delete() #@UndefinedVariable


def upload(request):
    if request.method == 'POST':
        for field_name in request.FILES:
            # We get the file name
            source_file = request.FILES[field_name]
            source_filename = source_file.name
            # We sanitize the file name : no space, only lower case.
            source_filename = ldt_utils_path.sanitize_filename(source_filename)
            destination_filepath = os.path.join(settings.STREAM_PATH, source_filename)
            base_source_filename = source_filename
            extension = base_source_filename.split(".")[-1]
            if extension == base_source_filename:
                extension = ""
                base_basename_filename = base_source_filename
            else:
                base_basename_filename = base_source_filename[:-1 * (len(extension) + 1)]
            i = 0
            # We search if there is already a file with the same name, and rename the target by name.X.ext
            while os.path.exists(destination_filepath):
                base_source_filename = "%s.%d.%s" % (base_basename_filename, i, extension)
                destination_filepath = os.path.join(settings.STREAM_PATH, base_source_filename)
                i += 1
                
            destination_file = open(destination_filepath, "wb")
            
            for chunk in source_file.chunks():
                destination_file.write(chunk)
            destination_file.close()
            

        # indicate that everything is OK for SWFUpload
        return HttpResponse("ok", mimetype="text/plain")
    else:
        return HttpResponse("notok", mimetype="text/plain")
    
