from django.conf import settings
from django.contrib.auth.decorators import login_required
from django.contrib.auth.models import Group, User
from django.core.urlresolvers import reverse
from django.http import HttpResponseServerError
from django.shortcuts import render_to_response, get_object_or_404
from django.template import RequestContext
from guardian.shortcuts import get_objects_for_group
from ldt.ldt_utils.models import Content, Project
from ldt.ldt_utils.views.workspace import search_index as ws_search_index, search_listing as ws_search_listing
from ldt.security.utils import add_change_attr


@login_required
def front_home(request):
    # Get the 3 last annotated contents
    last_contents = Content.objects.order_by('-last_annotated')[:3]
    # Get the most annotated contents
    most_contents = Content.objects.order_by('-nb_annotation')[:8]
    # Get the active groups
    active_groups = Group.objects.exclude(name=settings.PUBLIC_GROUP_NAME)[:5]
    
    
    is_gecko = ((request.META['HTTP_USER_AGENT'].lower().find("firefox")) > -1);

    return render_to_response("front/front_home.html",
                              {'last_contents': last_contents, 'most_contents':most_contents, 'active_groups':active_groups,
                               'is_gecko': is_gecko},
                              context_instance=RequestContext(request))


@login_required
def group_info(request, group_id):
    # Get the active group
    group = Group.objects.get(id=group_id)
    # list of contents annotated by the group (or everyone)
    everyone = Group.objects.get(name=settings.PUBLIC_GROUP_NAME)
    content_list = get_objects_for_group(group, 'ldt_utils.view_content') | get_objects_for_group(everyone, 'ldt_utils.view_content')
    # Group's users
    users = User.objects.filter(groups__in=[group])
    
    is_gecko = ((request.META['HTTP_USER_AGENT'].lower().find("firefox")) > -1);

    return render_to_response("front/front_group.html",
                              {'group': group, 'content_list':content_list, 'users':users,
                               'is_gecko': is_gecko},
                              context_instance=RequestContext(request))


@login_required
def all_contents(request):
    # Get all the public contents group
    content_list = add_change_attr(request.user, Content.safe_objects.all())
    
    is_gecko = ((request.META['HTTP_USER_AGENT'].lower().find("firefox")) > -1);

    return render_to_response("front/front_all_contents.html",
                              {'content_list':content_list, 
                               'is_gecko': is_gecko},
                              context_instance=RequestContext(request))


@login_required
def annot_content(request, content_iri_id, cutting_id=None):
    # Get the wanted content
    content = Content.objects.get(iri_id=content_iri_id)
    # Get the content src to see if it is a youtube video
    youtube_url = None
    if content.src is not None and "youtube.com" in content.src:
        youtube_url = content.src
    # The main project for the content
    proj = Project.objects.filter(contents__in=[content])[0]
    # Vars for player
    player_id = "player_project_" + proj.ldt_id
    
    json_url = reverse("ldt.ldt_utils.views.json.project_json_id", args=[proj.ldt_id])
    player_width = 550
    player_height = 380
    stream_mode = proj.stream_mode
    if stream_mode != "video":
        stream_mode = 'radio'
    
    is_gecko = ((request.META['HTTP_USER_AGENT'].lower().find("firefox")) > -1);


    return render_to_response("front/front_player.html",
                              {'content': content, 'project':proj, 'player_id': player_id, 
                              'json_url': json_url, 'player_width':player_width, 'player_height':player_height, 'stream_mode':stream_mode, 'youtube_url':youtube_url,
                               'is_gecko': is_gecko},
                              context_instance=RequestContext(request))


@login_required
def search_index(request):
    return ws_search_index(request, front_template=True)


@login_required
def search_listing(request):
    return ws_search_listing(request, front_template=True)


    
