from django import forms
from django.utils.translation import ugettext_lazy as _
from django.contrib.auth.models import Group
from ldt.forms import widgets as ldt_widgets
from ldt.security.forms import ShareForm
from models import Project, Content, Media
from utils import generate_uuid

class LdtImportForm(forms.Form):
    importFile = forms.FileField()
    videoPath = forms.CharField(required=False)
    flatten = forms.BooleanField(required=False, initial=True)   
    
class LdtAddForm(ShareForm):
    title = forms.CharField(required=True)
    contents = forms.ModelMultipleChoiceField(Content.objects.all())
    description = forms.CharField(widget=forms.Textarea, required=False)
    
    class Meta:
        model = Project
        exclude = ("ldt_id", "ldt", "created_by", "changed_by", "creation_date", "modification_date", "state", "owner")   

class ReindexForm(forms.Form):
    contents = forms.ModelMultipleChoiceField(Content.objects.all())
    index_projects = forms.BooleanField(required=False, initial=False)
    
class StatAnnotationForm(forms.Form):
    projects = forms.ModelMultipleChoiceField(Project.objects.all())
    choose_all = forms.BooleanField(required=False, initial=False)

class SearchForm(forms.Form):
    search = forms.CharField(label=_("Search"))
    field = forms.ChoiceField([(u"all", _("all")), (u"title", _("title")), (u"abstract", _("resume")), (u"tags", _("tags"))], label=_("Fields"))
    ldt_pres = forms.BooleanField(required=False, initial=True, label=_("Display the results in Lignes De Temps"))

class AddProjectForm (ShareForm):
    title = forms.CharField(widget=forms.TextInput(attrs={'class':'inputbox required'}))
    description = forms.CharField(widget=forms.Textarea, required=False)
    
    class Meta:
        model = Project
        exclude = ("ldt_id", "ldt", "created_by", "changed_by", "creation_date", "modification_date", "state", "owner", "contents")   

class CopyProjectForm (forms.Form):
    title = forms.CharField()
    group = forms.IntegerField(required=False)    

class ContentForm(ShareForm):
    iriurl = forms.CharField(max_length=1024, widget=forms.HiddenInput, required=False)
    content_creation_date = forms.SplitDateTimeField(widget=ldt_widgets.LdtSplitDateTime, required=False, label=_("content.content_creation_date"))
    media_input_type = forms.ChoiceField(required=False, label=_("content.media_input_type"), choices=(("upload", _("file_upload")), ("url", _("url")), ("link", _("existing_media")), ("create", _("create_media")), ("none", _("none_media"))))
    groups = forms.ModelMultipleChoiceField(Group.objects.all(), required=False)
    is_public = forms.BooleanField(required=False)  
        
    def clean_iri_id(self):
        data = self.cleaned_data.get('iri_id')
        if not data:
            data = generate_uuid()
        return data
    
    def clean(self):
        super(ContentForm, self).clean()
        cleaned_data = self.cleaned_data
        iriurl_data = cleaned_data.get('iriurl')
        iri_id_data = cleaned_data['iri_id']
        if not iriurl_data:
            iriurl_data = "%s/%s.iri" % (iri_id_data, iri_id_data)    
        cleaned_data['iriurl'] = iriurl_data
        return cleaned_data
    
    class Meta:
        model = Content
        exclude = ("creation_date", "update_date", "authors")
        widgets = {
                   'iri_id': forms.HiddenInput,
        }
        
    class Media:
        css = {
            'all' : ('admin/css/forms.css', 'admin/css/base.css', 'admin/css/widgets.css')
               }
        
class MediaForm(forms.ModelForm):
    
    media_creation_date = forms.SplitDateTimeField(widget=ldt_widgets.LdtSplitDateTime, required=False)
    media_file = forms.FileField(required=False);
    src = forms.CharField(required=False, max_length=1024)

    class Meta:
        model = Media
        exclude = ("creation_date", "update_date", "creator")
    
    class Media:
        css = {
            'all' : ('admin/css/forms.css', 'admin/css/base.css', 'admin/css/widgets.css')
               }
        
class GroupAddForm(ShareForm):
    name = forms.CharField(required=True)
    description = forms.CharField(widget=forms.Textarea, required=False)
    
    class Meta:
        model = Group
