"""
This file demonstrates two different styles of tests (one doctest and one
unittest). These will both pass when you run "manage.py test".

Replace these with more appropriate tests for your application.
"""

from django.contrib.auth import get_user_model
from django.core.management import call_command
from ldt.ldt_utils.models import Project, Content
from ldt.ldt_utils.utils import LdtUtils, LdtAnnotation, create_ldt
from ldt.test.client import Client
from ldt.test.testcases import TestCase
import lxml.etree
import uuid

User = get_user_model()

class ProjectTest(TestCase):
    
    fixtures = ['base_data.json', 'user_data.json']
    
    def setUp(self):
        self.client = Client()
        User.objects.create_superuser('blop', 'blop@blop.com', 'blop')
        
        _ = self.client.login(username='blop', password='blop')
        self.user = User()
        self.user.username = 'blop'
        self.LU = LdtUtils()
        self.project = Project(title="titleproj1", owner=self.user)
        self.project.ldt = '<iri ldtversion="1.0.3" xmlns:dc="http://dublincore.org/documents/dcmi-namespace/"><project id="af3b99e4-b695-11df-bfde-00145ea4a2be" user="admin" title="CA:reponse a TC" abstract=""/> <medias> <media extra="" id="laurentcantet_entrelesmurs" pict="" src="http://amateur.iri.centrepompidou.fr//atelier/static/media/ldt/laurentcantet_entrelesmurs/laurentcantet_entrelesmurs.iri" video="rtmp://media.iri.centrepompidou.fr/ddc_player/video/regardssignes/"/> </medias> <annotations> <content id="laurentcantet_entrelesmurs"> <ensemble id="ens_perso" idProject="fe0d5d4c-2201-11df-8a24-00145ea4a2be" title="Decoupages personnels" author="perso" abstract=""> <decoupage id="c_EFC3FFE7-0204-A086-EBEC-D2A03A0E56CB" author="perso"> <title>CA: prof et admin</title> <abstract/> <elements> <element id="s_442AAB3A-42DA-F9BF-75E7-D2A0663FD5FF" begin="985690" dur="373222" author="" date="2010/09/02" color="16711680" src=""> <title/> <abstract/> <audio source=""/> <tags/> </element> <element id="s_0050F043-3AD2-0A7C-6699-D2A03A1EBA02" begin="5052858" dur="124407" author="" date="2010/09/02" color="10053375" src=""> <title>conseil de classe</title> <abstract>Reprise de la figure precedente</abstract> <audio source="undefined"/> <tags/> </element> </elements> </decoupage> <decoupage id="c_EEEF5C29-86E1-4AAE-E068-04EB5B00E492" author="perso"> <title>TC: prof et admin</title> <abstract/> <elements> <element id="s_880D9D4B-8BC0-BA43-5ECA-04EBA9FC9E59" begin="2426894" dur="141478" author="" date="2010/02/25" color="10053375" src=""> <title>Conseil de classe</title> <abstract/> <audio source=""/> <tags> <tag>Argumentation</tag> </tags> </element> <element id="s_D568A57C-7110-DED2-3165-04EC54387060" begin="5052858" dur="124407" author="" date="2010/02/25" color="10053375" src=""> <title>conseil de classe</title> <abstract>Reprise de la figure precedente</abstract> <audio source="undefined"/> <tags/> </element> </elements> </decoupage> </ensemble> </content> </annotations> <displays> <display id="0" title="Init view" idsel="laurentcantet_entrelesmurs" tc="2426424" zoom="0" scroll="0" infoBAB=""> <audio source=""/> <content id="laurentcantet_entrelesmurs"> <decoupage idens="en_2" id="de_PPP" tagsSelect=""/> <decoupage idens="laurentcantet_entrelesmurs" id="c_14A2E638-1936-97DC-E303-2DBA6A82A8B3" tagsSelect=""/> <decoupage idens="ens_perso" id="c_EEEF5C29-86E1-4AAE-E068-04EB5B00E492" tagsSelect=""/> <decoupage idens="ens_perso" id="c_EFC3FFE7-0204-A086-EBEC-D2A03A0E56CB" tagsSelect=""/> </content> </display> </displays> <edits> <editing id="0" tags=""> <title>Bout a bout 1</title> <abstract/> <edit id="edit1" tags=""> <eList/> <caption/> <audio/> <mList/> </edit> <edit id="edit2" tags=""> <eList/> <caption/> <audio/> <mList/> </edit> </editing> </edits> </iri>'
        self.project.id = "11321"
        self.project.ldt_id = str(uuid.uuid1())
        self.project.description = "proj1description"
        self.project.save()
        
    def test_create_project(self):   
        self.cont1 = Content(iriurl="cont1_id/iriurl1", duration=123)
        self.cont1.iri_id = "cont1_id"
        self.cont1.save()
        
        self.cont2 = Content(iriurl="cont2_id/iriurl2", duration=100)
        self.cont2.iri_id = "cont2_id"
        self.cont2.save()
        
        self.project2 = Project(title="titleproj2", owner=self.user)
    
        self.project2.ldt = self.LU.generate_ldt(Content.objects.all())    
        self.project2.ldt_id = str(uuid.uuid1())
        self.project2.description = "proj2description"
        self.project2.save()
        
        self.assertEqual(Project.objects.get(ldt_id=self.project2.ldt_id), self.project2)
        
    #test deletion of project without annotation
    def test_del_project_v1(self):
        self.cont3 = Content(iriurl="cont3_id/iriurl3", duration=100)
        self.cont3.iri_id = "cont3_id"
        self.cont3.save()
        
        self.project3 = Project(title="titleproj3", owner=self.user)
        self.project3.ldt = self.LU.generate_ldt(Content.objects.all())  
        self.project3.id = "333"
        self.project3.ldt_id = str(uuid.uuid1())
        self.project3.description = "proj3description"
        self.project3.save()
        
        self.project3.delete()

        with self.assertRaises(Project.DoesNotExist):
            Project.objects.get(ldt_id=self.project3.ldt_id)
    
    #test deletion of project with annotations  
    def test_del_project_v2(self):
        self.cont4 = Content(iriurl="cont4_id/iriurl4", duration=100)
        self.cont4.iri_id = "cont4_id"
        self.cont4.save()
        
        self.project4 = Project(title="titleproj4", owner=self.user)
        self.project4.ldt=self.LU.generate_ldt(Content.objects.all())  
        self.project4.id = "444"
        self.project4.ldt_id = str(uuid.uuid1())
        self.project4.description = "proj4description"
        self.project4.save()
        
        create_ldt(self.project4, self.user)
        
        self.LA = LdtAnnotation(self.project4)

        self.LA.add("id11", "cutting_id", "cutting_title", "title", "text", ["tag1", "tag2"], "800",
                    "10000", "jdoe", "2011-09-10T09:12:58")
        self.LA.save()
        
        self.project4.delete()

        with self.assertRaises(Project.DoesNotExist):
            Project.objects.get(ldt_id=self.project4.ldt_id)
    
    def test_clean_database_project(self) :
        self.cont5 = Content(iriurl="cont5_id/iriurl5", duration=123)
        self.cont5.iri_id = "cont5_id"
        self.cont5.save()
        
        self.cont6 = Content(iriurl="cont6_id/iriurl6", duration=100)
        self.cont6.iri_id = "cont6_id"
        self.cont6.save()
        
        self.cont7 = Content(iriurl="cont7_id/iriurl7", duration=100)
        self.cont7.iri_id = "cont7_id"
        self.cont7.save()
        
        self.project.contents.add(self.cont5, self.cont6)
    
        doc = self.LU.generate_ldt(Content.objects.all())
        
        #project5 : valid project
        self.project5 = Project(title="titleproj5", owner=self.user)
        ldt = lxml.etree.tostring(doc, pretty_print = False)
        self.project5.ldt = ldt
        self.project5.id = "555"
        self.project5.ldt_id = str(uuid.uuid1())
        self.project5.description = "proj5description"
        self.project5.save()
        
        #project6 : project with empty ldt
        self.project6 = Project(title="titleproj5", owner=self.user)
        self.project6.id = "666"
        self.project6.ldt_id = str(uuid.uuid1())
        self.project6.description = "proj6description"
        self.project6.save()
        
        #project7 : project with a non-existing media
        
        self.project7 = Project(title="titleproj7", owner=self.user)
        self.project7.id = "777"
        self.project7.ldt_id = str(uuid.uuid1())
        self.project7.ldt = '<iri><project id="a18ec3de-49c1-11e2-8e5d-00161798aedb" title="" user="IRI Web" abstract=""/><medias><media id="non_existing_media" src="ldt/non_existing_media/non_existing_media_iri_url" video="%(stream_url)s" pict="" extra=""/></medias><annotations/><displays><display id="0" title="generated" idsel="cont5_id" tc="0"><content id="cont5_id"/><content id="cont6_id"/><content id="cont7_id"/></display></displays><edits/></iri>'
        self.project7.description = "proj7description"
        self.project7.save()
        
        #project8 : project with multiple medias some valid and some that don't exist
        self.project8 = Project(title="titleproj8", owner=self.user)
        self.project8.id = "888"
        self.project8.ldt_id = str(uuid.uuid1())
        self.project8.ldt = '<iri><project id="a18ec3de-49c1-11e2-8e5d-00161798aedb" title="" user="IRI Web" abstract=""/><medias><media id="cont5_id" src="ldt/cont5_id/iriurl5" video="%(stream_url)s" pict="" extra=""/><media id="non_existing_media" src="ldt/non_existing_media/non_existing_media_iri_url" video="%(stream_url)s" pict="" extra=""/><media id="cont6_id" src="ldt/cont6_id/iriurl6" video="%(stream_url)s" pict="" extra=""/><media id="cont7_id" src="ldt/cont7_id/iriurl7" video="%(stream_url)s" pict="" extra=""/></medias><annotations/><displays><display id="0" title="generated" idsel="cont5_id" tc="0"><content id="cont5_id"/><content id="cont6_id"/><content id="cont7_id"/></display></displays><edits/></iri>'
        self.project8.description = "proj8description"
        self.project8.save()
        
        #project9 : project without any media
        self.project9 = Project(title="titleproj9", owner=self.user)
        self.project9.id = "999"
        self.project9.ldt_id = str(uuid.uuid1())
        self.project9.ldt = '<iri><project id="a18ec3de-49c1-11e2-8e5d-00161798aedb" title="" user="IRI Web" abstract=""/><medias></medias><annotations/><displays><display id="0" title="generated" idsel="cont5_id" tc="0"><content id="cont5_id"/><content id="cont6_id"/><content id="cont7_id"/></display></displays><edits/></iri>'
        self.project9.description = "proj9description"
        self.project9.save()
        
        call_command('set_projectldtiri')

        with self.assertRaises(Project.DoesNotExist):
            Project.objects.get(ldt_id=self.project6.ldt_id)
        
        with self.assertRaises(Project.DoesNotExist):
            Project.objects.get(ldt_id=self.project7.ldt_id)
        
        with self.assertRaises(Project.DoesNotExist):
            Project.objects.get(ldt_id=self.project9.ldt_id)
        
        self.assertEqual(ldt, self.project5.ldt)    
        