<a class="github" href="exceptions.py"></a>

# Exceptions

> Exceptions… allow error handling to be organized cleanly in a central or high-level place within the program structure.
>
> &mdash; Doug Hellmann, [Python Exception Handling Techniques][cite]

## Exception handling in REST framework views

REST framework's views handle various exceptions, and deal with returning appropriate error responses.

The handled exceptions are:

* Subclasses of `APIException` raised inside REST framework.
* Django's `Http404` exception.
* Django's `PermissionDenied` exception.

In each case, REST framework will return a response with an appropriate status code and content-type.  The body of the response will include any additional details regarding the nature of the error.

By default all error responses will include a key `detail` in the body of the response, but other keys may also be included.

For example, the following request:

    DELETE http://api.example.com/foo/bar HTTP/1.1
    Accept: application/json

Might receive an error response indicating that the `DELETE` method is not allowed on that resource:

    HTTP/1.1 405 Method Not Allowed
    Content-Type: application/json
    Content-Length: 42

    {"detail": "Method 'DELETE' not allowed."}

## Custom exception handling

You can implement custom exception handling by creating a handler function that converts exceptions raised in your API views into response objects.  This allows you to control the style of error responses used by your API.

The function must take a single argument, which is the exception to be handled, and should either return a `Response` object, or return `None` if the exception cannot be handled.  If the handler returns `None` then the exception will be re-raised and Django will return a standard HTTP 500 'server error' response.

For example, you might want to ensure that all error responses include the HTTP status code in the body of the response, like so:

    HTTP/1.1 405 Method Not Allowed
    Content-Type: application/json
    Content-Length: 62

    {"status_code": 405, "detail": "Method 'DELETE' not allowed."}

In order to alter the style of the response, you could write the following custom exception handler:

    from rest_framework.views import exception_handler

    def custom_exception_handler(exc):
        # Call REST framework's default exception handler first,
        # to get the standard error response.
        response = exception_handler(exc)

        # Now add the HTTP status code to the response.
        if response is not None:
            response.data['status_code'] = response.status_code

        return response

The exception handler must also be configured in your settings, using the `EXCEPTION_HANDLER` setting key. For example:

    REST_FRAMEWORK = {
        'EXCEPTION_HANDLER': 'my_project.my_app.utils.custom_exception_handler'
    }

If not specified, the `'EXCEPTION_HANDLER'` setting defaults to the standard exception handler provided by REST framework:

    REST_FRAMEWORK = {
        'EXCEPTION_HANDLER': 'rest_framework.views.exception_handler'
    }

Note that the exception handler will only be called for responses generated by raised exceptions.  It will not be used for any responses returned directly by the view, such as the `HTTP_400_BAD_REQUEST` responses that are returned by the generic views when serializer validation fails.

---

# API Reference

## APIException

**Signature:** `APIException()`

The **base class** for all exceptions raised inside REST framework.

To provide a custom exception, subclass `APIException` and set the `.status_code` and `.default_detail` properties on the class.

For example, if your API relies on a third party service that may sometimes be unreachable, you might want to implement an exception for the "503 Service Unavailable" HTTP response code.  You could do this like so:

    from rest_framework.exceptions import APIException

    class ServiceUnavailable(APIException):
        status_code = 503
        default_detail = 'Service temporarily unavailable, try again later.'

## ParseError

**Signature:** `ParseError(detail=None)`

Raised if the request contains malformed data when accessing `request.DATA` or `request.FILES`.

By default this exception results in a response with the HTTP status code "400 Bad Request".

## AuthenticationFailed

**Signature:** `AuthenticationFailed(detail=None)`

Raised when an incoming request includes incorrect authentication.

By default this exception results in a response with the HTTP status code "401 Unauthenticated", but it may also result in a "403 Forbidden" response, depending on the authentication scheme in use.  See the [authentication documentation][authentication] for more details.

## NotAuthenticated

**Signature:** `NotAuthenticated(detail=None)`

Raised when an unauthenticated request fails the permission checks.

By default this exception results in a response with the HTTP status code "401 Unauthenticated", but it may also result in a "403 Forbidden" response, depending on the authentication scheme in use.  See the [authentication documentation][authentication] for more details.

## PermissionDenied

**Signature:** `PermissionDenied(detail=None)`

Raised when an authenticated request fails the permission checks.

By default this exception results in a response with the HTTP status code "403 Forbidden".

## MethodNotAllowed

**Signature:** `MethodNotAllowed(method, detail=None)`

Raised when an incoming request occurs that does not map to a handler method on the view.

By default this exception results in a response with the HTTP status code "405 Method Not Allowed".

## UnsupportedMediaType

**Signature:** `UnsupportedMediaType(media_type, detail=None)`

Raised if there are no parsers that can handle the content type of the request data when accessing `request.DATA` or `request.FILES`.

By default this exception results in a response with the HTTP status code "415 Unsupported Media Type".

## Throttled

**Signature:** `Throttled(wait=None, detail=None)`

Raised when an incoming request fails the throttling checks.

By default this exception results in a response with the HTTP status code "429 Too Many Requests".

[cite]: http://www.doughellmann.com/articles/how-tos/python-exception-handling/index.html
[authentication]: authentication.md
