# -*- coding: utf-8 -*-

from django.db import models
from p4l.models.common import P4lModel, P4lModelLang


class Imprint(P4lModelLang):
    record = models.ForeignKey('p4l.Record', related_name="imprints")
    imprintCity = models.CharField(max_length=512, blank=True, null=True, db_index=True)
    publisher = models.CharField(max_length=512, blank=True, null=True, db_index=True)
    imprintDate = models.CharField(max_length=512, blank=True, null=True, db_index=True)    


class Serie(P4lModel):
    title = models.CharField(max_length=2048, blank=False, null=False, db_index=True)
    volume = models.CharField(max_length=2048, blank=True, null=True, db_index=True)
    class Meta(P4lModel.Meta):
        unique_together = ['title', 'volume']

class RecordSerie(P4lModelLang):
    record = models.ForeignKey('p4l.Record', related_name="series+")
    serie = models.ForeignKey('p4l.Serie', related_name="records")
    class Meta(P4lModelLang.Meta):
        unique_together = ['record','serie']



class ProjectName(P4lModel):
    label = models.CharField(max_length=2048, blank=False, null=False, db_index=True)
    acronym = models.CharField(max_length=2048, blank=True, null=True, db_index=True) #iiep:acronym
    class Meta(P4lModel.Meta):
        unique_together = ['label', 'acronym']


class CorporateAuthor(P4lModel):
    label = models.CharField(max_length=2048, blank=False, null=False, db_index=True)
    acronym = models.CharField(max_length=2048, blank=True, null=True, db_index=True) #iiep:acronym
    class Meta(P4lModel.Meta):
        unique_together = ['label', 'acronym']


class Url(P4lModel):
    record = models.ForeignKey('p4l.Record', related_name="urls")
    address = models.CharField(max_length=2048, blank=False, null=False, db_index=True) #iiep:address
    display = models.CharField(max_length=2048, blank=True, null=True, db_index=True) #iiep:display
    accessLevel = models.CharField(max_length=512, blank=True, null=True, db_index=True) #iiep:accessLevel


class Subject(P4lModel):
    subject = models.URLField(max_length=2048, unique=True, db_index=True)

class Theme(P4lModel):
    theme = models.URLField(max_length=2048, unique=True, db_index=True)


class Country(P4lModel):
    country = models.URLField(max_length=2048, unique=True, db_index=True)


class Isbn(P4lModelLang):
    record = models.ForeignKey('p4l.Record', related_name="isbns")
    isbn = models.CharField(max_length=128) #iiep:isbn


class Issn(P4lModelLang):
    record = models.ForeignKey('p4l.Record', related_name="issns")
    issn = models.CharField(max_length=128) #iiep:issn

class DocumentCode(P4lModelLang):
    record = models.ForeignKey('p4l.Record', related_name="documentCodes")
    documentCode = models.CharField(max_length=128) #iiep:issn

class Language(P4lModel):
    language = models.URLField(max_length=2048, unique=True, db_index=True)



class BaseTitle(P4lModelLang):
    title = models.CharField(max_length=2048, blank=False, null=False, db_index=True)
    class Meta(P4lModelLang.Meta):
        abstract = True

class Title(BaseTitle):
    record = models.ForeignKey('p4l.Record', related_name="titles")

class AddedTitle(BaseTitle):
    record = models.ForeignKey('p4l.Record', related_name="addedTitles")

class TitleMainDocument(BaseTitle):
    record = models.ForeignKey('p4l.Record', related_name="titlesMainDocument")


class Abstract(P4lModelLang):
    record = models.ForeignKey('p4l.Record', related_name="abstracts")
    abstract = models.TextField(blank=True, null=True)


class Collation(P4lModelLang):
    record = models.ForeignKey('p4l.Record', related_name="collations")
    collation = models.CharField(max_length=1024, blank=False, null=False, db_index=True)


class VolumeIssue(P4lModelLang):
    record = models.ForeignKey('p4l.Record', related_name="volumeIssues")
    volume = models.CharField(max_length=1024, blank=True, null=True, db_index=True) #iiep:volume
    number = models.CharField(max_length=1024, blank=True, null=True, db_index=True) #iiep:number


class P4lPerson(P4lModel):
    name = models.CharField(max_length=2048, blank=False, null=False, db_index=True, unique=True)
    class Meta(P4lModel.Meta):
        abstract = True


class Author(P4lPerson):
    pass
    
class SubjectPerson(P4lPerson):
    pass


class Periodical(P4lModel):
    label = models.CharField(max_length=2048, blank=False, null=False, db_index=True, unique=True) #iiep:periodical

class RecordPeriodical(P4lModelLang):
    record = models.ForeignKey('p4l.Record', related_name="periodicals+")
    periodical = models.ForeignKey('p4l.Periodical', related_name="records")
    class Meta(P4lModelLang.Meta):
        unique_together = ['record','periodical']


class BaseMeeting(P4lModel):
    label = models.CharField(max_length=2048, blank=False, null=False, db_index=True) #rdfs:label
    meetingNumber = models.CharField(max_length=2048, blank=True, null=True, db_index=True) #iiep:meetingNumber
    meetingPlace = models.CharField(max_length=2048, blank=True, null=True, db_index=True) #iiep:meetingPlace
    meetingDate = models.CharField(max_length=2048, blank=True, null=True, db_index=True) #iiep:meetingDate
    meetingYear = models.PositiveSmallIntegerField(blank=True, null=True, db_index=True) #iiep:meetingYear
    class Meta(P4lModel.Meta):
        abstract = True
        unique_together = ['label', 'meetingNumber', 'meetingPlace', 'meetingDate', 'meetingYear']


class Meeting(BaseMeeting):
    pass

class RecordMeeting(P4lModelLang):
    record = models.ForeignKey('p4l.Record', related_name="meetings+")
    meeting = models.ForeignKey('p4l.Meeting', related_name="records")
    class Meta(P4lModelLang.Meta):
        unique_together = ['record','meeting']


class SubjectMeeting(BaseMeeting):
    pass


class CorporateBody(P4lModel):
    label = models.CharField(max_length=2048, blank=False, null=False, db_index=True) #rdfs:label
    acronym = models.CharField(max_length=2048, blank=True, null=True, db_index=True) #iiep:acronym
    class Meta(P4lModel.Meta):
        unique_together = ['label','acronym']

class Record(P4lModel):
    uri = models.URLField(max_length=2048, unique=True, db_index=True) #subject
    subjects = models.ManyToManyField('p4l.Subject') #dct:subject
    themes = models.ManyToManyField('p4l.Theme') #iiep:theme
    countries = models.ManyToManyField('p4l.Country') #iiep:country
    identifier = models.CharField(max_length=128, unique=True, db_index=True) #dct:identifier    
    notes = models.TextField(blank=True, null=True) #iiep:notes    
    #issns foreign key from Isbn #iiep:issn
    #isbns foreign key from Isbn #iiep:isbn
    #documentCodes foreign key from Isbn #iiep:documentCode
    language = models.ForeignKey('p4l.Language', blank=True, null=True) #dct:language
    otherLanguages = models.ManyToManyField('p4l.Language', related_name='otherLanguage_record') #iiep:otherLanguage
    #titles foreign Key from Title #dct:title
    #addedTitles foreign Key from AddedTitle #iiep:addedTitle
    #titlesMainDocument foreign Key from TitleMainDocument #iiep:titleMainDocument
    editionStatement = models.CharField(max_length=1024, blank=True, null=True) #iiep:editionStatement
    #imprints foreign Key from Imprint #iiep:imprint
    #collations = foreign Key from Collation #iiep:collation
    #volumeIssues = foreign Key from VolumeIssue #iiep:volumeIssue
    projectNames = models.ManyToManyField('p4l.ProjectName') #iiep:projectName
    periodicals = models.ManyToManyField('p4l.Periodical', through='p4l.RecordPeriodical') #iiep:periodical
    meetings = models.ManyToManyField('p4l.Meeting', through='p4l.RecordMeeting') #iiep:meeting
    series = models.ManyToManyField('p4l.Serie', through='p4l.RecordSerie')  #iiep:serie
    authors = models.ManyToManyField('p4l.Author') #iiep:author
    subjectPersons = models.ManyToManyField('p4l.SubjectPerson') #iiep:subjectPerson
    subjectCorporateBodies = models.ManyToManyField('p4l.CorporateBody') #iiep:subjectCorporateBody
    subjectMeetings = models.ManyToManyField('p4l.SubjectMeeting') #iiep:subjectMeeting
    corporateAuthors = models.ManyToManyField('p4l.CorporateAuthor') #iiep:corporateAuthor
    #urls foreign Key from Url #iiep:url
    recordType = models.URLField(max_length=2048) #dct:type
    isDocumentPart = models.BooleanField() #iiep:isDocumentPart

    def __unicode__(self):
        return "Record id %s { identifier: %s, uri: %s, editionStatement: %s,  recordType: %s, isDocumentPart: %s, notes: %s, language : %s}" \
            % (
                self.id,
                self.identifier,
                self.uri,
                self.editionStatement,
                self.recordType,
                self.isDocumentPart,
                self.notes[:100] if self.notes else None,
                self.language.id if self.language else None
            )

#('{http://www.iiep.unesco.org/plan4learning/model.owl#}issn', 3)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}documentCode', 3)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}country', 44)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}imprint', 5)
#('{http://purl.org/dc/terms/}title', 4)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}isDocumentPart', 1)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}notes', 1)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}isbn', 8)
#('{http://purl.org/dc/terms/}identifier', 1)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}meeting', 4)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}projectName', 10)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}addedTitle', 18)
#('{http://purl.org/dc/terms/}subject', 29)
#('{http://purl.org/dc/terms/}language', 1)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}serie', 4)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}volumeIssue', 3)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}url', 20)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}titleMainDocument', 3)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}otherLanguage', 13)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}periodical', 3)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}editionStatement', 1)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}collation', 4)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}subjectMeeting', 6)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}corporateAuthor', 7)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}author', 26)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}subjectPerson', 2)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}subjectCorporateBody', 8)
#('{http://www.iiep.unesco.org/plan4learning/model.owl#}theme', 6)
#('{http://purl.org/dc/terms/}abstract', 3)
#('{http://purl.org/dc/terms/}type', 1)
