# -*- coding: utf-8 -*-
'''
Created on Aug 27, 2013

@author: tc
'''

from django.conf import settings
from django.contrib.auth.decorators import login_required
from django.shortcuts import get_object_or_404
from django.utils.decorators import method_decorator
from django.views.generic import ListView, DetailView
from .models import Record
from .forms import RecordFilterForm
from .utils import get_labels_for_uris
import logging


logger = logging.getLogger(__name__)

class RecordListView(ListView):
    
    queryset = Record.objects.select_related("language").prefetch_related('titles').distinct()  # @UndefinedVariable
    paginate_by = settings.NB_RECORDS_BY_PAGE
    template_name = "p4l/p4l_home.html"
    form_class = RecordFilterForm
    
    @method_decorator(login_required)
    def dispatch(self, *args, **kwargs):
        return super(RecordListView, self).dispatch(*args, **kwargs)
    
    def get_context_data(self, **kwargs):
        context = ListView.get_context_data(self, **kwargs)
        context['filter_form'] = self.form_class()
        # Add filter params from GET params
        filter_params = {}
        if 'title' in self.request.GET:
            filter_params['title'] = self.request.GET['title']
        context['filter_params'] = filter_params
        return context
    
    def get_queryset(self):
        qs = super(RecordListView, self).get_queryset()
        filter_form = self.form_class(self.request.GET)
        if filter_form.is_valid():
            return filter_form.get_filter_qs(qs)
        else:
            return qs


class RecordDetailView(DetailView):
    
    model = Record
    template_name = "p4l/p4l_record_view.html"
    slug_field = "uri" # Even if it is useless because we override get_objet
    
    @method_decorator(login_required)
    def dispatch(self, *args, **kwargs):
        return super(RecordDetailView, self).dispatch(*args, **kwargs)
    
    def get_object(self, queryset=None):
        if "uri" not in self.request.GET:
            raise AttributeError(u"Record view must be called uri GET parameter")
        return get_object_or_404(Record.objects.select_related("language"), uri=self.request.GET["uri"])
    
    def get_context_data(self, **kwargs):
        context = DetailView.get_context_data(self, **kwargs)
        # We get the language, "fr" by default
        lang = "fr"
        if "lang" in self.request.GET:
            lang = self.request.GET["lang"]
        elif hasattr(self.request, "LANGUAGE_CODE") and self.request.LANGUAGE_CODE in ["fr","en","es"]:
            lang = self.request.LANGUAGE_CODE
        # self.object is the record entry
        # We get the subjects'labels with the Thesaurus repository
        uri_list = [s.uri for s in self.object.subjects.all()]
        uris_labels = get_labels_for_uris(uri_list, "http://skos.um.es/unescothes/CS000", lang, False)
        context['subjects_labels'] = uris_labels
        # We get the themes'labels with the Themes repository
        uri_list = [s.uri for s in self.object.themes.all()]
        uris_labels = get_labels_for_uris(uri_list, "http://www.iiep.unesco.org/plan4learning/scheme/Themes", lang, False)
        context['themes_labels'] = uris_labels
        # We get the countries'labels with the Thesaurus repository
        uri_list = [s.uri for s in self.object.countries.all()]
        uris_labels = get_labels_for_uris(uri_list, "http://skos.um.es/unescothes/CS000/Countries", lang, False)
        context['countries_labels'] = uris_labels
        # We get the languages'labels with the Languages repository
        if self.object.language:
            uri_list = [self.object.language.uri]
            uris_labels = get_labels_for_uris(uri_list, "http://www.iiep.unesco.org/plan4learning/scheme/Languages", lang, False)
            context['language_label'] = uris_labels[self.object.language.uri]
        # We get the other languages'labels with the Languages repository
        uri_list = [s.uri for s in self.object.otherLanguages.all()]
        uris_labels = get_labels_for_uris(uri_list, "http://www.iiep.unesco.org/plan4learning/scheme/Languages", lang, False)
        context['otherLanguages_labels'] = uris_labels
        # We get the project'labels with the Projects repository
        uri_list = [s.uri for s in self.object.projectNames.all()]
        uris_labels = get_labels_for_uris(uri_list, "http://www.iiep.unesco.org/plan4learning/scheme/Projects", None, True)
        context['projects_labels'] = uris_labels
        # We get the subjectCorporateBodies'labels with the Organizations repository
        uri_list = [s.uri for s in self.object.subjectCorporateBodies.all()]
        uris_labels = get_labels_for_uris(uri_list, "http://www.iiep.unesco.org/plan4learning/scheme/Organizations", None, True)
        context['subjectCorporateBodies_labels'] = uris_labels
        # We get the corporateAuthors'labels with the Organizations repository
        uri_list = [s.uri for s in self.object.corporateAuthors.all()]
        uris_labels = get_labels_for_uris(uri_list, "http://www.iiep.unesco.org/plan4learning/scheme/Organizations", None, True)
        context['corporateAuthors_labels'] = uris_labels
        # We get the recordType'labels with the DocumentType repository
        if self.object.recordType:
            uri_list = [self.object.recordType]
            uris_labels = get_labels_for_uris(uri_list, "http://www.iiep.unesco.org/plan4learning/scheme/DocumentType", lang, False)
            context['recordType_label'] = uris_labels[self.object.recordType]
        
        return context

