package csl.tools.algos;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.Properties;
import java.util.Vector;

public class Graph {
	List<Vertex> vertices = new Vector<Vertex>();

	List<Edge> edges = new Vector<Edge>();

	Hashtable<Vertex, List<Edge>> vertex2edges = new Hashtable<Vertex, List<Edge>>();

	Hashtable<Vertex, Hashtable<Vertex, Edge>> vertex2vertex2edge = new Hashtable<Vertex, Hashtable<Vertex, Edge>>();

	private String name;

	public static Graph smallGraph() {

		Graph g = new Graph("Small Graph");
		Vertex solarPanel, windTurb, laptop, cellPhone, battery;
		g.addVertex(solarPanel = new Vertex("Solar Panel", 0, false));
		g.addVertex(windTurb = new Vertex("Wind Turbine", 1, false));
		g.addVertex(laptop = new Vertex("Laptop", 2, true));
		g.addVertex(cellPhone = new Vertex("Cell Phone", 3, false));
		g.addVertex(battery = new Vertex("Battery", 4, true));

		g.setEdge(solarPanel, cellPhone, new Edge(10));
		g.setEdge(windTurb, battery, new Edge(8));
		g.setEdge(battery, laptop, new Edge(5));
		g.setEdge(laptop, cellPhone, new Edge(2));
		return g;
	}

	public static Graph mediumGraph() {

		Graph g = new Graph("Medium Graph");
		Vertex solarPanel1, solarPanel2, windTurb, laptop, cellPhone, battery;
		g.addVertex(solarPanel1 = new Vertex("Sol1", 0, false));
		g.addVertex(solarPanel2 = new Vertex("Sol2", 0, false));
		g.addVertex(windTurb = new Vertex("Wind", 1, false));
		g.addVertex(laptop = new Vertex("Laptop", 2, true));
		g.addVertex(cellPhone = new Vertex("Cell", 3, false));
		g.addVertex(battery = new Vertex("Batt", 4, true));

		g.setEdge(solarPanel1, cellPhone, new Edge(10));
		g.setEdge(solarPanel2, cellPhone, new Edge(5));
		g.setEdge(solarPanel2, laptop, new Edge(10));
		g.setEdge(windTurb, battery, new Edge(8));
		g.setEdge(battery, laptop, new Edge(5));
		g.setEdge(laptop, cellPhone, new Edge(2));
		return g;
	}

	public static Graph longGraph() {

		Graph g = new Graph("Medium Graph");
		Vertex solarPanel1, solarPanel2, windTurb, laptop, cellPhone, pda, mp3, battery;
		g.addVertex(solarPanel1 = new Vertex("Sol1", "solar.jpg", 0, false));
		g.addVertex(solarPanel2 = new Vertex("Sol2", "solar.jpg", 0, false));
		g.addVertex(windTurb = new Vertex("Wind", "wind.jpg", 1, false));
		g.addVertex(laptop = new Vertex("Laptop", "laptop.jpg", 2, true));
		g.addVertex(cellPhone = new Vertex("Cell", "cellphone.jpg", 3, false));
		g.addVertex(pda = new Vertex("PDA", "pda.jpg", 2, false));
		g.addVertex(mp3 = new Vertex("MP3", "mp3.jpg", 3, false));
		g.addVertex(battery = new Vertex("Batt", "battery.bmp", 4, true));

		g.setEdge(solarPanel1, cellPhone, new Edge(7));
		g.setEdge(solarPanel1, pda, new Edge(3));
		g.setEdge(solarPanel2, cellPhone, new Edge(5));
		g.setEdge(solarPanel2, laptop, new Edge(500));
		g.setEdge(windTurb, battery, new Edge(8));
		g.setEdge(battery, laptop, new Edge(5));
		g.setEdge(battery, mp3, new Edge(7));
		g.setEdge(laptop, cellPhone, new Edge(2));
		g.setEdge(laptop, pda, new Edge(2));
		g.setEdge(laptop, mp3, new Edge(2));
		return g;
	}

	public static Graph longGraph(Properties p) {

		Graph g = new Graph("Medium Graph");
		Vertex solarPanel1, solarPanel2, windTurb, laptop, cellPhone, pda, mp3, battery;
		g.addVertex(solarPanel1 = new Vertex("Sol1", "solar.jpg", 0, false));
		g.addVertex(solarPanel2 = new Vertex("Sol2", "solar.jpg", 0, false));
		g.addVertex(windTurb = new Vertex("Wind", "wind.jpg", 1, false));
		g.addVertex(laptop = new Vertex("Laptop", "laptop.jpg", 2, true));
		g.addVertex(cellPhone = new Vertex("Cell", "cellphone.jpg", 3, false));
		g.addVertex(pda = new Vertex("PDA", "pda.jpg", 2, false));
		g.addVertex(mp3 = new Vertex("MP3", "mp3.jpg", 3, false));
		g.addVertex(battery = new Vertex("Batt", "battery.bmp", 4, true));

		g.setEdge(solarPanel1, cellPhone, new Edge(edgeValueFor(solarPanel1,
				cellPhone, p)));
		g
				.setEdge(solarPanel1, pda, new Edge(edgeValueFor(solarPanel1,
						pda, p)));
		g.setEdge(solarPanel2, cellPhone, new Edge(edgeValueFor(solarPanel2,
				cellPhone, p)));
		g.setEdge(solarPanel2, laptop, new Edge(edgeValueFor(solarPanel2,
				laptop, p)));
		g.setEdge(windTurb, battery, new Edge(
				edgeValueFor(windTurb, battery, p)));
		g.setEdge(battery, laptop, new Edge(edgeValueFor(battery, laptop, p)));
		g.setEdge(battery, mp3, new Edge(edgeValueFor(battery, mp3, p)));
		g.setEdge(laptop, cellPhone, new Edge(
				edgeValueFor(laptop, cellPhone, p)));
		g.setEdge(laptop, pda, new Edge(edgeValueFor(laptop, pda, p)));
		g.setEdge(laptop, mp3, new Edge(edgeValueFor(laptop, mp3, p)));
		return g;
	}

	public static Graph demoGraph1(Properties p) {

		Graph g = new Graph("Demo Graph 1");
		Vertex solarPanel, fuelCell, windTurb, laptop, pda, mp3;
		g.addVertex(solarPanel = new Vertex("Sol1", "solar.jpg", 0, false));
		g.addVertex(fuelCell = new Vertex("FuelCell", "battery.bmp", 0, false));
		g.addVertex(windTurb = new Vertex("Wind", "wind.jpg", 1, false));
		g.addVertex(laptop = new Vertex("Laptop", "laptop.jpg", 2, true));
		g.addVertex(pda = new Vertex("PDA", "pda.jpg", 2, false));
		g.addVertex(mp3 = new Vertex("MP3", "mp3.jpg", 3, false));

		g.setEdge(solarPanel, pda, new Edge(edgeValueFor(solarPanel, pda, p)));
		g
				.setEdge(fuelCell, laptop, new Edge(edgeValueFor(fuelCell,
						laptop, p)));
		g.setEdge(windTurb, mp3, new Edge(edgeValueFor(windTurb, mp3, p)));
		return g;
	}

	public static Graph demoGraph2(Properties p) {

		Graph g = new Graph("Demo Graph 2");
		Vertex solarPanel, fuelCell, windTurb, laptop, pda, mp3, flyWheel;
		g.addVertex(solarPanel = new Vertex("Sol1", "solar.jpg", 0, false));
		g.addVertex(fuelCell = new Vertex("FuelCell", "battery.bmp", 0, false));
		g.addVertex(windTurb = new Vertex("Wind", "wind.jpg", 1, false));
		g.addVertex(laptop = new Vertex("Laptop", "laptop.jpg", 2, true));
		g.addVertex(pda = new Vertex("PDA", "pda.jpg", 2, false));
		g.addVertex(mp3 = new Vertex("MP3", "mp3.jpg", 3, false));
		g.addVertex(flyWheel = new Vertex("FlyWheel", "flywheel.jpg", 4, true));

		g.setEdge(solarPanel, pda, new Edge(edgeValueFor(solarPanel, pda, p)));
		g
				.setEdge(fuelCell, laptop, new Edge(edgeValueFor(fuelCell,
						laptop, p)));
		g.setEdge(windTurb, flyWheel, new Edge(edgeValueFor(windTurb, flyWheel,
				p)));
		g.setEdge(flyWheel, mp3, new Edge(edgeValueFor(flyWheel, mp3, p)));
		g.setEdge(laptop, pda, new Edge(edgeValueFor(laptop, pda, p)));
		g.setEdge(laptop, mp3, new Edge(edgeValueFor(laptop, mp3, p)));
		return g;
	}

	public static Graph demoGraph3(Properties p) {

		Graph g = new Graph("Demo Graph 3");
		Vertex solarPanel, fuelCell, windTurb, laptop, cellPhone, pda, mp3, flyWheel;
		g.addVertex(solarPanel = new Vertex("Sol", "solar.jpg", 0, false));
		g.addVertex(fuelCell = new Vertex("FuelCell", "battery.bmp", 0, false));
		g.addVertex(windTurb = new Vertex("Wind", "wind.jpg", 1, false));
		g.addVertex(laptop = new Vertex("Laptop", "laptop.jpg", 2, true));
		g.addVertex(cellPhone = new Vertex("Cell", "cellphone.jpg", 3, false));
		g.addVertex(pda = new Vertex("PDA", "pda.jpg", 2, false));
		g.addVertex(mp3 = new Vertex("MP3", "mp3.jpg", 3, false));
		g.addVertex(flyWheel = new Vertex("FlyWheel", "flywheel.jpg", 4, true));

		g.setEdge(solarPanel, cellPhone, new Edge(edgeValueFor(solarPanel,
				cellPhone, p)));
		g.setEdge(solarPanel, pda, new Edge(edgeValueFor(solarPanel, pda, p)));
		g.setEdge(fuelCell, cellPhone, new Edge(edgeValueFor(fuelCell,
				cellPhone, p)));
		g
				.setEdge(fuelCell, laptop, new Edge(edgeValueFor(fuelCell,
						laptop, p)));
		g.setEdge(windTurb, flyWheel, new Edge(edgeValueFor(windTurb, flyWheel,
				p)));
		g
				.setEdge(flyWheel, laptop, new Edge(edgeValueFor(flyWheel,
						laptop, p)));
		g.setEdge(flyWheel, mp3, new Edge(edgeValueFor(flyWheel, mp3, p)));
		g.setEdge(laptop, cellPhone, new Edge(
				edgeValueFor(laptop, cellPhone, p)));
		g.setEdge(laptop, mp3, new Edge(edgeValueFor(laptop, mp3, p)));
		return g;
	}

	private static double edgeValueFor(Vertex from, Vertex to, Properties p) {

		if (p == null)
			return 0;
		Enumeration names = p.propertyNames();
		while (names.hasMoreElements()) {
			String s = p.getProperty((String) names.nextElement());
			String[] ss = s.split("-");
			if (ss[0].equalsIgnoreCase(from.name)
					&& ss[1].equalsIgnoreCase(to.name))
				return Double.parseDouble(ss[2]);
		}
		return 0;
	}

	public static Graph largeGraph() {

		Graph g = new Graph("Large Graph");
		Vertex s1, s2, s3;
		Vertex d1, d2, d3, d4, d5, d6, d7, d8, d9;

		g.addVertex(s1 = new Vertex("S1", 0));
		g.addVertex(s2 = new Vertex("S2", 1));
		g.addVertex(s3 = new Vertex("S3", 1));
		g.addVertex(d1 = new Vertex("D1", 2));
		g.addVertex(d2 = new Vertex("D2", 3));
		g.addVertex(d3 = new Vertex("D3", 4));
		g.addVertex(d4 = new Vertex("D4", 2));
		g.addVertex(d5 = new Vertex("D5", 3));
		g.addVertex(d6 = new Vertex("D6", 4));
		g.addVertex(d7 = new Vertex("D7", 2));
		g.addVertex(d8 = new Vertex("D8", 3));
		g.addVertex(d9 = new Vertex("D9", 4));

		/*
		 * S1
		 */
		g.setEdge(s1, d1, new Edge(1));
		g.setEdge(s1, d2, new Edge(1));
		g.setEdge(s1, d3, new Edge(1));

		g.setEdge(s1, d4, new Edge(20));
		g.setEdge(s1, d5, new Edge(20));

		g.setEdge(d1, d3, new Edge(1));

		/*
		 * S2
		 */
		g.setEdge(s2, d1, new Edge(30));
		g.setEdge(s2, d2, new Edge(30));
		g.setEdge(s2, d3, new Edge(30));

		g.setEdge(s2, d4, new Edge(1));
		g.setEdge(s2, d5, new Edge(1));

		g.setEdge(s2, d6, new Edge(10));
		g.setEdge(s2, d7, new Edge(10));
		g.setEdge(s2, d8, new Edge(10));
		g.setEdge(s2, d9, new Edge(10));

		/*
		 * S3
		 */
		g.setEdge(s3, d4, new Edge(15));
		g.setEdge(s3, d5, new Edge(15));

		g.setEdge(s3, d6, new Edge(1));
		g.setEdge(s3, d7, new Edge(1));
		g.setEdge(s3, d8, new Edge(1));
		g.setEdge(s3, d9, new Edge(1));

		g.setEdge(s1, s2, new Edge(5));
		g.setEdge(s2, s1, new Edge(5));
		g.setEdge(s1, s3, new Edge(45));
		g.setEdge(s3, s1, new Edge(45));
		g.setEdge(s2, s3, new Edge(5));
		g.setEdge(s3, s2, new Edge(5));

		g.setEdge(d1, d2, new Edge(1));
		g.setEdge(d2, d3, new Edge(1));
		g.setEdge(d1, d3, new Edge(1));

		return g;
	}

	public Graph(String string) {

		name = string;
	}

	public double bestPossibleCost(List<Vertex> path, FloydWarshall fw) {

		double c = 0;
		for (int i = 0; i < path.size() - 1; i++) {
			c += edgeCost(path.get(i), path.get(i + 1));
			if (path.get(i).isBattery && !path.get(i).isLoaded()) {
				System.out.println(path.get(i) + " MISSING");
				c += bestPossibleCost(fw.shortestPathTo(path.get(i)), fw);
			}
		}
		return c;
	}

	public double cost(List<Vertex> path) {

		double c = 0;
		for (int i = 0; i < path.size() - 1; i++)
			c += edgeCost(path.get(i), path.get(i + 1));
		return c;
	}

	public double initialCost(List<Vertex> list) {

		double c = 0;
		for (int i = 0; i < list.size() - 1; i++)
			c += edgeInitialCost(list.get(i), list.get(i + 1));
		return c;
	}

	public void setVertices(List<Vertex> vlist) {

		for (Vertex v : vlist)
			addVertex(v);
	}

	private void addVertex(Vertex v) {
		if (vertices.contains(v))
			return;
		vertex2edges.put(v, new Vector<Edge>());
		vertex2vertex2edge.put(v, new Hashtable<Vertex, Edge>());
		vertices.add(v);
	}

	public void setEdge(Vertex start, Vertex end, Edge edge) {

		vertex2edges.get(start).add(edge);
		vertex2vertex2edge.get(start).put(end, edge);
		edges.add(edge);
		edge.setStart(start);
		edge.setEnd(end);
	}

	public double edgeCost(Vertex start, Vertex end) {

		if (start == end)
			return 0;
		Edge edge = edge(start, end);
		if (edge == null)
			return Double.MAX_VALUE;
		return edge.cost();
	}

	public double edgeInitialCost(Vertex start, Vertex end) {

		if (start == end)
			return 0;
		Edge edge = edge(start, end);
		if (edge == null)
			return Double.MAX_VALUE;
		return edge.initialCost;
	}

	public Edge edge(Vertex start, Vertex end) {

		return vertex2vertex2edge.get(start).get(end);
	}

	public double edgeCost(int i, int j) {

		return edgeCost(vertex(i), vertex(j));
	}

	public double edgeInitialCost(int i, int j) {

		return edgeInitialCost(vertex(i), vertex(j));
	}

	public Vertex vertex(int i) {

		return vertices.get(i);
	}

	public int vertexCount() {

		return vertices.size();
	}

	public int edgeCount() {

		return edges.size();
	}

	@Override
	public String toString() {

		String str = "Graph: " + name + "\n\nVertices:\n";
		for (Vertex v : vertices)
			str += "\t" + v + "\n";
		str += "\nEdges:\n";
		for (Edge e : edges)
			str += "\t" + e + "\n";
		return str;
	}

	public Vertex vertexNamed(String string) {
		for (Vertex v : vertices) {
			if (v.name.equalsIgnoreCase(string))
				return v;
		}
		return null;
	}

	private void preLoad(Vertex batt, FloydWarshall fw) {

		List<Vertex> loadingPath = fw.shortestPathTo(batt);
		for (Vertex v : loadingPath) {
			if (v != batt && v.isBattery && !v.isLoaded())
				preLoad(v, fw);
		}
		double cost = initialCost(loadingPath);
		for (Edge e : batt.edges) {
			e.swapCosts();
			// e.increaseCost(cost);
		}
		batt.setIsLoaded(true);
		fw.init();
		fw.run();
	}

	public boolean loadBattery(Vertex battery, FloydWarshall fw,
			Properties transferProperties, boolean echo) {

		if (battery.isLoaded())
			return false;

		List<Vertex> loadingPath = fw.shortestPathTo(battery);
		Vertex v0 = loadingPath.get(0);
		if (v0.isBattery && !v0.isLoaded()) {
			List<Vertex> prePath = fw.shortestPathTo(v0);
			double cost = initialCost(prePath);
			v0.setIsLoaded(true);
			for (Edge e : v0.edges) {
				if (e.cost() > 9000)
					e.swapCosts();
				e.increaseCost(cost);
			}
		}
		double loadingCost = cost(loadingPath);
		for (Edge e : battery.edges) {
			if (e.cost() > 9000)
				e.swapCosts();
			e.increaseCost(loadingCost);
		}
		if (echo) {
			if (loadingPath.size() != 2)
				System.out.println("Should not reach here");
			else {
				System.out.println(loadingPath.get(0).name + "-"
						+ loadingPath.get(1).name + "-" + 0 + "-"
						+ cost(loadingPath));
				transferProperties.setProperty("transfer",
						loadingPath.get(0).name + "-" + loadingPath.get(1).name
								+ "-" + 0 + "-" + cost(loadingPath));
			}
		}
		battery.setIsLoaded(true);
		fw.init();
		fw.run();
		return true;
	}

	public void unload(Vertex battery) {

		if (!battery.isLoaded())
			return;
		battery.setIsLoaded(false);
	}

	/**
	 * Saves files for Eugenio Tisselli's toolbox
	 * 
	 * @throws IOException
	 * 
	 */
	public void saveETFiles(File folder) throws IOException {

		folder = new File(folder, name);
		folder.mkdir();
		for (Vertex v : vertices) {
			File f = new File(folder, v.name + ".txt");
			Writer fw = new FileWriter(f);
			Properties prop = new Properties();
			prop.setProperty("deviceDeviceID", v.name);
			prop.setProperty("deviceManufacturer", "Sony Corp.");
			prop.setProperty("deviceProduct", v.name);
			prop.setProperty("deviceName", v.name);
			prop.setProperty("deviceDeviceLocation", "Sony CSL Paris");
			prop.store(fw, null);
		}

		File deviceFile = new File(folder, "devices.txt");
		Writer fw = new FileWriter(deviceFile);
		int i = 0;
		Properties prop = new Properties();
		for (Vertex vertex : vertices) {
			prop.setProperty("dev" + i++, vertex.name + ".txt" + "-"
					+ vertex.imgFileName);
		}
		prop.store(fw, null);

		prop = new Properties();
		File edgeFile = new File(folder, "edges.txt");
		fw = new FileWriter(edgeFile);
		i = 0;
		for (Edge edge : edges) {
			prop.setProperty("edge" + i++, edge.getStart().name + "-"
					+ edge.getEnd().name + "-" + edge.initialCost);
		}
		prop.store(fw, null);
	}

	public Graph copy() {
		Graph g = new Graph(name);
		List<Vertex> vlist = new Vector<Vertex>();
		for (Vertex v : vertices)
			vlist.add(v.copy());
		g.setVertices(vlist);
		for (Edge e : edges)
			g.setEdge(g.vertexNamed(e.getStart().name), g.vertexNamed(e
					.getEnd().name), new Edge(e.initialCost));
		return g;
	}

	public static Graph demoGraph(String graphIndex) {
		return demoGraph(Integer.parseInt(graphIndex), null);
	}

	public static Graph demoGraph(int graphIndex, Properties p) {
		if (graphIndex == 1)
			return demoGraph1(p);
		if (graphIndex == 2)
			return demoGraph2(p);
		if (graphIndex == 3)
			return demoGraph3(p);
		return null;
	}
}
