package csl.tools.gui;

import java.io.File;
import java.util.List;
import java.util.Stack;
import java.util.Vector;

import javax.swing.ComboBoxModel;
import javax.swing.JFrame;
import javax.swing.WindowConstants;
import javax.swing.event.ListDataListener;

import csl.tools.prefs.PreferenceManager;

/**
 * Manages lists of instances of class <code>I</code>
 * 
 * @author roy
 * 
 * @param <I>
 */
public abstract class RecentItemListManager<I> implements ComboBoxModel {

	private static final String	ITEM_PREF_LABEL			= "item";
	private static final String	ITEM_COUNT_PREF_LABEL	= "item.count";
	private static final String	MAX_SIZE_PREF_LABEL		= "max.size";
	private static final int	DEFAULT_MAX_SIZE		= 10;

	PreferenceManager			pref;
	Stack<I>					elements;
	private Object				selectedItem;
	private int					maxSize;
	private int					actualSize;

	abstract public I fromString(String s);

	abstract public String toString(I item);

	abstract public String displayString(I item);

	public RecentItemListManager(PreferenceManager pm, int size) {

		pref = pm;
		maxSize = size;
		actualSize = 0;
		elements = new Stack<I>();
		loadPref();
	}

	private void loadPref() {

		maxSize = maxSize(pref);
		for (I item : getElements(pref))
			basicPush(item);
	}

	private int maxSize(PreferenceManager p) {

		String s = p.get(MAX_SIZE_PREF_LABEL);
		if (s == null)
			return DEFAULT_MAX_SIZE;
		return Integer.parseInt(s);
	}

	private int itemCount(PreferenceManager p) {

		String s = p.get(ITEM_COUNT_PREF_LABEL);
		if (s == null)
			return 0;
		return Integer.parseInt(s);
	}

	private List<I> getElements(PreferenceManager p) {

		List<I> list = new Vector<I>();
		for (int i = 0; i < itemCount(p); i++) {
			String str = p.get(ITEM_PREF_LABEL + i);
			if (str != null)
				list.add(fromString(str));
		}
		return list;
	}

	private I basicPush(I item) {

		if (elements.contains(item))
			return moveToLast(item);
		if (actualSize > maxSize)
			removeOldest();
		actualSize++;
		return elements.push(item);
	}

	public I push(I item) {

		I elt = basicPush(item);
		savePreferences();
		return elt;
	}

	private void removeOldest() {

		actualSize--;
		elements.remove(elements.lastElement());
		savePreferences();
	}

	private I moveToLast(I item) {

		elements.remove(item);
		elements.push(item);
		savePreferences();
		return item;
	}

	@SuppressWarnings("unchecked")
	public void setSelectedItem(Object anItem) {

		moveToLast((I) anItem);
		savePreferences();
		selectedItem = anItem;
	}

	@SuppressWarnings("unchecked")
	private void savePreferences() {

		pref.set(MAX_SIZE_PREF_LABEL, "" + maxSize);
		pref.set(ITEM_COUNT_PREF_LABEL, "" + actualSize);
		for (int i = 0; i < actualSize; i++)
			pref.set(ITEM_PREF_LABEL + i, toString((I) getElementAt(i)));
	}

	public Object getSelectedItem() {

		return selectedItem;
	}

	public int getSize() {

		return actualSize;
	}

	public Object getElementAt(int index) {

		if (elements.isEmpty())
			return null;
		return elements.get(getSize() - 1 - index);
	}

	public void addListDataListener(ListDataListener l) {

		// TODO Auto-generated method stub
	}

	public void removeListDataListener(ListDataListener l) {

		// TODO Auto-generated method stub
	}

	public static void main(String[] args) {

		JFrame f = new JFrame();
		f.setSize(500, 100);
		f.setDefaultCloseOperation(WindowConstants.EXIT_ON_CLOSE);
		// RecentItemSingleSelectionPanel<String> p = new
		// RecentItemSingleSelectionPanel<String>();
		RecentItemMultiSelectionPanel<String> p = new RecentItemMultiSelectionPanel<String>();
		PreferenceManager pref = new PreferenceManager(new File(System
				.getProperty("user.home"), "Tmp\\avirer.txt"), null);
		RecentItemListManager<String> man = new RecentItemListManager<String>(
				pref, 5) {

			@Override
			public String fromString(String s) {

				return s;
			}

			@Override
			public String toString(String item) {

				return item;
			}

			@Override
			public String displayString(String item) {

				return "[" + item + "]";
			}
		};
		man.push("Toto");
		man.push("Titi");
		man.push("Tata");
		man.push("Tyty");
		man.push("Tete");
		man.push("Trrr");
		man.push("Tchi");
		p.setModel(man);
		f.setContentPane(p);
		f.setVisible(true);
	}
}
