package csl.tools.prefs;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.Serializable;
import java.util.Properties;

public class PreferenceManager implements Serializable {

	private static final long	serialVersionUID	= 8357485317510442116L;

	public final File			outputFile;
	public final File			defaultFile;
	public final Properties		properties;

	public PreferenceManager(File f, File defaultFile) {

		this.outputFile = f;
		this.defaultFile = defaultFile;
		this.properties = new Properties();
		init();
	}

	private void init() {

		try {
			properties.load(new FileInputStream(outputFile));
		} catch (Exception e) {
			System.err.println("Could not load the preference file "
					+ outputFile.getPath());
			try {
				properties.load(new FileInputStream(defaultFile));
			} catch (Exception e1) {
				System.err
						.println("Could not load the DEFAULT preference file "
								+ defaultFile
								+ "\nCaution: Preferences not initialized");
			}
		}
	}

	public void save() throws FileNotFoundException, IOException {

		saveAs(outputFile);
	}

	public void saveAs(File scriptFile) throws FileNotFoundException,
			IOException {

		scriptFile.getParentFile().mkdirs();
		properties.store(new FileOutputStream(scriptFile), "property file");
	}

	private void tryToSave() {

		try {
			save();
		} catch (IOException e) {
			System.err.println("Could not save preferences");
			e.printStackTrace();
		}
	}

	public void set(String name, String val) {

		properties.setProperty(name, val);
		tryToSave();
	}

	public void setBoolean(String name, boolean b) {

		properties.setProperty(name, Boolean.toString(b));
		tryToSave();
	}

	public void setInt(String name, int i) {

		properties.setProperty(name, Integer.toString(i));
		tryToSave();
	}

	public void setDouble(String name, double d) {

		properties.setProperty(name, Double.toString(d));
		tryToSave();
	}

	public void setFile(String name, File file) {

		properties.setProperty(name, file.getPath());
		tryToSave();
	}

	public void setFile(String name, String fname) {

		properties.setProperty(name, fname);
		tryToSave();
	}

	public String get(String name) {

		return properties.getProperty(name);
	}

	public boolean booleanValue(String prop) {

		String s = get(prop);
		boolean b = false;
		if (s != null)
			if (s.equalsIgnoreCase("true"))
				b = true;
		return b;
	}

	public File fileValue(String prop) {

		String s = get(prop);
		File f = null;
		if (s != null)
			f = new File(s);
		return f;
	}

	public int intValue(String prop, int defaultValue) {

		try {
			return Integer.parseInt(get(prop));
		} catch (NumberFormatException e) {
			return defaultValue;
		} catch (NullPointerException e) {
			return defaultValue;
		}
	}

	public double doubleValue(String prop, double defaultValue) {

		try {
			return Double.parseDouble(get(prop));
		} catch (NumberFormatException e) {
			return defaultValue;
		} catch (NullPointerException e) {
			return defaultValue;
		}
	}

	public Iterable<String> stringPropertyNames() {

		return properties.stringPropertyNames();
	}

	@Override
	public String toString() {

		return properties.toString();
	}
}
