package csl.tools.weka;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.util.Random;

import weka.classifiers.Classifier;
import weka.classifiers.Evaluation;
import weka.classifiers.functions.SMO;
import weka.classifiers.functions.supportVector.RBFKernel;
import weka.classifiers.trees.J48;
import weka.core.Attribute;
import weka.core.FastVector;
import weka.core.Instance;
import weka.core.Instances;
import weka.filters.Filter;
import weka.filters.unsupervised.attribute.NumericToBinary;
import weka.filters.unsupervised.attribute.StringToNominal;
import weka.filters.unsupervised.attribute.StringToWordVector;
import csl.tools.files.FileTools;

public class TextMining {

	/**
	 * Construit une Instances dont chaque Instance represente un des fichiers.
	 * Il y a trois attributs: 1) le filename; 2) le contenu du fichier; 3) la
	 * classe, i.e. le nom du repertoire contant le fichier
	 * 
	 * @param rootDir
	 * @return
	 * @throws Exception
	 */
	public Instances instances(File rootDir) throws Exception {

		/*
		 * Liste des sous-repertoires
		 */
		File[] subDirs = rootDir.listFiles(new FileFilter() {

			public boolean accept(File pathname) {

				return pathname.isDirectory();
			}
		});

		/*
		 * Vector contenant les attributs
		 */
		FastVector atts = new FastVector(3);
		// Le nom du fichier
		atts.addElement(new Attribute("filename", (FastVector) null));
		// Le contenu du fichier texte ("raw")
		atts.addElement(new Attribute("contents", (FastVector) null));

		/*
		 * Les "classes" des fichiers sont les noms des sous-repertoires
		 */
		FastVector classes = new FastVector(subDirs.length);
		for (File subDir : subDirs) {
			classes.addElement(subDir.getName());
		}
		// La "classe" du fichier
		atts.addElement(new Attribute("class", classes));
		Instances data = new Instances("Files: " + rootDir.getName(), atts, 0);
		data.setClassIndex(2);

		/*
		 * Construction du fichier ARFF
		 */
		for (File subDir : subDirs) {
			File[] txtFiles = subDir.listFiles(FileTools.txtFF());
			int classInd = classes.indexOf(subDir.getName());
			for (File txtFile : txtFiles) {

				double[] newInst = new double[3];
				newInst[0] = (double) data.attribute(0).addStringValue(
						txtFile.getPath());
				InputStreamReader is;
				is = new InputStreamReader(new FileInputStream(txtFile));
				StringBuffer txtStr = new StringBuffer();
				int c;
				while ((c = is.read()) != -1) {
					txtStr.append((char) c);
				}
				newInst[1] = (double) data.attribute(1).addStringValue(
						txtStr.toString());

				newInst[2] = (double) classInd;

				data.add(new Instance(1.0, newInst));
			}
		}
		return data;
	}

	/**
	 * Construit une Instances dont chaque Instance represente un fichier. Mais
	 * au lieu du contenu, cette instance contient l'hitogramme des mots
	 * presents dans le fichier.
	 * 
	 * @param instances
	 * @param dictionarySize
	 * @return
	 * @throws Exception
	 */
	public Instances wordFilter(Instances instances, int dictionarySize)
			throws Exception {

		/*
		 * Le "filter" sert a transformer le contenu des fichiers de "instances"
		 * en histogramme de frequence des mots
		 */
		StringToWordVector filter = new StringToWordVector();
		/*
		 * Option indiquant que le contenu est la colonne 2
		 */
		String[] options = new String[2];
		options[0] = "-R";
		options[1] = "2";
		filter.setOptions(options);
		/*
		 * Nombres de mots a considerer
		 */
		filter.setWordsToKeep(dictionarySize);
		/*
		 * On ne considere que les "mots"
		 */
		filter.setOnlyAlphabeticTokens(true);
		/*
		 * Filtrage effectif, qui retourne une nouvelle Instances, "inst"
		 */
		filter.setInputFormat(instances);
		Instances inst = Filter.useFilter(instances, filter);
		inst.deleteAttributeAt(0);
		return inst;
	}

	public Evaluation classify(Instances instances) throws Exception {

		/*
		 * Le classifieur
		 */
		Classifier j48 = new J48();
		((J48) j48).setSubtreeRaising(true);
		((J48) j48).setNumFolds(2);
		((J48) j48).setUnpruned(true);
		j48.buildClassifier(instances);

		Classifier smo = new SMO();
		((SMO) smo).setKernel(new RBFKernel());
		((SMO) smo).setNumFolds(10);
		smo.buildClassifier(instances);

		/*
		 * L'evaluation (stocke la f-measure, la precision, le recall etc.
		 */
		Evaluation eval = new Evaluation(instances);

		/*
		 * 10-fold cross-validation
		 */
		eval.crossValidateModel(smo, instances, 10, new Random());
		System.out.println(smo);
		return eval;
	}

	public static void main(String[] args) {

		/*
		 * Nombre de mots a retenir pour l'histogramme (par ordre decroissant de
		 * frequence)
		 */
		int dictionarySize = 500;
		TextMining tm = new TextMining();
		try {
			/*
			 * Le repertoire racine
			 */
			File root = new File("C:/tmpWeka/train");
			/*
			 * On construit l'Instances avec les contenus des fichiers
			 */
			Instances instances = tm.instances(root);
			System.out.println(instances);
			/*
			 * On fikltre cette instances pour obtenir une nouvelle instances
			 * avec les histogrammes
			 */
			Instances wordInstances = tm.wordFilter(instances, dictionarySize);
			System.out.println(wordInstances);
			/*
			 * Sauvegarde et affichage
			 */
			PrintWriter w = FileTools.printWriterOn(new File(root,
					"instances.arff"));
			w.write(instances.toString());
			w.close();
			w = FileTools.printWriterOn(new File(root, "word-instances.arff"));
			w.write(wordInstances.toString());
			w.close();
			/*
			 * Classification et affichage du resultat
			 */
			Evaluation ev = tm.classify(wordInstances);
			System.out.println(ev.toClassDetailsString());
			System.out.println(ev.toMatrixString());
			System.out.println(ev.toSummaryString(true));
		} catch (Exception e) {
			System.err.println(e.getMessage());
			e.printStackTrace();
		}
	}
}
