package csl.tools.algos;

import java.text.DecimalFormat;
import java.util.Hashtable;
import java.util.List;
import java.util.Vector;

public class FloydWarshall {

	private static DecimalFormat df = new DecimalFormat();

	private double[][] path;

	private Graph graph;

	private int N;

	private Hashtable<Vertex, Hashtable<Vertex, Vertex>> paths = new Hashtable<Vertex, Hashtable<Vertex, Vertex>>();

	public Graph getGraph() {
		return graph;
	}

	public FloydWarshall(Graph graph) {
		super();
		this.graph = graph;
		this.N = graph.vertexCount();
		this.path = new double[N][N];
		init();
	}

	public void init() {
		for (int i = 0; i < N; i++)
			for (int j = 0; j < N; j++)
				path[i][j] = graph.edgeCost(i, j);
	}

	public void run() {
		for (int k = 0; k < N; k++)
			for (int i = 0; i < N; i++)
				for (int j = 0; j < N; j++) {
					double p_i_j = path[i][j];
					double p_i_k_j = path[i][k] + path[k][j];
					if (p_i_k_j < p_i_j)
						insert(k, i, j);
					path[i][j] = Math.min(p_i_j, p_i_k_j);
				}
	}

	private void insert(int k, int i, int j) {

		Hashtable<Vertex, Vertex> hashtable = paths.get(graph.vertex(i));
		if (hashtable == null)
			paths.put(graph.vertex(i), new Hashtable<Vertex, Vertex>());
		paths.get(graph.vertex(i)).put(graph.vertex(j), graph.vertex(k));
	}

	@Override
	public String toString() {

		String str = "\t";
		for (int i = 0; i < N; i++) {
			str += graph.vertex(i) + "\t";
		}
		str += "\n";
		for (int i = 0; i < N; i++) {
			str += graph.vertex(i) + "\t";
			for (int j = 0; j < N; j++) {
				double d = path[i][j];
				if (d >= Double.MAX_VALUE)
					str += "INF\t";
				else
					str += df.format(d) + "\t";
			}
			str += "\n";
		}
		return str;
	}

	public List<Vertex> path(Vertex from, Vertex to) {

		if (from == to)
			return null;

		List<Vertex> l = new Vector<Vertex>();
		l.add(from);
		l.add(to);
		interm(from, to, l);
		if (graph.cost(l) >= Double.MAX_VALUE)
			return null;
		return l;
	}

	private List<Vertex> interm(Vertex from, Vertex to, List<Vertex> l) {
		int k = 0;
		while (k < l.size() - 1) {
			for (int i = k; i < l.size() - 1; i++) {
				k++;
				Hashtable<Vertex, Vertex> h = paths.get(l.get(i));
				if (h == null)
					continue;
				Vertex interm = h.get(l.get(i + 1));
				if (interm == null)
					continue;
				l.add(i + 1, interm);
				k--;
			}
		}
		return l;
	}

	public static void main(String[] args) {

		Graph graph = Graph.smallGraph();
		// Graph graph = Graph.largeGraph();
		System.out.println(graph.toString());
		FloydWarshall fw = new FloydWarshall(graph);
		System.out.println(fw);
		fw.run();
		System.out.println(fw);
		for (int i = 0; i < 2; i++)
			for (int j = 2; j < graph.vertexCount(); j++) {
				List<Vertex> path = fw.path(fw.graph.vertex(i), fw.graph
						.vertex(j));
				if (path != null)
					System.out.println(path + " COST: " + graph.cost(path));
			}
	}

	public List<Vertex> shortestPathTo(Vertex dest) {

		List<Vertex> shortest = null;
		double bestCost = Double.MAX_VALUE;
		for (Vertex start : graph.vertices) {
			List<Vertex> path = path(start, dest);
			if (path == null)
				continue;
			double cost = graph.initialCost(path);
			if (cost < bestCost) {
				shortest = path;
				bestCost = cost;
			}
		}
		return shortest;
	}

	public void setGraph(Graph g) {
		graph = g;
	}
}
