package csl.tools.algos;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Properties;
import java.util.Vector;

public class ElectricityNetwork implements SimulationInterface {

	public final String name;

	public Graph graph;

	private final List<Vertex> targets = new Vector<Vertex>();

	private int transferCounter = 0;

	private Properties transferProperties;

	public ElectricityNetwork(String string, Graph g) {

		name = string;
		graph = g;
	}

	public List<Vertex> bestPossiblePathTo(Vertex target, FloydWarshall fw) {

		double bestCost = Double.MAX_VALUE;
		List<Vertex> bestPath = null;
		for (Vertex vert : graph.vertices) {
			List<Vertex> path = fw.path(vert, target);
			if (path == null)
				continue;
			double cost = graph.cost(path);
			if (cost == 10000) {
				cost = graph.initialCost(path);
				if (!path.get(0).isBattery)
					System.out.println("Should not reach here");
				if (path.get(0).isLoaded())
					System.out.println("Should not reach here");
				cost += graph.cost(fw.shortestPathTo(vert));
			}
			if (cost < bestCost) {
				bestCost = cost;
				bestPath = path;
			}
		}
		return bestPath;
	}

	private List<Vertex> bestExistingPathTo(Vertex target, FloydWarshall fw) {

		double bestCost = Double.MAX_VALUE;
		List<Vertex> bestPath = null;
		for (Vertex vert : graph.vertices) {
			List<Vertex> path = fw.path(vert, target);
			double cost = path == null ? Double.MAX_VALUE : graph.cost(path);
			if (cost < bestCost) {
				bestCost = cost;
				bestPath = path;
			}
		}
		return bestPath;
	}

	public Vertex mostExpensiveTarget(FloydWarshall fw) {

		Vertex mostExp = null;
		double bestCost = 0 - Double.MAX_VALUE;
		for (Vertex t : targets) {
			Vertex target = graph.vertexNamed(t.name);
			if (target.isBattery && target.isLoaded())
				continue;
			List<Vertex> path = bestExistingPathTo(target, fw);
			double cost = graph.bestPossibleCost(path, fw);
			if (cost > bestCost && cost < 10000) {
				bestCost = cost;
				mostExp = target;
			} else if (cost == bestCost) {
				// System.out.println(target + " == " + mostExp);
			}
		}
		return mostExp;
	}

	public void fillsTargets(List<Vertex> targetList) {
		targets.clear();
		transferProperties = new Properties();
		for (Vertex v : targetList) {
			if (graph.vertexNamed(v.name) != null
					&& !targets.contains(graph.vertexNamed(v.name)))
				targets.add(graph.vertexNamed(v.name));
		}
		FloydWarshall fw = new FloydWarshall(graph.copy());
		fw.run();
		fw.setGraph(graph);
		while (!targets.isEmpty()) {
			Vertex target = mostExpensiveTarget(fw);
			if (target == null)
				break;
			List<Vertex> bestPathTo = bestPossiblePathTo(target, fw);
			realize(bestPathTo, fw);
			targets.remove(target);
		}
	}

	private void realize(List<Vertex> path, FloydWarshall fw) {

		double additionalCost = 0;
		for (int i = 0; i < path.size() - 1; i++) {
			Vertex from = path.get(i);
			if (from.isBattery) {
				graph.loadBattery(from, fw, transferProperties, true);
			}
			Vertex to = path.get(i + 1);
			double c = from.edgeTo(to).cost();
			if(c >= 10000 || to == path.get(path.size() - 1))
				c = from.edgeTo(to).initialCost;
			additionalCost += c;
			for (Edge e : from.edges) {
				if (e.getEnd() != to) {
					if (e.cost() >= 10000)
						e.swapCosts();
					e.increaseCost(additionalCost);
				}
			}
		}

		/*
		 * The last item in the path (the destination)
		 */
		Vertex dest = path.get(path.size() - 1);
		for (Edge e : dest.edges) {
			if (e.cost() > 9000)
				e.swapCosts();
			e.increaseCost(additionalCost);
		}

		if (path.size() != 2)
			System.out.println("Should not reach here");

		System.out.println(path.get(0).name + "-" + path.get(1).name + "-"
				+ (graph.cost(path) - graph.initialCost(path)) + "-"
				+ graph.cost(path));

		transferProperties.setProperty("transfer" + transferCounter++, path
				.get(0).name
				+ "-"
				+ path.get(1).name
				+ "-"
				+ (graph.cost(path) - graph.initialCost(path))
				+ "-"
				+ graph.cost(path));

		if (dest.isBattery)
			dest.setIsLoaded(true);
	}

	public static void mainMediumGraph() {

		Graph g = Graph.mediumGraph();
		ElectricityNetwork network = new ElectricityNetwork("Medium Net", g);
		List<Vertex> targets = new Vector<Vertex>();
		targets.add(g.vertexNamed("cell"));
		targets.add(g.vertexNamed("laptop"));
		network.fillsTargets(targets);
	}

	public static void mainLongGraph() throws IOException {

		Graph g = Graph.longGraph();
		g.saveETFiles(new File("X:\\GRAPHS"));
		ElectricityNetwork network = new ElectricityNetwork("Medium Net", g);
		List<Vertex> targets = new Vector<Vertex>();
		targets.add(g.vertexNamed("cell"));
		targets.add(g.vertexNamed("laptop"));
		targets.add(g.vertexNamed("pda"));
		targets.add(g.vertexNamed("mp3"));
		System.out.println(g);
		network.fillsTargets(targets);
	}

	public static void mainDemoGraph1() throws IOException {

		Graph g = Graph.demoGraph1(null);
		// g.saveETFiles(new File("X:\\GRAPHS"));
		ElectricityNetwork network = new ElectricityNetwork("Demo Net 1", g);
		List<Vertex> targets = new Vector<Vertex>();
		targets.add(g.vertexNamed("laptop"));
		targets.add(g.vertexNamed("pda"));
		targets.add(g.vertexNamed("mp3"));
		System.out.println(g);
		network.fillsTargets(targets);
	}

	public static void mainDemoGraph2() throws IOException {

		Graph g = Graph.demoGraph2(null);
		// g.saveETFiles(new File("X:\\GRAPHS"));
		ElectricityNetwork network = new ElectricityNetwork("Demo Net 2", g);
		List<Vertex> targets = new Vector<Vertex>();
		targets.add(g.vertexNamed("laptop"));
		targets.add(g.vertexNamed("pda"));
		targets.add(g.vertexNamed("mp3"));
		System.out.println(g);
		network.fillsTargets(targets);
	}

	public static void mainDemoGraph3() throws IOException {

		Graph g = Graph.demoGraph3(null);
		// g.saveETFiles(new File("X:\\GRAPHS"));
		ElectricityNetwork network = new ElectricityNetwork("Demo Net 3", g);
		List<Vertex> targets = new Vector<Vertex>();
		targets.add(g.vertexNamed("cell"));
		targets.add(g.vertexNamed("laptop"));
		targets.add(g.vertexNamed("pda"));
		targets.add(g.vertexNamed("mp3"));
		System.out.println(g);
		network.fillsTargets(targets);
	}

	public static void main(String[] args) throws Exception {
		mainDemoGraph1();
		mainDemoGraph2();
		mainDemoGraph3();
	}

	public Properties compute(Properties edgeProp, int graphIndex) {
		System.out.println("\nNew Resolution\n");
		graph = Graph.demoGraph(graphIndex, edgeProp);
		List<Vertex> targetList = new Vector<Vertex>();
		if (graph.vertexNamed("cell") != null)
			targetList.add(graph.vertexNamed("cell"));
		targetList.add(graph.vertexNamed("laptop"));
		targetList.add(graph.vertexNamed("pda"));
		targetList.add(graph.vertexNamed("mp3"));
		System.out.println(graph);
		fillsTargets(targetList);
		return transferProperties;
	}
}
