package csl.tools.weka;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.HashSet;
import java.util.Set;

import weka.classifiers.Classifier;
import weka.core.Instance;
import weka.core.Instances;

public class CslClassifier {

	private Instances		trainInstances;
	private Classifier		classifier;
	final private File		filePath;
	final private String	filePrefix;
	private int				version;
	private Set<File>		trainingFiles;

	public CslClassifier(String filePath, String filenamePrefix) {

		super();
		this.version = 0;
		this.filePath = new File(filePath);
		this.filePath.mkdirs();
		this.filePrefix = filenamePrefix;
		this.trainingFiles = new HashSet<File>();
	}

	public void buildClassifier(Instances data) throws Exception {

		classifier.buildClassifier(data);
		trainInstances = data;
	}

	public void train(Instances data) throws Exception {

		buildClassifier(trainInstances);
	}

	public void addTrainingFile(File f) {

		trainingFiles.add(f);
	}

	public void increaseVersion() {

		version++;
	}

	public void save() throws IOException {

		File filename = filename(version);
		ObjectOutputStream writer = new ObjectOutputStream(
				new FileOutputStream(filename));
		System.out
				.println("Saved " + filename.getName() + "\r\n\t"
						+ "Classifier type: " + classifier.getClass()
						+ "\r\n\t" + "Trained on: "
						+ trainInstances.numInstances() + " instances");
		System.out.println("\r\nTrained on files:");
		int i = 0;
		for (File f : trainingFiles)
			System.out.println(++i + "\t" + f);
		writer.writeObject(classifier);
		writer.writeObject(trainInstances);
		writer.writeObject(trainingFiles);
		writer.close();
	}

	@SuppressWarnings("unchecked")
	public void loadVersion(int ver) throws IOException, ClassNotFoundException {

		version = ver;
		File filename = filename(ver);
		ObjectInputStream reader = new ObjectInputStream(new FileInputStream(
				filename));
		classifier = (Classifier) reader.readObject();
		trainInstances = (Instances) reader.readObject();
		trainingFiles = (Set<File>) reader.readObject();
		System.out
				.println("Loaded " + filename.getName() + "\r\n\t"
						+ "Classifier type: " + classifier.getClass()
						+ "\r\n\t" + "Trained on: "
						+ trainInstances.numInstances() + " instances");
		System.out.println("\r\nTrained on files:");
		int i = 0;
		for (File f : trainingFiles)
			System.out.println(++i + "\t" + f);
	}

	/**
	 * Loads the classifier
	 * 
	 * @param ver
	 *            the version number, -1 denotes the last version
	 * @return
	 */
	private File filename(int ver) {

		if (ver == -1) {
			File[] files = new File(filePrefix).getParentFile().listFiles(
					new FileFilter() {

						public boolean accept(File pathname) {

							return pathname.getName().endsWith(".wekamodel");
						}
					});
			for (File f : files) {
				int a = f.getName().indexOf('_') + 1;
				int b = f.getName().indexOf('.');
				int v = Integer.parseInt(f.getName().substring(a, b));
				if (v > ver)
					ver = v;
			}
		}
		version = ver;
		return new File(filePrefix + "_" + ver + ".wekamodel");
	}

	public double classifyInstance(Instance instance) throws Exception {

		return classifier.classifyInstance(instance);
	}

	public void setClassifier(Classifier classifier) {

		this.classifier = classifier;
	}

	public boolean notBuilt() {

		return classifier == null;
	}

	public Instances getTrainInstances() {

		return trainInstances;
	}

	public void setTrainInstances(Instances newInstances) {

		trainInstances = newInstances;
	}
}
