/**
* js/pianoroll.js
*
* pianoroll basic component
*
*/

'use strict';

var PIXI = require('pixi');
var randomColor = require('randomColor');

function PianoRoll(parentContainer, xInit, yInit, height, linesDown, pixelsPerSecond, width, noteColors, colorsReg, lineColor, lineInterval, offsetMusic, noteHeight){
    var _this = this;
    this.container = new PIXI.DisplayObjectContainer();
    this.container.position.x = xInit;
    this.container.position.y = yInit;
    parentContainer.addChild(this.container);

    this.linesDown = linesDown;
    this.height = height;
    this.pixelsPerSecond = pixelsPerSecond;
    this.width = width;
    this.noteColors = noteColors;
    this.colorsReg = colorsReg || {};
    this.lineColor = lineColor;
    this.lineInterval = lineInterval;
    this.offsetMusic = offsetMusic || 0;
    this.noteHeight = noteHeight;

    this.addNote = function(note, startTime, duration, velocity, canal){
        //console.log("coucou 1", note, timeFromZero, ts, velocity, pixelsPerSecond, container, prHeight);
        var beginX = (this.offsetMusic + startTime) * this.pixelsPerSecond / 1000;
        var width = duration * this.pixelsPerSecond / 1000;
        if((beginX+width) <  Math.abs(this.container.x) - this.width) {
            // not visible. do nothing
            return;
        }
        // We draw the rectangle
        var graphics = new PIXI.Graphics();
        //console.log("beginX = ", beginX, "canal = ", canal, "color = ", noteColor[canal], "width = ", width, "note = ", note, "velocity = ", velocity);
        var color = this.colorsReg[canal];
        if(typeof(color) === 'undefined') {
            var colorsRegSize = Object.keys(this.colorsReg).length;
            if(colorsRegSize < this.noteColors.length) {
                color = this.colorsReg[canal] = this.noteColors[colorsRegSize];
            }
            else {
                color = this.colorsReg[canal] = parseInt(randomColor({ luminosity: 'light', hue: 'random', format:'hex'}).replace(/^#/, ''), 16);
            }
        }
        graphics.beginFill(color, (velocity / 128));
        var y = (128-note) * this.height / 128; // (128-note) because y = 0 is for note = 128 and y = 128 for note = 0
        graphics.drawRect(0, Math.floor(y - (noteHeight/2) + ((this.height / 128)/2)), width, noteHeight);
        graphics.endFill();
        graphics.x = beginX;
        this.container.addChild(graphics);
    };

    this.addLine = function(lineNb){
        var graphics = new PIXI.Graphics();
        var x = -this.container.x;
        graphics.beginFill(0xFFFF00);
        graphics.lineStyle(1, this.lineColor);
        var y = this.linesDown ? this.height - 20 : 0;
        graphics.moveTo(x, y);
        graphics.lineTo(x, y + 20);
        graphics.endFill();
        this.container.addChild(graphics);
        // Add text
        var totalSec = lineNb * this.lineInterval / 1000;
        var hours = parseInt( totalSec / 3600 ) % 24;
        var minutes = parseInt( totalSec / 60 ) % 60;
        var seconds = totalSec % 60;
        var timeStr = (hours < 10 ? '0' + hours : hours) + ':' + (minutes < 10 ? '0' + minutes : minutes) + ':' + (seconds  < 10 ? '0' + seconds : seconds);
        var fontObj = { font: '10pt Arial', fill: '#444444' };
        var t = new PIXI.Text(timeStr, fontObj);
        t.x = x + 2;
        t.y = this.linesDown ? this.height - 15 : 2;
        this.container.addChild(t);
    };

    this.moveTo = function(diffTime){
        this.container.x = Math.floor(diffTime*this.pixelsPerSecond);
    };

    this.removePassedObjets = function(){
        var nbChilds = _this.container.children.length;
        var i = 0, childIsNowDisplayed = false, childrenToRemove = [];
        while(i<nbChilds && !childIsNowDisplayed){
            var child = _this.container.children[i++];
            //console.log("remove ? ", child.x, child.width, ((child.x + child.width) < (Math.abs(_this.container.x) - _this.width)));
            if(typeof(child) == 'undefined') {
                continue;
            }
            if((child.x + child.width) < (Math.abs(_this.container.x) - _this.width)){
                childrenToRemove.push(child);
                //console.log("    remove !!!");
            }
            else {
                childIsNowDisplayed = true;
                //console.log("    childIsNowDisplayed");
            }
        }
        childrenToRemove.forEach(function(child) {
            _this.container.removeChild(child);
        });
        //console.log("before : ", nbChilds, ", after : ", _this.container.children.length);
    };

    // remove notes each scene width
    //var removeInterval = window.setInterval(this.removePassedObjets, 1000 * sceneWidth / this.pixelsPerSecond );
    window.setInterval(this.removePassedObjets, 1000 * this.width / this.pixelsPerSecond );

}

module.exports = PianoRoll;
