/**
* js/annotstimeline
*
* annotstimeline basic component
*
*/

'use strict';

var PIXI = require('pixi');
var _ = require('lodash');
var rgb2hex = require('./utils');

var defaultOptions = {		
    logger: undefined,
    intervalWidth: 10,
    intervalHeight: 5,
    maxCellHeight: 200,
    radius: 300,
    circleX:500,
    circleY:500
};


function AnnotsTimeLine(options){
    var _this = this;
    var opts = _(options).defaults(defaultOptions).value();
    
    this.container = new PIXI.DisplayObjectContainer();
    this.container.x = opts.xInit;
    this.container.y = opts.yInit;
    this.container.width = opts.width;
    this.container.height = opts.height;    
    
    this.timeBegin = opts.timeBegin;
    this.timeEnd = opts.timeEnd;
    this.duration = (this.timeEnd - this.timeBegin)/1000;
    this.width = opts.width;
    this.height = opts.height;
    this.intervalHeight = opts.intervalHeight;
    this.intervalWidth = opts.intervalWidth;
    this.maxCellHeight = opts.maxCellHeight;
    this.annotCategories = opts.annotCategories;
    
    this.circleX = opts.circleX;
    this.circleY = opts.circleY;
    this.radius = opts.radius;
    this.perimeter = 2*Math.PI* this.radius;
    this.intervalDuration = (this.intervalWidth * this.duration / this.perimeter);
    
    var totalIndex = Math.floor(this.perimeter/this.intervalWidth);
    	
    this.cells = []
    for (var i=0; i<(this.perimeter/this.intervalWidth) ; i++){
    	this.cells[i] = [];
    	this.cells[i].totalAnnots = 0;
    	this.cells[i].graphics = new PIXI.Graphics();
    	this.cells[i].graphics.position.x = this.circleX + this.radius * Math.sin(i*(360/totalIndex)*(Math.PI/180));
    	this.cells[i].graphics.position.y = this.circleY - this.radius * Math.cos(i*(360/totalIndex)*(Math.PI/180));
    	this.cells[i].graphics.rotation = (i)*(360/totalIndex)*(Math.PI/180) + (360/(totalIndex*2))*(Math.PI/180);
    	this.container.addChild(this.cells[i].graphics);
    	this.cells[i].categories = {};
    	
    	for (var category in this.annotCategories[0].colors){
    		this.cells[i].categories[category] = {
				"count": 1,
				"color": this.annotCategories[0].colors[category]
    		};
    	}
    }
    
    var ws = opts.ws;
    var stageView = opts.stageView;

    var graphics = new PIXI.Graphics();
    graphics.lineStyle(1, 0x000000)
    	.drawCircle(this.circleX, this.circleY, this.radius)
    	.moveTo(this.circleX, this.circleY)
    	.lineTo(this.circleX, this.circleY - this.radius - this.maxCellHeight - 30)
    	.endFill()
    this.container.addChild(graphics);
    
    stageView.registerComponent(this);

    //Add Annotation to the TimeLine
    this.addAnnot = function(data){
    	if (typeof(this.annotCategories[0].colors[data.content.category.code]) !== 'undefined'){
    		var annotCode = data.content.category.code;
    	} else {
    		var annotCode = this.annotCategories[0].order[this.annotCategories[0].order.length -1];
    	}
    	var annotTime = Date.parse(data.ts);
    	
    	if (this.timeEnd > Date.parse(data.ts)){
	    	var i = Math.floor((Date.parse(data.ts)-this.timeBegin)/(1000*this.intervalDuration));
			this.cells[i].categories[annotCode].count += 1;
			this.cells[i].totalAnnots +=1;
			this.redrawCell(this.cells[i], i);
    	}
    };
    
    //Draw the cellule
    this.redrawCell = function(cell){
    	var x = 0; 
    	var y = 0;
    	cell.graphics.clear();
    	
    	//Check if total height is higher than Max Cell Height
    	if ((cell.totalAnnots*this.intervalHeight) > this.maxCellHeight){
    		var heightStep = this.maxCellHeight/cell.totalAnnots;
    	} else {
    		var heightStep = this.intervalHeight;
    	}
    	//Draw the rect depending on the height step calculated
    	for (var i=0; i< this.annotCategories[0].order.length; i++){
    		var currentCode = this.annotCategories[0].order[i];
			cell.graphics.beginFill(cell.categories[currentCode].color.replace("#", "0x"))
    			.drawRect(0, y, this.intervalWidth, -cell.categories[currentCode].count * heightStep)
    			.endFill();
    		y -= cell.categories[currentCode].count*heightStep;
    	}
    }
    
    this.init = function() {

    	ws.message(function(data) {
            _this.addAnnot(data);
        });

    };
    
    this.start = function() {
    	for (var i=0; i<(this.perimeter/this.intervalWidth) ; i++){
    		this.redrawCell(this.cells[i]);
    	}
    };
    
    this.refresh = function() {
    };
    
    this.stop = function(){
    };
    
    return this;
}

module.exports = {
		AnnotsTimeLine: AnnotsTimeLine
};
