/**
* scripts/doubleroll.js
*
* This is the starting point for your application.
* Take a look at http://browserify.org/ for more info
*/

'use strict';


var PIXI = require('pixi');
var _ = require('lodash');
var PianoRoll = require('./pianoroll');
var Utils = require('./utils.js');

var defaultConfig = {
    orientation: 'horizontal',
    logger: undefined,
    sceneWidth: 1024,
    pianorolls : [
      {
        height: 384,
        timeWidth: 10,
        lineInterval: 5000,
        noteHeight: undefined,
        range: {
    		bottom: 40,
    		top: 90,
        },
        dynamicRange: true,
      },
      {
        height: 384,
        timeWidth: 60,
        lineInterval: 5000,
        noteHeight: undefined,
        range:{
    		bottom: 0,
    		top: 128,
        },
        dynamicRange: false,
      },
    ],
    framerate: 25,
    offsetMusic: false,
    sceneBgColor: 0xFFFFFF,
    lineColor: 0x444444,
    lineFillColor: 0xFFFF00,
    noteColors: [0xB90000, 0x4BDD71, 0xAF931E, 0x1C28BA, 0x536991],
    noteHeight: undefined,
    zeroShift: 0.9,
    timeWidth: 60,
    lineInterval: 5000,
//    wsUri: undefined,
//    eventCode: undefined

};

function DoubleRoll(options) {

    var _this = this;
    var opts = _(options).defaults(defaultConfig).value();

    var orientation = opts.orientation;
    var isHorizontal = (orientation !== 'vertical');

    this.logger = opts.logger;
    this.lineColor = opts.lineColor;
    this.lineFillColor = opts.lineFillColor;
    this.framerate = opts.framerate;
    this.offsetMusic = opts.offsetMusic;
    this.noteColors = opts.noteColors;

    var noteHeight = opts.noteHeight;
    var sceneHeight = opts.sceneHeight || _(opts.pianorolls).reduce(function(s,p) { return s + p.height; }, 0);
    var timeWidth = opts.timeWidth;
    var lineInterval = opts.lineInterval;
    var offsetMusic = opts.offsetMusic;

    var sceneWidth = opts.sceneWidth;
    var stageView = opts.stageView;

    var zeroShift = opts.zeroShift;

    var ws = opts.ws;

    var colorsReg = {};

    this.container = new PIXI.DisplayObjectContainer();
    this.container.x = Math.floor(sceneWidth*zeroShift);
    this.container.y = 0;

    stageView.registerComponent(this);

    var pianorollList = [];

    var pianorollOptions = {
        parentContainer: this.container,
        orientation: orientation,
        xInit: 0,
        width: sceneWidth,
        noteColors: this.noteColors,
        colorsReg: colorsReg,
        lineColor: this.lineColor,
        lineInterval: lineInterval,
        offsetMusic: offsetMusic,
    };

    var yInit = opts.yInit || 0;
    var linesDown = true;
    _(opts.pianorolls).forEach(function(prDef, i) {
        var prNoteHeight = noteHeight || prDef.noteHeight || prDef.height / (prDef.range.top - prDef.range.bottom + 1);
        var prTimeWidth = prDef.timeWidth || timeWidth;
        pianorollList.push(new PianoRoll(_({
            yInit: yInit,
            height: prDef.height,
            linesDown: linesDown,
            pixelsPerSecond: Math.floor(sceneWidth / prTimeWidth),
            noteHeight: prNoteHeight,
            lineInterval: prDef.lineInterval,
            range: prDef.range,
            dynamicRange: prDef.dynamicRange,
        }).defaults(pianorollOptions).value()));
        yInit += prDef.height;
        linesDown = !linesDown;

        if(i<(opts.pianorolls.length-1)) {
            var lineGraphics = new PIXI.Graphics()
                .beginFill(_this.lineFillColor)
                .lineStyle(1, _this.lineColor)
                .moveTo(Math.floor(sceneWidth*zeroShift), yInit)
                .lineTo(-sceneWidth - Math.floor(sceneWidth*zeroShift), yInit)
                .endFill();
            _this.container.addChild(lineGraphics);
        }
    });

    var currentTimeText = new PIXI.Text('-- : -- : --', { font: '40pt Arial', fill: '#646464' });
    console.log(currentTimeText);
    currentTimeText.x = -currentTimeText.width - 60; 
    currentTimeText.y = sceneHeight - currentTimeText.height - 30;
    this.container.addChild(currentTimeText);
    
    
    if(!isHorizontal) {
        this.container.rotation = Math.PI/2; 
        this.container.y = sceneHeight;
        this.container.x = sceneWidth;
    }


    this.init = function() {

    	ws.message(function(data) {
            _this.addNotes(data);
        });

    };
    
    this.updateTime = function(){
        currentTimeText.setText(Utils.formatTime(Date.now()));
    };

    this.addNotes = function(data) {

        pianorollList.forEach(function(c) {
            c.addNoteRaw(data);
        });
    };

    this.refresh = function() {
        pianorollList.forEach(function(c) {
            c.move();
        });
    };

    // Init page and intervals
    var startTs;
    var refreshTimeInterval;

    this.start = function() {
    	refreshTimeInterval = setInterval(function() {_this.updateTime();}, 1000);
    	
        startTs = Date.now();
        pianorollList.forEach(function(c) {
            c.start();
        });
    };

    this.stop = function() {

        pianorollList.forEach(function(c) {
            c.stop();
        });
    };


    this.log = function(m) {
        if(this.logger) {
            this.logger.log(m);
        }
    };



    return this;
}

module.exports = {
    DoubleRoll: DoubleRoll
};
