/**
 * scripts/main.js
 *
 * This is the starting point for your application.
 * Take a look at http://browserify.org/ for more info
 */

'use strict';


var PIXI = require('pixi');
var rgb2hex = require('./utils');

// Config vars
var horizontalView = false;
var logger = false;
var sceneWidth = 1920;
var sceneHeight = 1080;
var prSize1 = 435;
var prSize2 = 435;
var prSize3 = 300;
var sceneBgColor = 0xFFFFFF;
var lineColor = 0x444444;
if (horizontalView){
	var pixelsPerSecond1 = Math.floor(sceneWidth / 10); // nb of pixels per second
} else{
	var pixelsPerSecond1 = Math.floor(sceneHeight / 10); // nb of pixels per second
}
var manualFramerate = pixelsPerSecond1 / 4;
if (horizontalView){
	var pixelsPerSecond2 = Math.floor(sceneWidth / 60); // nb of pixels per second
} else {
	var pixelsPerSecond2 = Math.floor(sceneHeight / 60); // nb of pixels per second
}
var pixelsPerSecond3 = Math.floor(sceneHeight / 60); // nb of pixels per second
var lineInterval = 5000; // means line every 5 seconds
var nbLines = -1;
var noteHeight = 110;
var noteColors = [0xB90000, 0x4BDD71, 0xAF931E, 0x1C28BA, 0x536991];
var colorsReg = {};
// Vars
var noteDict = [];
// Timecode method
var timePageLoaded = Date.now();
var offsetMusic = false;
var categoriesColor = {		
	"ntm" : rgb2hex(205,200,63),
    "iam" : rgb2hex(205,200,63),
    "hip" : rgb2hex(205,200,63),
    "hop" : rgb2hex(205,200,63),
    "rock" : rgb2hex(222,139,83),
    "rap" : rgb2hex(222,139,83),
    "classic" : rgb2hex(222,139,83),
    "drums" : rgb2hex(197,163,202),
    "guitar" : rgb2hex(197,163,202),
    "bass" : rgb2hex(121,187,146),
    "default": rgb2hex(128,128,128)
};

//create an new instance of a pixi stage
var stage = new PIXI.Stage(sceneBgColor);

//create a renderer instance.
var renderer = PIXI.autoDetectRenderer(sceneWidth, sceneHeight);

//add the renderer view element to the DOM
document.getElementById('canvasContainer').appendChild(renderer.view);

var uberContainer = new PIXI.DisplayObjectContainer();
if (horizontalView){
	uberContainer.position.x = Math.floor(sceneWidth*9/10);
	uberContainer.position.y = 0;
} else {
	uberContainer.position.x = 0;
	uberContainer.position.y = Math.floor(sceneHeight*9/10);
}
stage.addChild(uberContainer);

/* ---------------------------------------------------------------- */
/* ------------------- Init Pianoroll containers ------------------ */
/* ---------------------------------------------------------------- */

var PianoRoll = require('./pianoroll.js')

var containerList = [];

if (horizontalView){
	containerList.push(new PianoRoll(uberContainer, 0, 0, prSize1, true, pixelsPerSecond1, sceneWidth, noteColors, colorsReg, lineColor, lineInterval, offsetMusic, prSize1 / 128, horizontalView));
	containerList.push(new PianoRoll(uberContainer, 0, prSize1, prSize2, false, pixelsPerSecond2, sceneWidth, noteColors, colorsReg, lineColor, lineInterval, offsetMusic, prSize2 / 128, horizontalView));
} else {
//	containerList.push(new PianoRoll(uberContainer, sceneWidth - prSize1, 0, sceneHeight, true, pixelsPerSecond1, prSize1, noteColors, colorsReg, lineColor, lineInterval, offsetMusic, prSize1 / 128, horizontalView));
//	containerList.push(new PianoRoll(uberContainer, sceneWidth - (prSize1 + prSize2), 0, sceneHeight, false, pixelsPerSecond2, prSize2, noteColors, colorsReg, lineColor, lineInterval, offsetMusic, prSize2 / 128, horizontalView));
	containerList.push(new PianoRoll(uberContainer, sceneWidth - prSize1, 0, sceneHeight, false, pixelsPerSecond2, prSize2, noteColors, colorsReg, lineColor, lineInterval, offsetMusic, prSize2 / 128, horizontalView));
}

// Line between two containers
var graphics = new PIXI.Graphics();
graphics.beginFill(0xFFFF00);
graphics.lineStyle(1, lineColor);
if (horizontalView){
	graphics.moveTo(0, prSize1);
	graphics.lineTo(sceneWidth, prSize1);
} else {
	graphics.moveTo(sceneWidth - prSize1, 0);
	graphics.lineTo(sceneWidth - prSize1, sceneHeight);
	graphics.moveTo(sceneWidth - (prSize1 + prSize3), 0);
	graphics.lineTo(sceneWidth - (prSize1 + prSize3), sceneHeight);
}
graphics.endFill();
stage.addChild(graphics);

function addNotes(data){
    if(!offsetMusic){
        // get difference between the current note timecode and my zero to set the difference between the canvas's zero and the music's zero
        // in order to place in real time
        var now = Date.now();
        var timeBetweenNowAndStart = now - timePageLoaded;
        offsetMusic = timeBetweenNowAndStart - data.content[1];
    }
    var note = data.content[3];
    var velocity = data.content[4];
    if(velocity===0){
        if(typeof noteDict[data.content[2]][note]!=='undefined'){
            // We close the note in container one
            var duration = data.content[1] - noteDict[data.content[2]][note].ts;
            for(var i=0;i<containerList.length;i++){
                //               addNote(note, startTime,                          duration, velocity,                                 canal)
                containerList[i].addNote(note, noteDict[data.content[2]][note].ts, duration, noteDict[data.content[2]][note].velocity, data.content[2]);
            }
            // delete entry
            delete noteDict[data.content[2]][note];
        }
    }
    else{
        if(typeof noteDict[data.content[2]]==='undefined'){
            noteDict[data.content[2]] = {};
        }
        noteDict[data.content[2]][note] = {ts: data.content[1], velocity:velocity};
    }
}

function addLine(){
    var ts = new Date();
    for(var i=0;i<containerList.length;i++){
        containerList[i].addLine(ts);
    }
}

/* ---------------------------------------------------------------- */
/* ------------------- Init AnnotsRoll containers ----------------- */
/* ---------------------------------------------------------------- */

var AnnotsRoll = require('./annotsRoll.js')

var AnnotationRoll = new AnnotsRoll(uberContainer, sceneWidth - (prSize2 + prSize3), 0, prSize3 + prSize2, sceneHeight, prSize3, pixelsPerSecond3, lineInterval);

/* ---------------------------------------------------------------- */
/* ------------------- Init generalView container ----------------- */
/* ---------------------------------------------------------------- */

var GeneralView = require('./generalView.js')

var GeneralRoll = new GeneralView(uberContainer, 0, 0, sceneWidth - (prSize2 + prSize3), sceneHeight, Date.now(), Date.now() + 300000, 30, 5, 100, categoriesColor);


function addAnnots(data){
    var title = data.content.category.label;
    var code = data.content.category.code;
    var user = data.content.user;
    if (typeof(categoriesColor[code]) != 'undefined'){
    	var color = categoriesColor[code];
    }
    else {
    	var code = "default";
    	var color = categoriesColor[code];
    }
    
    AnnotationRoll.addAnnot(title, user, color);
    GeneralRoll.addAnnot(code, Date.now());
}

/* ---------------------------------------------------------------- */
/* ----------------------- Socket management ---------------------- */
/* ---------------------------------------------------------------- */

var sock = null;
var sock2 = null;
var ellog = null;
function log(m) {
    if(logger){
        ellog.innerHTML += m + '\n';
        ellog.scrollTop = ellog.scrollHeight;
    }
}
window.onload = function(){

    if(logger){
        ellog = document.getElementById('log');
    }
    else{
        document.body.removeChild(document.getElementById('log'));
    }

    var wsuriInit;
    var wsuri;
    var wsuri2;
    
    if (window.location.protocol === 'file:') {
    	wsuriInit = 'ws://127.0.0.1:8090/broadcast';
    } else {
    	wsuriInit = 'ws://' + window.location.hostname + ':8090/broadcast';
    }
    wsuri = wsuriInit + '?channel=PIANOROLL&event_code='+eventCode;
    wsuri2 = wsuriInit + '?channel=ANNOT&event_code='+eventCode;
    
    if ('WebSocket' in window) {
        sock = new WebSocket(wsuri);
        sock2 = new WebSocket(wsuri2);
    } else if ('MozWebSocket' in window) {
        sock = new MozWebSocket(wsuri);
        sock2 = new WebSocket(wsuri2);
    } else {
        log('Browser does not support WebSocket!');
        window.location = 'http://autobahn.ws/unsupportedbrowser';
    }

    if (sock) {
        sock.onopen = function(){
            if(logger){
                log('Connected to ' + wsuri);
            }
        };

        sock.onclose = function(e) {
            if(logger){
                log('Connection closed (wasClean = ' + e.wasClean + ', code = ' + e.code + ', reason = \'' + e.reason + '\')');
            }
            sock = null;
        };

        sock.onmessage = function(e) {
            if(logger){
                log('Got message: ' + e.data);
            }
            addNotes(JSON.parse(e.data));
        };
    }
    
    if (sock2) {
        sock2.onopen = function(){
            if(logger){
                log('Connected to ' + wsuri);
            }
        };

        sock2.onclose = function(e) {
            if(logger){
                log('Connection closed (wasClean = ' + e.wasClean + ', code = ' + e.code + ', reason = \'' + e.reason + '\')');
            }
            sock2 = null;
        };

        sock2.onmessage = function(e) {
            if(logger){
                log('Got message: ' + e.data);
            }
//            console.log(e);
            addAnnots(JSON.parse(e.data));
        };
    }
};

function replaceContainers(){
    var diff = (Date.now() - timePageLoaded)/1000;// nb of seconds since page loaded
    
    for(var i=0;i<containerList.length;i++){
        containerList[i].moveTo(-diff);
        AnnotationRoll.moveTo(-diff);
    }
    
    renderer.render(stage);
}

/* ---------------------------------------------------------------- */
/* ---------------------------- Main ------------------------------ */
/* ---------------------------------------------------------------- */

// Init page and intervals
addLine();
var moveInterval = window.setInterval(replaceContainers, 1000/manualFramerate);
var verticalLinesInterval = window.setInterval(addLine, lineInterval);

// Little inteval to show time
var nbSec = 0;
var mySpan = document.getElementById('myspan');
function updateTime(){
    nbSec++;
    var hours = parseInt( nbSec / 3600 ) % 24;
    var minutes = parseInt( nbSec / 60 ) % 60;
    var seconds = nbSec % 60;
    var timeStr = (hours < 10 ? '0' + hours : hours) + ':' + (minutes < 10 ? '0' + minutes : minutes) + ':' + (seconds  < 10 ? '0' + seconds : seconds);
    mySpan.innerHTML = timeStr;
}
var secondInterval = window.setInterval(updateTime, 1000);

module.exports = {
    moveInterval: moveInterval,
    verticalLinesInterval: verticalLinesInterval,
    secondInterval: secondInterval
};
